#
## <SHAREFILE=algebra/diffop/diffop.mpl >
## <DESCRIBE>
##        This package factorizes differential operators.
##                It also computes formal and rational solutions of
##                systems of linear ordinary differential equations.
## See also: diffop1.ms diffop2.ms diffop3.ms  
##                AUTHOR:  Mark van Hoeij,  hoeij@sci.kun.nl
## </DESCRIBE>
## <UPDATE=R4 >

# diffop:=`_diffop`:
##########################
# standard input
###
# Convert from 5.2 to Release 3
# Author: Mark van Hoeij
# University of Nijmegen, Netherlands
# e-mail: hoeij@sci.kun.nl
# If you change/improve this program, or if you find a bug, please let me know

# diffop package: This package provides factorizations of local (i.e. power
# series coefficients) and global (i.e. rational functions coefficients)
# linear differential operators. It also provides formal solutions for systems
# of linear differential equations (matrix differential equations).

# date: november 1994
# date previous version: februari 1994
# Changes: added code for matrix differential equations. Included
# irreducibility proving in the global factorization algorithm.

# More information about the methods used is available by e-mail request.
# Currently the local factorization has been written in LaTeX and the global
# factorization is being written. Also a list of descriptions of most of the
# procedures and variables used in diffop is available.

x:=evaln(x):     # We use these 3 variables as global variables.
xDF:=evaln(xDF): # xDF = x*d/dx
DF:=evaln(DF):   # DF  = d/dx

#############################
#  groundfield computation  #-------------------------------------------------
#############################			diffop_groundfield

# Note: the procedures in this section are copied from the file IntBasis.
# For brief descriptions see IntBasis.
# The main purpose is to have ground field independent procedures like
# g_expand (groundfield dependent expand) which work at a reasonable
# efficiency. The field is always given by the argument ext.

macro(	bug=`diffop/bug`,
	degree_ext=`diffop/degree_ext`,
	ext_to_coeffs=`diffop/ext_to_coeffs`,
	g_conversion1=`diffop/g_conversion1`,
	g_conversion2=`diffop/g_conversion2`,
	g_evala=`diffop/g_evala`,
	g_evala_rem=`diffop/g_evala_rem`,
	g_expand=`diffop/g_expand`,
	g_ext=`diffop/g_ext`,
	g_ext_r=`diffop/g_ext_r`,
	g_factor=`diffop/g_factor`,
	g_factors=`diffop/g_factors`,
	g_gcdex=`diffop/g_gcdex`,
	g_normal=`diffop/g_normal`,
	g_quo=`diffop/g_quo`,
	g_quotient=`diffop/g_quotient`,
	g_rem=`diffop/g_rem`,
	g_solve=`diffop/g_solve`,
	g_zero_of=`diffop/g_zero_of`,
	modm=`diffop/modm`,
	modulus=`diffop/modulus`,
	truncate=`diffop/truncate`,
	truncate_x=`diffop/truncate_x`,
	v_ext_m=`diffop/v_ext_m`,
	factors=readlib('factors')
):

g_conversion1:={}: # RootOf syntax -> my own syntax
g_conversion2:={}: # my syntax -> RootOf syntax
modulus:=0:	   # For modular computation

modm:=proc() if modulus=0 then args else args mod modulus fi end:
# modm:=proc() args end:

g_solve:=proc()
	local v,w,res,i,a,ext;
	if nargs<3 then
		userinfo(10,diffop,`solving linear equations`)
	fi;
if modulus=0 or nargs=4 then
	modm(subs(g_conversion1,evala(solve(op(subs(g_conversion2,modm([args[1
	 ..min(2,nargs)]])))))))
else
	# Solving linear equations modulo a number, not very efficiently written.
	v:=args[1];
	if nargs=3 then ext:=args[3] else ext:=g_ext(v) fi;
	if nargs>=2 then w:=args[2] else
		w:=indets(v) minus {op(ext)}
	fi;
	v:=modm(map(g_expand,v,ext)) minus {0};
	if v={} then RETURN(v) fi;
	a:=v[1];
	i:=indets(a) intersect w;
	if i=[] then RETURN(NULL) fi;
	i:=i[1];
	res:=g_normal(-coeff(a,i,0)/coeff(a,i,1));
	a:=g_solve(subs(i=res,v minus {a}),w minus {i},ext);
	if a=NULL then a else a union {i=modm(g_expand(subs(a,res),ext))} fi
fi	
end:

g_quotient:=proc(aa,bb,n,ext)
	local a,b,res,i,g,la,lb;
	a:=modm(g_expand(aa,ext));
	b:=modm(g_expand(bb,ext));
	if a=0 or b=0 then RETURN(0) fi;
	g:=g_normal(1/tcoeff(b,x));
	res:=0;
	la:=ldegree(a,x);
	lb:=ldegree(b,x);
	for i from la-lb to n-1 do
		res:=res+g*x^i*coeff(a,x,i+lb);
		a:=a-g_expand(b*g*x^i*coeff(a,x,i+lb),ext)
	od;
	modm(g_expand(res,ext))
end:

g_gcdex:=proc(a,b,c,ext)
	local r,ss,tt,s,t;
	options remember;
	if ext=[] then
		r:=gcdex(a,b,c,'ss','tt')
	else
		r:=subs(g_conversion1,evala(Gcdex(op(subs(g_conversion2
		 ,[a,b,c,'ss','tt'])))))
	fi;
	s:=subs(g_conversion1,ss);
	t:=subs(g_conversion1,tt);
	[r,s,t]
end:

g_rem:=proc(a,b,y,ext)
	g_expand(rem(g_expand(a,ext),b,y),ext)
end:

g_quo:=proc(a,b,y,ext)
	modm(g_expand(quo(g_expand(a,ext),modm(g_expand(b,ext)),y),ext))
end:

g_factors:=proc(f,ext)
	if ext=[] then
		factors(f)
	else
		subs(g_conversion1,factors(
		 op(subs(g_conversion2,[f,ext]))))
	fi
end:

g_factor:=proc(ff,ext)
local f;
	f:=numer(g_normal(ff));
	userinfo(8,diffop,`factorizing :`,f,`over Q(`,op(ext),`)`);
	if ext=[] then factor(f)
	else
		subs(g_conversion1,evala(Factor(subs(g_conversion2,f)
		 ,op(subs(g_conversion2,ext)))))
	fi
end:

degree_ext:=proc(aa,bb)
local a,b,v,i,all,d,var;
options remember;
	a:=subs(g_conversion2,aa);
	b:=subs(g_conversion2,bb);
	v:=indets(a,RootOf) minus indets(b,RootOf);
	all:=[op(indets([a,b],RootOf))];
	all:={seq(all[i]=var[i],i=1..nops(all))};
	d:=1;
	for i in v do
		d:=d*degree(subs(all,op(i)),_Z)
	od;
	d
end:

g_expand:=proc(a,ext)
	g_evala(expand(a),ext)
end:

g_evala:=proc(a,ext)
	local dummy,e;
	if nops(ext)=0 then RETURN(a)
	elif not type(a,polynom(anything,ext[1])) then
		RETURN(expand(g_normal(a)))
	elif nops(ext)=1 then
		e:=ext[1];
		expand(convert([seq(coeff(a,e,dummy)*g_evala_rem(e^dummy)
		 ,dummy=0..degree(a,e))],`+`))
	else
	e:=g_evala(a,ext[2..nops(ext)]);
	g_evala(expand(convert([seq(coeff(e,ext[1],dummy)*g_evala_rem(
	 ext[1]^dummy),dummy=ldegree(e,ext[1])..degree(e,ext[1]))],`+`)),
	 ext[2..nops(ext)])
	fi
end:

g_evala_rem:=proc()
	options remember;
	expand(subs(g_conversion1,evala(Expand(subs(g_conversion2,args)))))
end:

g_normal:=proc(aa)
	local a;
	if indets(aa,RootOf)<>{} then 
		RETURN(evala(Normal(aa)))
	fi;
	if modulus<>0 then
		a:=subs(g_conversion2,Normal(aa) mod modulus);
		if indets(a,RootOf)<>{} then
			subs(g_conversion1,evala(Normal(a)))
		else
			a
		fi
	else
		a:=subs(g_conversion2,aa);
		if indets(a,RootOf)={} then
			normal(a)
		else
			subs(g_conversion1,evala(Normal(a)))
		fi
	fi
end:

g_zero_of:=proc(k,y,ext)
	global  g_conversion1, g_conversion2;
	local a,vv;
#bug:     options remember;
	if degree(k,y)=1 then
		ext:=NULL;
		RETURN(g_normal(-coeff(k,y,0)/coeff(k,y,1)))
	fi;
	a:=RootOf(subs(g_conversion2,k),y);
#bug:     if subs(g_conversion1,a)=a then
	if member(_Z,indets(op(subs(g_conversion1,a)))) then
		if g_conversion1={} then g_conversion1:=NULL fi;
		vv:=nops(g_conversion2);
		g_conversion1:=a=`diffop/rootof`.vv,g_conversion1;
		g_conversion2:={`diffop/rootof`.vv=a,op(g_conversion2)};
		ext:=`diffop/rootof`.vv
	else
		ext:=subs(g_conversion1,a)
	fi
end:

# Gives the zeros of the factors, their multiplicities and algebraic extensions
v_ext_m:=proc(f,y)
local ext,nulp,i,result;
	if degree(f,y)=0 then RETURN({}) fi;
	if type(f,`^`) then
		nulp:=g_zero_of(op(f)[1],y,'ext');
		ext:=eval(ext);
		RETURN({[nulp,op(f)[2],[ext],degree(op(f)[1],y)]})
	fi;
	if type(f,`*`) then
		result:={};
		for i in {op(f)} do
			result:=result union v_ext_m(i,y)
		od;
		RETURN(result)
	fi;
	nulp:=g_zero_of(f,y,'ext');
	ext:=eval(ext);
	{[nulp,1,[ext],degree(f,y)]}
end:

ext_to_coeffs:=proc(a,ext)
	local dummy,aa;
	aa:=(indets(a) minus indets(ext)) intersect
	{seq(`diffop/rootof`.dummy,dummy=0..nops(g_conversion2))};
	coeffs(a,[op(aa)])
end:

# Takes the lowest coefficients
truncate:=proc(aa,n,y,ext)
local dummy,a;
	a:=collect(aa,y);
	a:=expand(convert([seq(y^dummy*coeff(a,y,dummy)
	 ,dummy=ldegree(a,y)..n-1)],`+`));
	modm(g_evala(a,ext))
end:

truncate_x:=proc(a,f,n)
local dummy;
	convert([seq(x^dummy*coeff(f,x,dummy),dummy=ldegree(f,x)..n-1)],`+`)
end:

bug:=proc()
	lprint(` Bug alert: please send this example to hoeij@sci.kun.nl`);
	ERROR(args)
end:

################################
#      computation in k((x))   #----------------------------------------------
################################		diffop_laurent

# where k is the groundfield, an algebraic extension of Q
# variables:
# LL.1 LL.2 ... these stand for the Laurent series in x
# with finite pole order.
# accuracy_laurent value_laurent description_laurent: these are
# tables

# The purpose of this section is computation with infinite series.
# This section uses the following variables
macro(	LL=`diffop/LL`,
	accuracy_laurent=`diffop/accuracy_laurent`,
	description_laurent=`diffop/description_laurent`,
	set_laurents=`diffop/set_laurents`,
	value_laurent=`diffop/value_laurent`
):
set_laurents:={}:

# It uses the global variable x and the procedures:
macro(	differentiate=`diffop/differentiate`,
	eval_laurent=`diffop/eval_laurent`,
	lowerbound_val=`diffop/lowerbound_val`,
	max0_val=`diffop/max0_val`,
	new_laurent=`diffop/new_laurent`,
	new_laurent2=`diffop/new_laurent2`,
	nmterms_laurent=`diffop/nmterms_laurent`,
	nt=`diffop/nt`,
	nterms_expression=`diffop/nterms_expression`,
	nthterm_laurent=`diffop/nthterm_laurent`,
	ramification_laur=`diffop/ramification_laur`,
	lift_ramification_laur=`diffop/lift_ramification_laur`,
	subsvalueslaurents=`diffop/subsvalueslaurents`,
	series_val=`diffop/pseries_val`,
	valuation_laurent=`diffop/valuation_laurent`
):

# g_ext: gives a list of the algebraic extensions.
g_ext_r:=proc(a)
local v,vv,i,tail;
options remember;
	v:=indets(a,RootOf);
	if nops(v)=0 then RETURN([]) fi;
	vv:={};
	for i in v do vv:=vv union indets(op(i),RootOf) od;
	tail:=g_ext_r(vv);
	v:=[op(v minus vv)];
	[op(v),op(tail)]
end:

# Gives the algebraic extensions appearing in aa.
# For laurent series, the groundfield must be given in their descriptions.
#
# CF 95-11-01: Modification -- no longer uses lists in the substitution of
#              description_laurent since lists can now be added together in R4
#              Instead, use an unevaluated function call `diffop/LIST`.  This
#              gets handled in the same way that lists do by g_ext_r.
g_ext:=proc(aa)
global  g_conversion1, g_conversion2;
local v,i,result,ii,vv;
options remember;
	v:=aa;
	if indets(aa) intersect set_laurents<>{} then
		for i in indets(aa) intersect set_laurents do
			v:=subs(i=`diffop/LIST`(op(description_laurent[i])),v)
		od;
	fi;
	v:=g_ext_r(subs(g_conversion2,v));
	vv:=subs(g_conversion1,v);
	result:=NULL;
	for i from 0 to nops(g_conversion2) do if member(`diffop/rootof`.i,vv)
		then result:=`diffop/rootof`.i,result
	fi od;
	for i from nops(v) by -1 to 1 do if not member(subs(g_conversion1,v[i])
	 ,{seq(`diffop/rootof`.ii,ii=0..nops(g_conversion2))}) then
		if g_conversion1={} then g_conversion1:=NULL fi;
		vv:=nops(g_conversion2);
		g_conversion1:=v[i]=`diffop/rootof`.vv,g_conversion1;
		g_conversion2:={`diffop/rootof`.vv=v[i],op(g_conversion2)};
		result:=subs(g_conversion1,v[i]),result
	fi od;
	[result]
end:

new_laurent:=proc()
	global  set_laurents, accuracy_laurent, value_laurent, description_laurent;
	local i, first, last, list_length;
	i:=nops(set_laurents)+1;
	set_laurents:=set_laurents union {LL.i};
	if nargs=3 then
		accuracy_laurent[LL.i]:=args[1];
		value_laurent[LL.i]:=args[2];
                list_length:=nops(args[3]):
                if list_length < 1 then
                    first:=NULL;
                    last:=NULL;
                elif list_length < 2 then
                    first:=op(args[3][1..min(1,nops(args[3]))]);
                    last:=NULL;
                else
                    first:=op(args[3][1..min(1,nops(args[3]))]);
                    last:=op(args[3][2..nops(args[3])]);
                fi;
#lprint();
#lprint(`-----begin-------debug------`);
#print(args[3]);
#print(first);
#print(last);
#lprint(`-----end-------debug------`);
#lprint();

		description_laurent[LL.i]:= [first, LL.i, last];
		 #[op(args[3][1..min(1,nops(args[3]))]),LL.i
		 #,op(args[3][2..nops(args[3])])]
	else
		accuracy_laurent[LL.i]:=NULL;
		value_laurent[LL.i]:=NULL;
		description_laurent[LL.i]:=NULL
	fi;
	LL.i
end:

subsvalueslaurents:=proc(a)
	local i,res;
	res:=a;
	for i in indets(a) intersect set_laurents do
		res:=subs(i=value_laurent[i],res)
	od;
	res
end:

# Computes a laurent series modulo x^n
# Input: laurent series, upperbound n, (optional) lower bound m
nmterms_laurent:=proc()
	global  accuracy_laurent, value_laurent;
	local g,procedure,dummy;
	if accuracy_laurent[args[1]]<args[2] then
		g:=description_laurent[args[1]];
		procedure:=g[1];
		g:=procedure(op(g[2..nops(g)]),args[2]);
		if g=NULL then
			# procedure has handled everything, checking accuracy:
			if accuracy_laurent[args[1]]<args[2] then
				RETURN(nmterms_laurent(args))
			fi
		else
			accuracy_laurent[args[1]]:=args[2];
			value_laurent[args[1]]:=g
		fi
	fi;
	g:=value_laurent[args[1]];
	convert([seq(coeff(g,x,dummy)*x^dummy,dummy=max(args[3..nargs],
	 ldegree(g,x))..args[2]-1)],`+`)
end:

# Gives the n'th coefficient
nthterm_laurent:=proc(l,n)
	if not member(l,set_laurents) then RETURN(coeff(l,x,n)) fi;
	if accuracy_laurent[l]<=n then
		coeff(nmterms_laurent(l,n+1,n),x,n)
	else
		coeff(value_laurent[l],x,n)
	fi
end:

# Ramification of an element in k((x)). Needed because the
# procedure ramification_of allows only degree(f,xDF)>0.
ramification_laur:=proc(L,r,ext)
	options remember;
	if r=x then RETURN(L) fi;
	new_laurent(max0_val(L)*degree(r,x),0,[lift_ramification_laur,L,r,ext])
end:

lift_ramification_laur:=proc(laur,L,r,ext,ac)
	g_expand(subs(x=r,nmterms_laurent(L,ceil(ac/degree(r,x))
	,ceil(accuracy_laurent[laur]/degree(r,x)))),ext)+
	value_laurent[laur]
end:

# Converts an expression to a Laurent series.
# Arguments: f, optional ext,
# optional var (then f is a polynomial in var)
eval_laurent:=proc()
	local v,f,ext,i;
	options remember;
	f:=args[1];
	if nargs=1 then ext:=g_ext(f) else ext:=args[2] fi;
	if nargs=3 then
		v:=args[3];
		if lcoeff(f,v)=1 then
			RETURN(expand(convert([seq(eval_laurent(coeff(f,v,i)
			 ,ext)*v^i,i=0..degree(f,v)-1),v^degree(f,v)],`+`)))
		else
			RETURN(expand(convert([seq(eval_laurent(coeff(f,v,i)
			 ,ext)*v^i,i=0..degree(f,v))],`+`)))
		fi
	fi;
	if has(f,ExpInt) or has(f,log(x)) then for i in indets(f,anything) do
	 if (type(i,function) and op(0,i)=ExpInt) or i=log(x) then
		RETURN(expand(convert([seq(i^v*eval_laurent(coeff(f,i,v),ext)
		,v=ldegree(f,i)..degree(f,i))],`+`)))
	fi od fi;
	if member(f,set_laurents) then RETURN(f) fi;
	v:=indets(f) intersect set_laurents;
	if v={} then if type(f,ratpoly(rational,[x,op(ext)])) then
		f:=[g_expand(numer(f),ext),g_expand(denom(f),ext)];
		if type(f[2],polynom(rational,x))
		and ldegree(f[2],x)=degree(f[2],x) then
			f:=expand(f[1]/f[2]);
			RETURN(new_laurent(0,truncate_x(f,f,0),[truncate_x,f]))
		fi;
		RETURN(new_laurent2([seq(new_laurent(0,truncate_x(i,i,0),
		 [truncate_x,i]),i=f)],`/`,ext))
	elif type(f,procedure) then
		RETURN(new_laurent2(f,procedure,ext))
	else
		RETURN(new_laurent2(evala(Normal(subs(g_conversion2,f))),maple,ext))
	fi fi;
	if type(f,polynom(rational,v)) and degree(f,v)<=2 and type(f,`+`)
	and coeffs(f,v)=1 then
		RETURN(new_laurent2([f,v],`deg 2 polynom`,ext))
	fi;
	v:=[op(v)];
	v:=v[1];
	if f=x^degree(f,x)*v then
		RETURN(new_laurent2([degree(f,x),v],`*x^n`,ext))
	elif not type(f,polynom(anything,v)) then
		v:=g_normal(f);
		RETURN(new_laurent2(map(eval_laurent,[numer(v),denom(v)])
		 ,`/`,ext))
	fi;
	f:=collect(f,v);
	if f=v^degree(f,v) then
		f:=eval_laurent(f/v);
		RETURN(new_laurent2([f,v],`*`,ext))
	fi;
	new_laurent2([eval_laurent(coeff(f,v,0)),new_laurent2(
	 [eval_laurent(expand((f-coeff(f,v,0))/v)),v],`*`,ext)],`+`,ext)
end:


# For use with eval_laurent.
new_laurent2:=proc()
	options remember;
	if member(`*`,[args]) or member(`deg 2 polynom`,[args]) then
		new_laurent(2*lowerbound_val([args]),0,[nterms_expression,args])
	else
		new_laurent(-10,nterms_expression(0,args,-10),[nterms_expression,args])
	fi
end:

# For use with eval_laurent
# This procedure computes an expression mod x^n.
nterms_expression:=proc(lau,aa,what,ext,n)
	local a,s,d,v1,v2;
if what=maple then
	s:=series(aa,x=0,max(n+series_val(denom(aa)),
	 series_val(denom(aa))+2));
	if order(s)<>infinity then
	d:=0;
	while order(s)<>infinity and order(s)<n and s<>aa do
		d:=d+1;
		s:=series(aa,x=0,max(n+series_val(denom(aa))
		 ,series_val(denom(aa))+2)+d)
	od
	fi;
	s:=evala(Expand(convert(s,polynom)));
	subs(g_conversion1,convert(
	 [seq(coeff(s,x,d)*x^d,d=ldegree(s,x)..n-1)],`+`))
elif what=`+` then
	if lau<>0 then
		convert(map(nmterms_laurent,aa,n,accuracy_laurent[lau]),`+`)
		 +value_laurent[lau]
	else
		convert(map(nmterms_laurent,aa,n),`+`)
	fi
elif what=`*` then
	v2:=max0_val(aa[2]);
	a:=nmterms_laurent(aa[1],n-v2);
	modm(g_expand(value_laurent[lau]+convert([seq(coeff(a,x,d)*x^d
	 *nmterms_laurent(aa[2],n-d,accuracy_laurent[lau]-d)
	 ,d=ldegree(a,x)..degree(a,x))],`+`),ext))
elif what=`*x^n` then
	expand(x^aa[1]*nmterms_laurent(aa[2],n-aa[1]))
elif what=`/` then
	v1:=max0_val(aa[1]);
	v2:=valuation_laurent(aa[2],infinity);
	if lau=0 or value_laurent[lau]=0 then
		g_quotient(nmterms_laurent(aa[1],v2+n),nmterms_laurent(
		 aa[2],2*v2+n-v1),n,ext)
	else
		a:=nmterms_laurent(aa[2],2*v2+n-v1);
		value_laurent[lau]+g_quotient(nmterms_laurent(aa[1],n+v2
		 ,accuracy_laurent[lau]+v2)-
		convert([seq(convert([seq(x^s*coeff(value_laurent[lau],x,s)
		 ,s=accuracy_laurent[lau]-d+v2..n-d-1+v2)],`+`)*x^d
		 *coeff(a,x,d),d=ldegree(a,x)..degree(a,x))],`+`),a,n,ext)
	fi
elif what=differentiate then
	diff(nmterms_laurent(aa,n+1),x)
elif what=`deg 2 polynom` then
	# Now aa[1] is a sum
	convert(map(nterms_expression,[op(aa[1])],accuracy_laurent[lau],
	`deg 2 1 term`,ext,n),`+`)
elif what=`deg 2 1 term` then
	if type(lau,`*`) then
		v2:=max0_val(op(2,lau));
		a:=nmterms_laurent(op(1,lau),n-v2);
		modm(g_expand(convert([seq(coeff(a,x,d)*x^d
		 *nmterms_laurent(op(2,lau),n-d,aa-d)
		 ,d=ldegree(a,x)..degree(a,x))],`+`),ext))
	else
		nmterms_laurent(lau,n,aa)
	fi
elif what=procedure then
	convert([value_laurent[lau],seq(aa(d)*x^d
	,d=accuracy_laurent[lau]..n)],`+`)
else bug()
fi
end:

# Gives the derivative of an expression
differentiate:=proc(ff)
	local f,dummy,v;
	v:=indets(ff) intersect set_laurents;
	if v={} then RETURN(diff(ff,x)) fi;
	v:=[op(v)];
	v:=v[1];
	f:=collect(ff,v);
	if type(f,polynom(anything,v)) then
		expand(convert([seq(differentiate(coeff(f,v,dummy))*v^dummy
		 +dummy*v^(dummy-1)*coeff(f,v,dummy)*new_laurent2(v
		 ,differentiate,g_ext(v)),dummy=0..degree(f,v))],`+`))
	else
		f:=g_normal(f);
		g_normal((denom(f)*differentiate(numer(f))-numer(f)
		 *differentiate(denom(f)))/denom(f)^2)
	fi
end:

# valuation of a. It looks at most as far as bound
valuation_laurent:=proc(a,bound)
	local i;
	options remember;
	if value_laurent[a]<>0 then
		RETURN(min(bound,ldegree(value_laurent[a],x)))
	fi;
	i:=accuracy_laurent[a]-1;
	while (bound=infinity or i<bound) and nmterms_laurent(a,i+1)=0 do
		i:=i+1
	od;
	if i=bound then i else ldegree(value_laurent[a],x) fi
end:

# This procedure computes the valuation of an expression.
series_val:=proc(a)
	local d,n;
	options remember;
	if a=0 then ERROR(`division by zero`) fi;
	d:=1;
	n:=0;
	while n=0 do
		d:=d+1;
		n:=evala(Expand(convert(series(a,x=0,d),polynom)))
	od;
	ldegree(n,x)
end:

# Returns a lower bound for the valuation of the Laurent series in f.
lowerbound_val:=proc(f)
	local i;
	min(seq(op([accuracy_laurent[i],ldegree(value_laurent[i],x)]),
	 i=indets(f) intersect set_laurents))
end:

# A lower bound (not much too low) for the valuation
max0_val:=proc(a)
	local v;
	v:=value_laurent[a];
	if v=0 then
		v:=accuracy_laurent[a];
		if v<0 then valuation_laurent(a,0) else v fi
	else
		ldegree(v,x)
	fi
end:

# nt (n terms), compute f mod x^n
# Only for expressions linear in Laurent series, like the standard form
# for local differential operators.
nt:=proc(f,n)
	local a,i;
	if has(f,log(x)) then RETURN(subs(a=log(x),nt(subs(log(x)=a,f),n))) fi;
	if has(f,ExpInt) then for i in indets(f,anything) do
	 if type(i,function) and op(0,i)=ExpInt then
		RETURN(subs(a=i,nt(subs(i=a,f),n)))
	fi od fi;
	if type(f,list) then RETURN(map(nt,f,n)) fi;
	map(nmterms_laurent,indets(f) intersect set_laurents,n);
	truncate(subsvalueslaurents(f),n,x,[])
end:

##############################
# Computation in k((x))[xDF] #------------------------------------------------
##############################			diffop_local

# Everything from here until the help section deals with
# the local factorization. The syntax for local operators is:
# f=xDF^n + LL.? * xDF^(n-1) + ..
# So f is monic, and has elements of set_laurents as coefficients.
# Furthermore these operators must use my own syntax for algebraic numbers,
# and must use xDF (differentiation followed by a multiplication by x)
# as a syntax for operators, instead of DF.
macro(	Newtonpolygon=`diffop/Newtonpolygon`,
	coefs_operator=`diffop/coefs_operator`,
	factor_newton=`diffop/factor_newton`,
	factor_newton2=`diffop/factor_newton2`,
	factor_op=`diffop/factor_op`,
	factor_riccati=`diffop/factor_riccati`,
	faster_riccati_split=`diffop/faster_riccati_split`,
	indeterminate=`diffop/indeterminate`,
	indeterminates_op=`diffop/indeterminates_op`,
	lift_newton=`diffop/lift_newton`,
	lift_ramification=`diffop/lift_ramification`,
	lift_rightdivision=`diffop/lift_rightdivision`,
	lift_rightfactor=`diffop/lift_rightfactor`,
	lift_rsplit=`diffop/lift_rsplit`,
	lift_rsplit2=`diffop/lift_rsplit2`,
	lift_substitute=`diffop/lift_substitute`,
	make_rightfactor=`diffop/make_rightfactor`,
	nm_mult=`diffop/nm_mult`,
	nm_block=`diffop/nm_block`,
	nm_block2=`diffop/nm_block2`,
	op_with_slope=`diffop/op_with_slope`,
	ram_laur=`diffop/ram_laur`,
	ramification_of=`diffop/ramification_of`,
	rem_lift=`diffop/rem_lift`,
	rightdivision=`diffop/rightdivision`,
	skipped_factors=`diffop/skipped_factors`,
	substitute=`diffop/substitute`
):

# Gives terms of a multiplication in k[x,xDF] from x^low to x^high
# So the accuracy is high+1
nm_mult:=proc(l,r,low,high,ext)
	local j,i;
	if nargs=6 then # use reverted multiplication
		RETURN(nm_mult(r,l,low,high,ext))
	fi;
	modm(g_expand(convert([seq(coeff(r,x,j)*x^j*subs(xDF=xDF+j
	,convert([seq(x^i*coeff(l,x,i),i=max(ldegree(l,x),low-j)..min(degree(
	l,x),high-j))],`+`)),j=max(ldegree(r,x),low-degree(l,x))..min(high
	-ldegree(l,x),degree(r,x)))],`+`),ext))
end:

# Multiplication works for both syntaxes xDF and DF, but these syntaxes
# must not be mixed.
# Multiplication of local operators is not optimized. Could be done faster
# using a liftable result, lifted by nm_mult.
mult:=proc()
	local a,b,i,Dib,result,D,xx,ext;
	options remember;
	if not type(args[nargs],list) then RETURN(mult(args,g_ext([args]))) fi;
	if has([args],RootOf) then
		RETURN(subs(g_conversion2,mult(op(subs(g_conversion1,[args])))))
	fi;
	if nargs=2 then RETURN(args[1]) fi;
	ext:=args[nargs];
	if nargs>3 then RETURN(mult(args[1],mult(args[2..nargs]),ext)) fi;
	if type(args[1],polynom(anything,x)) then
		a:=g_expand(args[1],ext)
	else
		a:=collect(args[1],g_normal)
	fi;
	b:=args[2];
	result:=0;
	if member(xDF,indets([args])) then 
		xx:=x;
		D:=xDF
	else
		xx:=1;
		D:=DF
	fi;
	for i from 0 to degree(a,D) do
		if i=0 then
			Dib:=b
		else
			Dib:=xx*differentiate(Dib)+Dib*D;
			if has(Dib,set_laurents) then
				Dib:=eval_laurent(expand(Dib),ext,D)
			elif type(Dib,polynom(anything,x)) then
				Dib:=g_expand(Dib,ext)
			else
				Dib:=collect(Dib,D,g_normal)
			fi
		fi;
		result:=result+coeff(a,D,i)*Dib
	od;
	if has(result,set_laurents) then
		result:=eval_laurent(expand(result),ext,D)
	elif type(result,polynom(anything,x)) or D=xDF then
		result:=g_expand(result,ext)
	else
		result:=collect(result,D,g_normal)
	fi;
	result
end:

# result: [a,b] such that a*right+b=f
# right must be monic
# If the 4th argument slope is specified we get a faster division, but no
# remainder. We can use this 4'th argument only if this slope is the
# only slope.
rightdivision:=proc(f,right,ext,slope)
	local a,b,t;
	options remember;
if nargs=4 then
	a:=op_with_slope(degree(f,xDF)-degree(right,xDF),slope,0,
	 [lift_rightdivision,f,right,ext]);
#       rem_lift[a]:=0; I used this variable for computation of the remainder
#       but that introduced a bug, and I didn't need the remainder anyway.
	RETURN([a,`not implemented`])
fi;
	a:=0;
	b:=f;
if member(DF,indets(f)) then # global factor
	userinfo(5,diffop,`right division`,f,right);
	while degree(b,DF)>=degree(right,DF) do
		a:=a+lcoeff(b,DF)*DF^(degree(b,DF)-degree(right,DF));
		b:=collect(b-lcoeff(b,DF)*mult(seq(DF
		 ,t=1..degree(b,DF)-degree(right,DF)),right,ext),DF,g_normal)
	od;
	userinfo(5,diffop,`done right division`)
else
	while degree(b,xDF)>=degree(right,xDF) do
		a:=a+lcoeff(b,xDF)*xDF^(degree(b,xDF)-degree(right,xDF));
		b:=eval_laurent(expand(b-lcoeff(b,xDF)*mult(seq(xDF
		 ,t=1..degree(b,xDF)-degree(right,xDF)),right,ext)),ext,xDF)
	od
fi;
	[a,b]
end:

lift_rightdivision:=proc(llaur,order_l,slope,f,right,ext,left,acc)
	global  value_laurent, accuracy_laurent;
	local lau,l,r,lr,n_lifts,n_known,mult_args,i,l_extra,le,fe,r_low;
	n_lifts:=acc-accuracy_laurent[llaur];
	if n_lifts<=0 then RETURN() fi;
	n_known:=accuracy_laurent[coeff(left,xDF,0)]+degree(left,xDF)*slope;
	if n_known=0 then
		l:=0
	else
		l:=expand(subsvalueslaurents(left))
	fi;
	r:=nm_block(right,0,n_known+n_lifts,slope,ceil);
	r_low:=coefs_operator(r,slope,-numer(slope)*degree(right,xDF),0);
	mult_args:=ceil(n_known-slope*degree(f,xDF)),
	 ceil(n_known+n_lifts)-1,ext;
#       lr:=rem_lift[left]-nm_block(f,n_known,n_known+n_lifts,slope,ceil)
#        +nm_mult(l,r,mult_args[2]+1-n_lifts,mult_args[2..3]);
	lr:=-nm_block(f,n_known,n_known+n_lifts,slope,ceil)
	 +nm_mult(l,r,mult_args);
	le:=-numer(slope)*degree(left,xDF);
	fe:=-numer(slope)*degree(f,xDF);
	for i from n_known*denom(slope) to (n_known+n_lifts)*denom(slope)-1 do
		l_extra:=coefs_operator(g_quo(-coefs_operator(lr
		 ,slope,i+fe,0),r_low,x,ext),slope,i+le,1);
		l:=l+l_extra;
		lr:=lr+nm_mult(l_extra,r,mult_args)
	od;
#       rem_lift[left]:=lr;
	l:=collect(l,xDF);
	for i from 0 to degree(left,xDF)-1 do
		lau:=coeff(left,xDF,i);
		value_laurent[lau]:=coeff(l,xDF,i);
		accuracy_laurent[lau]:=accuracy_laurent[lau]+n_lifts
	od;
	NULL
end:

# ff is f with xDF + a for xDF substituted. This procedure lifts ff.
# Note: the resulting Laurents may have a higher degree then their
# accuracy. The terms higher than the accuracy are not yet correct.
lift_substitute:=proc(l,d,slope,a,f,ext,ff,acc)
global  value_laurent, accuracy_laurent;
local i,res,n_lifts;
	n_lifts:=acc-accuracy_laurent[l];
	if n_lifts<=0 then RETURN() fi;
	i:=accuracy_laurent[coeff(ff,xDF,0)]+d*slope;
	res:=substitute(a,nm_block(f,i,i+n_lifts,slope,ceil),ext);
	for i from 0 to degree(f,xDF)-1 do
		value_laurent[coeff(ff,xDF,i)]:=coeff(res,xDF,i)
		 +value_laurent[coeff(ff,xDF,i)];
		accuracy_laurent[coeff(ff,xDF,i)]:=
		 accuracy_laurent[coeff(ff,xDF,i)]+n_lifts
	od;
	NULL
end:

# This procedure substitutes a for xDF in an operator f in internal form.
# a must be xDF + something of valuation at least -slope
# arguments: a,f,slope,shift,ext
# Other call syntax: a,f,ext
substitute:=proc()
	local i,ide,res,f,D;
if type(args[2],list) then
	f:=args[2];
	if nops(f)=4 and f[4]=`alg over k((x))` then
		RETURN([substitute(ramification_of(
		 args[1],f[3],args[nargs]),f[1],args[3..nargs]),op(f[2..4])])
	else
		RETURN([seq(substitute(args[1],i,args[3..nargs]),i=f)])
	fi
elif nargs=5 then
	op_with_slope(degree(args[2],xDF),args[3..4],
	 [lift_substitute,args[1..2],args[5]])
elif nargs=3 then
	if has(args[2],DF) then D:=DF else D:=xDF fi;
	res:=coeff(args[2],D,0);
	for i from 1 to degree(args[2],D) do
		if i=1 then ide:=args[1] else ide:=mult(args[1],ide,args[3]) fi;
		res:=res+coeff(args[2],D,i)*ide
	od;
	if type(res,polynom(anything,x)) or D=xDF then
		g_expand(res,args[3])
	else
		collect(res,D,g_normal)
	fi
elif nargs=2 or nargs=4 then substitute(args,g_ext([args]))
fi
end:

# Returns an operator having a given slope.
op_with_slope:=proc(order,slope,shift,d)
global  description_laurent;
local i,result;
	result:=xDF^args[1];
	for i from 0 to args[1]-1 do
		result:=result+new_laurent(ceil((i-args[1])*
		 slope+shift),0,[])*xDF^i # Note: shift <= 0.
	od;
	if d=[] then RETURN(result)
	elif member(d[1],{lift_newton,lift_rsplit}) then
		result:=[result,op_with_slope(d[2],slope,0,[])]
	fi;
	for i in indets(result) minus {xDF} do
		description_laurent[i]:=
		 [d[1],i,args[1..2],op(d[2..nops(d)]),result]
	od;
	result
end:

`convert/diffop`:=proc()
	local f,i,l;
	f:=args[1];
if nargs=1 then
	if member(xDF,indets(f)) then
		f:=substitute(x*DF,args)
	elif member(X,indets(f)) then # reduce's syntax
		f:=subs({X=x,Y=1},eval(subs(DF
		 =proc() if nargs=2 then xDF else xDF^args[3] fi end,f)));
		f:=op(1,sort(subs(xDF=DF,f),[x,DF]))
	elif nops(indets(subs(y(x)=1,f)))<>nops(eval(indets(subs(y(x)=1,f))))
	or has(f,D)
	then
		# Maple's syntax
		f:=convert(f,D);
		i:=1;
		l:=diff(y(x),x);
		while has(f,D) do
			f:=subs(convert(l,D)=DF^i,f);
			l:=diff(l,x);
			i:=i+1
		od;
		f:=op(1,subs(y(x)=1,f))
	fi;
	RETURN(collect(f,DF,g_normal))
elif nargs>=2 then
if nargs=3 and args[2]=truncated then
	RETURN(subs(g_conversion2,nt(f,args[3])))
elif args[2]=`expanded LCM` then
	if not (type(f,list) and f[1]=`LCM of`) then RETURN(f) fi;
	g_ext(f);
	f:=subs(g_conversion1,f);
	RETURN(subs(g_conversion2,`diffop/LCM`(f[2],f[4..nops(f)-1])))
fi;
f:=convert(f,diffop);
	if type(args[2],equation) and op(args[2])[1]=x then
		l:=op(args[2])[2];
		if subs(infinity=0,l)=l then
			f:=substitute(1/x*xDF,subs(DF=xDF,eval_laurent(subs(
			 x=x+l,collect(f,DF,g_normal)),g_ext([f,l]),DF)))
		else
			f:=substitute(-x*xDF,subs({DF=xDF,x=1/x},f))
		fi;
		#17jan:
		if nargs=3 then
			RETURN(collect(f/lcoeff(f,xDF),xDF,g_normal))
		fi;
		RETURN(eval_laurent(collect(f/lcoeff(f,xDF)
		 ,xDF,g_normal),g_ext(f),xDF))
	elif args[2]=reduce then
		RETURN(subs(x=X,coeff(f,DF,0)*Y+sum(coeff(f,DF,i)*DF(Y,X,i)
		 ,i=1..degree(f,DF)))=0)
	elif args[2]=axiom then
		RETURN(coeff(f,DF,0)*y(x)+sum(coeff(f,DF,i)*''D''(y(x),x,i)
		 ,i=1..degree(f,DF))=0)
	elif args[2]=maple then
		RETURN(sum('coeff(f,DF,i)*diff(y(x),x$i)',i=1..degree(f
		 ,DF))+coeff(f,DF,0)*y(x)=0)
	elif args[2]=monic then
		RETURN(collect(f/lcoeff(f,DF),DF,g_normal))
	fi
fi;
ERROR(`wrong arguments`)
end:

# Gives all terms from x^n to x^m of f if slope=0.
nm_block:=proc(f,n,m,slope,round_off)
	local i,res;
	if round_off(n)<=0 and 0<round_off(m)
		then res:=xDF^degree(f,xDF)
	else
		res:=0
	fi;
	for i from degree(f,xDF)-1 by -1 to 0 do
		res:=res+nmterms_laurent(coeff(f,xDF,i),
		 round_off(m+slope*(i-degree(f,xDF))),
		 round_off(n+slope*(i-degree(f,xDF))))*xDF^i
	od;
	modm(expand(res))
end:

#11jan: bugfix: need degree as input:
nm_block2:=proc(f,n,m,slope,round_off,deg)
	local i,j;
	modm(convert([seq(seq(coeff(coeff(f,xDF,i),x,j)*x^j*xDF^i,j=
	 round_off(n+slope*(i-deg))..round_off(m+slope*(i-deg))-1
	),i=0..degree(f,xDF))],`+`))
end:

#########################################
#       Factorization in k((x))[xDF]    #
#########################################

# Input: a monic operator f in xDF, in Laurent form
# a string what:
# `split over k((x))` gives a factorization over k((x))
# `all right factors` gives all safe right factors over k((x))
# `all alg factors` gives all safe right factors over alg. closure of k((x))
# `alg factor` gives one safe right factor over alg. closure k((x))
# ext: gives the constant field
factor_op:=proc()
	local r,dummy;
	options remember;
	userinfo(6,diffop,`factorizing`,args);
	if nargs=1 then
		RETURN(factor_op(eval_laurent(args[1],[],xDF)
		 ,`split over k((x))`,[]))
	fi;
	if degree(args[1],xDF)<=1 then RETURN([args[1]]) fi;
	r:=factor_newton(args[1],args[2..nargs]);
	[seq(op(factor_riccati(dummy,args[2..nargs])),dummy=r)]
end:

#################################################
#       Factorization using the Newton method   #
#################################################

# Input: an operator f, and (optional) a second argument
# Output (if called with 2 arguments): a list giving 3 things: 
#  1) the coordinates of the extreme points of the Newton polygon
#  2) the slope of a point to the next point
#  3) the Newton polynomial of this slope
# If called with only 1 argument then only the extreme points will be given.
Newtonpolygon:=proc()
	global  Newtonpolygon;
	local f,n,vals,dummy,res,m,i,powd,
	 val_powd,npg,slope;
	options remember;
	userinfo(8,diffop,`Computing Newton polygon of`,args);
	f:=args[1];
	n:=degree(f,xDF);
	vals:=[seq(valuation_laurent(coeff(f,xDF,dummy),0),dummy=0..n-1),0];
	powd:=0;
	val_powd:=min(op(vals));
	npg:=[[powd,val_powd]];
	while powd<n do
		m:=min(seq((vals[powd+dummy+1]-val_powd)/dummy
		 ,dummy=1..n-powd));
		i:=n;
		while vals[i+1]-val_powd<>(i-powd)*m do i:=i-1 od;
		powd:=i;val_powd:=vals[i+1];
		npg:=[op(npg),[i,vals[i+1]]]
	od;
	if nargs=1 then RETURN(npg) else Newtonpolygon(args[1]):=npg fi;
	res:=NULL;
	for i from 1 to nops(npg)-1 do
		# Now we compute the Newton polynomial of slope number i
		slope:=(npg[i+1][2]-npg[i][2])/(npg[i+1][1]-npg[i][1]);
		res:=res,[op(npg[i]),slope,convert([seq(
		 nthterm_laurent(coeff(f,xDF,denom(slope)*dummy+npg[i][1]),
		  dummy*numer(slope)+npg[i][2])
		 *x^dummy,dummy=0..(npg[i+1][1]-npg[i][1])/denom(slope)
		)],`+`)]
	od;
	[res,npg[nops(npg)]]
end:

# Input: operator f, monic in xDF
# a string what:
# `split over k((x))` results in a complete Newton factorization, i.e. the
#  broken Newton polygon and the gcd 1 reducible Newton polynomial cases
#  will be factored
# `all right factors` gives all possible safe right factors according to
#  the Newton method
# `all alg factors` idem
# `alg factor` gives only one right factor, the one where the slope is the
#  least steep
# A Newton polynomial like (a-1)^2 will only be factored when slope=0
# (regular singular case). No algebraic extensions will be made here, they
# will be made in factor_riccati
factor_newton:=proc(f,what,ext)
	global  skipped_factors;
	local np,v,dummy,i,j,k,d,e,res,unsafe,n_unsafe;
	options remember;
	np:=Newtonpolygon(f,`include the Newton polynomials`);
	userinfo(7,diffop,`Newton method factorizing`,args);
	res:=NULL;
	unsafe:={};
for k from 1 to nops(np)-1 do
	v:=g_factors(np[k][4],ext)[2];
	if np[k][3]<>0 then
		v:=[seq([g_expand(dummy[1]^dummy[2],ext),0],dummy=v)]
	else
		# regular singular case, we compute the unsafe factors
		unsafe:={};
		for i in v do n_unsafe[i]:=0 od;
		for i from 1 to nops(v)-1 do for j from i+1 to nops(v) do if
		 degree(v[i][1],x)=degree(v[j][1],x) then
			d:=degree(v[i][1],x)-1;
			e:=g_expand(coeff(v[i][1],x,d)-coeff(v[j][1],x,d),ext);
			if type(e,integer) and e<>0 and irem(e,d+1)=0 and
			 g_expand(v[i][1]-subs(x=x+e/(d+1),v[j][1]),ext)=0 then if
				e>0 then
					unsafe:=unsafe union {v[i]};
					n_unsafe[v[j]]:=n_unsafe[v[j]]+v[i][2]
				else
					unsafe:=unsafe union {v[j]};
					n_unsafe[v[i]]:=n_unsafe[v[i]]+v[j][2]

			fi fi
		fi od od;
		v:=[op({op(v)} minus unsafe)]
	fi;
	# Now v contains the safe right factors of slope number k
	for i in v do
		if degree(v[1][1],x)*denom(np[k][3])=degree(f,xDF) then
			RETURN([f])
		fi;
		j:=factor_newton2(f,i[1],np[k][3],ext);
		if np[k][3]=0 then
			skipped_factors(j[2]):=n_unsafe[i]
		fi;
		if what=`alg factor` then
			RETURN([j[2]])
		elif what=`split over k((x))` then
			RETURN([op(factor_newton(j[1],what,ext)),j[2]])
		fi;
		res:=res,j[2]
	od
od; [res]
end:

# For use with: factor_op( .. , `all right factors`, [])
# It gives for each factor the number of factors we skipped, because
# the integer difference of the residues (the unsafe factors).
skipped_factors:=proc(f)
	local v;
	options remember;
	v:=[op(indets(f) intersect set_laurents)];
	v:=description_laurent[v[1]];
	if v[1]=lift_rightfactor then
		skipped_factors(v[5])
	elif v[1]=lift_rsplit then
		skipped_factors(v[8])
	elif v[1]=lift_substitute then
		skipped_factors(v[6])
	elif member(v[1],{lift_newton,nterms_expression,truncate_x}) then
	# the options remember must treat the regular singular case
		0
	else
		ERROR(`?`)
	fi
end:

# Factor monic operator f, such that r is the Newton polynomial of
# the right factor
# Output: a list of 2 factors
#
# Change (Sep 94): Added an extra option to give a factorization [L',R']
# such that R' has the specified slope and Newton polynomial, but
# is actually the left factor. f=R'L'. This option is used by
# factor_newton_LCLM. The only change is in nm_mult. The only changes
# in factor_newton2 and lift_newton are meant carry this option
# to nm_mult. This option will probably not work in the regular singular
# case (that would require changes to lift_newton that I don't
# want to make), or if L' and R' have some characteristic classes in
# common (because then such a splitting is not always possible).
#
factor_newton2:=proc(f,r,slope,ext,extra_option)
	global  rem_lift;
	local np,l,i,res,shift,sw;
	options remember;
	if nargs=5 and extra_option=`swap multiplication in nm_mult` then
		sw:=extra_option
	else
		sw:=`use normal multiplication`
	fi;
	userinfo(7,diffop,`Computing factor with Newton polynomial:`,r);
	np:=Newtonpolygon(f,`include the Newton polynomials`);
	for i from 1 to nops(np)-1 do if np[i][3]=slope then l:=i fi od;
	if l=evaln(l) then ERROR(`slope not found`) fi;
	shift:=min(0,np[l][2]+(degree(f,xDF)-np[l][1])*slope);
	# We will compute the Newton polynomial of the left factor
	np:=g_expand(x^np[l][1]*subs(x=x^denom(slope),g_quo(np[l][4],r,x,ext)),ext);
	res:=op_with_slope(degree(f,xDF)-degree(r,x)*denom(slope)
	 ,slope,shift,[lift_newton,degree(r,x)*denom(slope),f,np,
	 subs(x=x^denom(slope),r),shift,sw,ext]);
	rem_lift[res]:=0;
	res
end:

# This procedure lifts a Newton factorization. That is a combination of
# a Newton polygon factorization and a Newton polynomial factorization, see
# my paper. Meaning of the variables:
# llaur: the Laurent series we lift
# acc: desired accuracy of LL. Note that the other series in left, right
# are lifted too.
# n_lifts: number of lifts to do
# n_known: number of computed coefficients
# slope
# mult_args: arguments for nm_mult to determine which terms are needed
# ff, left, right: exact operators f=left*right
# l, r: the lowest parts of left and right
# f: a middle part of ff, precisely the terms we need
# lr: the product l*r
# rem_lift: Those terms of the previous lift of lr, that we can use now.
# l_low, r_low: the lowest line of left and right. These have gcd 1.
# If they have gcd<>1 than it is not uniquely liftable.
# l_extra, r_extra: the new terms we computed of l and r.
# ext: the algebraic extension
lift_newton:=proc(llaur,order_l,slope,order_r,ff,l_low,r_low,shift,sw,ext,v,acc)
	global  rem_lift, value_laurent, accuracy_laurent;
	local f,lr,l,r,l_extra,r_extra,s,t,n_known,i,lau
	 ,n_lifts,mult_args,left,right,ll_low,rr_low,fe,le,re;
	n_lifts:=acc-accuracy_laurent[llaur];
	if n_lifts<=0 then RETURN() fi;
	left:=v[1];
	right:=v[2];
	n_known:=accuracy_laurent[coeff(right,xDF,0)]+degree(right,xDF)*slope;
	mult_args:=ceil(n_known+shift-slope*degree(ff,xDF)),
	 ceil(n_known+shift+n_lifts)-1,ext;
	if sw=`swap multiplication in nm_mult` then
		mult_args:=mult_args,sw
	fi;
	f:=nm_block(ff,n_known+shift,n_known+n_lifts+shift,slope,ceil);
	l:=expand(subsvalueslaurents(left));
	if n_known+shift<=0 then l:=l-xDF^degree(l,xDF) fi;
	if n_known=0 then
		r:=0
	else
		r:=expand(subsvalueslaurents(right))
	fi;
## CF 95-11-01:  BUG ##
## 
#       lr:=rem_lift[v]-f+nm_mult(l,r,mult_args[2]+1-n_lifts,
#                                 op(mult_args[2..nops([mult_args])]));
#                                 ^^
#  No need to use `op' here; mult_args[...] is already a sequence, not a list.
#  If you use op(), you end up doing op(0,[`...`]) which means "what is the
#  type of [....]", since op(0,blah) tells you the type of blah .
##
	lr:=rem_lift[v]-f+nm_mult(l,r,mult_args[2]+1-n_lifts,
	                          mult_args[2..nops([mult_args])]);
	#Just to be sure:
	#lr:=-f+nm_mult(l,r,mult_args);
if slope=0 then # regular singular case
	rr_low:=subs(x=xDF,r_low);
	for i from n_known to n_known+n_lifts-1 do
		if i=0 then
			r_extra:=subs(x=xDF,r_low);
			l_extra:=expand(x^shift*subs(x=xDF,l_low))
		else
			ll_low:=subs(x=xDF,g_expand(subs(x=x+i,l_low),ext));
			s:=g_gcdex(rr_low,ll_low,xDF,ext);
			if s[1]<>1 then
				ERROR(`unsafe factor`)
			fi;
			t:=s[3];
			s:=s[2];
			r_extra:=-g_rem(t*coeff(lr,x,i+shift),rr_low,xDF,ext);
			l_extra:=modm(collect(-x^(i+shift)*g_quo(coeff(lr,x
			 ,i+shift)+r_extra*ll_low,rr_low,xDF,ext),x));
			r_extra:=modm(collect(x^i*r_extra,x))
		fi;
		l:=l+l_extra;
		lr:=modm(lr+nm_mult(l,r_extra,mult_args)+nm_mult(l_extra,r
		 ,mult_args));
		r:=r+r_extra
	od
else
	s:=g_gcdex(r_low,l_low,x,ext);
	t:=s[3];
	s:=s[2];
	le:=denom(slope)*shift-numer(slope)*degree(left,xDF);
	re:=-numer(slope)*degree(right,xDF);
	fe:=denom(slope)*shift-numer(slope)*degree(ff,xDF);
	for i from n_known*denom(slope) to (n_known+n_lifts)*denom(slope)-1 do
		if i=0 then
			r_extra:=coefs_operator(r_low,slope,re,1);
			l_extra:=coefs_operator(l_low,slope,le,1)
		else
			r_extra:=g_rem(t*coefs_operator(lr,slope,
			 i+fe,0),r_low,x,ext);
			l_extra:=coefs_operator(modm(g_expand(-g_quo((
			 coefs_operator(lr,slope,i+fe,0)-r_extra*l_low),r_low,x,ext)
			 ,ext)),slope,i+le,1);
			r_extra:=coefs_operator(modm(-r_extra),slope,i+re,1)
		fi;
		l:=l+l_extra;
		lr:=lr+nm_mult(l_extra,r,mult_args)+nm_mult(l,r_extra,mult_args);
		r:=r+r_extra
	od;
fi;
	rem_lift[v]:=lr;
	for f in [[left,collect(l,xDF)],[right,collect(r,xDF)]] do
		for i from 0 to degree(f[1],xDF)-1 do
			lau:=coeff(f[1],xDF,i);
			value_laurent[lau]:=coeff(f[2],xDF,i);
			accuracy_laurent[lau]:=accuracy_laurent[lau]+n_lifts
		od
	od;
	NULL
end:

# Input: an operator or a polynomial f
# Output: a polynomial if f is an operator, and an operator if f is
# a polynomial. 
# This procedure either takes coefficients from an operator and gives them as
# a polynomial, or constructs an operator from a given set of coefficients
# (given as a polynomial). Something like a Newton polynomial.
# slope must be > 0
coefs_operator:=proc(f,slope,i,what)
	local dummy,start_x,start_D;
	start_D:=modp(-i/numer(slope),denom(slope));
	start_x:=start_D*slope+i/denom(slope);
	if what=0 then
		convert([seq(coeff(coeff(f,x,start_x+numer(slope)*dummy),xDF,
		 start_D+dummy*denom(slope))*x^(start_D+dummy*denom(slope)),
		 dummy=0..floor(degree(f,xDF)/denom(slope)))],`+`)
	else
		convert([seq(coeff(f,x,start_D+dummy*denom(slope))*x^(start_x
		 +numer(slope)*dummy)*xDF^(start_D+dummy*denom(slope))
		 ,dummy=0..ceil(degree(f,x)/denom(slope)))],`+`)
	fi
end:

#################################################
#       Factorization using a Riccati solution  #
#################################################

# f should be monic, have only 1 slope, and the Newton polynomial is the
# power of an irreducible polynomial. The dummy argument is because of the
# options remember in case of a different groundfield.
# output: same as factor_op
factor_riccati:=proc(f,what,ext)
	local np,slope,gr,res,r,l,lr,i,v,n,dummy;
	options remember;
	np:=Newtonpolygon(f,`include the Newton polynomials`)[1];
	slope:=np[3];
	np:=g_factors(np[4],ext)[2][1];
if (degree(f,xDF)<=1) or (np[2]=1 and member(what
 ,{`split over k((x))`,`all right factors`})) then
	RETURN([f])
elif degree(np[1],x)=1 and denom(slope)=1 then
	np:=(x-np[1])*x^(-slope);
	userinfo(8,diffop,`substituting`,xDF=xDF+np);
	v:=factor_op(substitute(xDF+np,f,slope,0,ext),what,ext);
# Bugfix:	RETURN(substitute(xDF-np,v,slope,0,ext))
	RETURN(substitute(xDF-np,v,slope,0,g_ext(v)))
elif what=`split over k((x))` then
	r:=factor_riccati(f,`alg factor`,ext);
	# An order 1 factor over the alg. closure of k((x))
	r:=make_rightfactor(f,r[1],ext); # a righthand factor over k((x))
	if r=f then RETURN([f]) fi;
	l:=rightdivision(f,r,ext,slope)[1];
	# We factorized f here in l and r, however, these l and r lift very
	# slowly, so we try
	userinfo(7,diffop,`Computing factor using Riccati solution`,r);
	lr:=faster_riccati_split(f,l,r,ext,slope);
	RETURN([op(factor_riccati(lr[1],what,ext)),lr[2]])
elif what=`all right factors` then
	v:=factor_riccati(f,`all alg factors`,ext);
	res:=NULL;
	for i in v do
		r:=make_rightfactor(f,i,ext);
		if r=f then res:=f else
			l:=rightdivision(f,r,ext,slope)[1];
			userinfo(7,diffop,
			 `Computing factor using Riccati solution`,r);
			lr:=faster_riccati_split(f,l,r,ext,slope);
			res:=res,lr[2]
		fi
	od;
	RETURN([res])
elif degree(np[1],x)>1 then
	# Now we need an algebraic extension on k
	gr:=g_zero_of(np[1],x,dummy);
	userinfo(7,diffop,`Making an algebraic extension:`,gr);
	r:=factor_newton2(f,g_expand((x-gr)^np[2],[gr,op(ext)])
	 ,slope,[gr,op(ext)]);
	r:=factor_riccati(r[2],what,[gr,op(ext)]);
	# Now we denote the algebraic extension in a list:
	res:=NULL;
	for i in r do if type(i,list) then res:=res,i
		else res:=res,[i,[gr,op(ext)],x,`alg over k((x))`]
	fi od;
	RETURN([res])
else
	# Now we need a ramification
	userinfo(7,diffop,`Applying a ramification`);
	n:=mods(1/numer(slope),denom(slope));
	r:=(x-np[1])^n*x^denom(slope);
	v:=factor_newton2(ramification_of(f,r,ext),g_expand((x-denom(slope)
	 *(x-np[1])^((1-n*numer(slope))/denom(slope)))^np[2],ext)
	 ,numer(slope),ext);
	v:=factor_riccati(v[2],what,ext);
	res:=NULL;
	for i in v do if type(i,list) then
		res:=res,[i[1],i[2],ramification_of(r,i[3],ext),i[4]]
	else
		res:=res,[i,ext,r,`alg over k((x))`]
	fi od;
	RETURN([res])
fi
end:

# gives a ramification, it maps xDF to xDF* 1/degree(r), and maps x to r
# r is a power of x (then we have a pure ramification), or a power of x
# times a constant. We allow these latter kind of ramifications, because
# then we need less algebraic extensions.
ramification_of:=proc(f,r,ext)
	options remember;
	if r=x then RETURN(f)
	elif indets(f) intersect set_laurents = {} then
		RETURN(g_expand(subs(x=r,f),ext))
	fi;
	op_with_slope(degree(f,xDF),-degree(r,x)*valuation_laurent(
	 coeff(f,xDF,0),0)/degree(f,xDF),0,[lift_ramification,f,r,ext])
end:

lift_ramification:=proc(laur,order,slope,ff,r,ext,f,acc)
global  value_laurent, accuracy_laurent;
local i,res,n_lifts;
	n_lifts:=acc-accuracy_laurent[laur];
	if n_lifts<=0 then RETURN() fi;
	n_lifts:=ceil(n_lifts/degree(r,x))*degree(r,x);
	i:=(accuracy_laurent[coeff(f,xDF,0)]+order*slope)/degree(r,x);
	res:=g_expand(degree(r,x)^order*subs({x=r,xDF=xDF/degree(r,x)},
	 nm_block(ff,i,i+n_lifts/degree(r,x),slope/degree(r,x),ceil)),ext);
	for i from 0 to degree(f,xDF)-1 do
		value_laurent[coeff(f,xDF,i)]:=
		 value_laurent[coeff(f,xDF,i)]+coeff(res,xDF,i);
		accuracy_laurent[coeff(f,xDF,i)]:=
		 accuracy_laurent[coeff(f,xDF,i)]+n_lifts
	od;
	NULL
end:

###########################################################
#  make right-factor over k((x)) using a Riccati solution #
###########################################################

# f is an operator
# ric is a right-factor of order 1 over the algebraic closure of k((x))
# output: a right factor over k((x))
make_rightfactor:=proc(f,ric,ext)
	local d,v;
	d:=degree(ric[3],x);
	v:=-valuation_laurent(coeff(ric[1],xDF,0),0)/d;
	# Now we multiply by the degree of the algebraic extension
	d:=d*degree_ext(ric[2],ext);
	if d=degree(f,xDF) then RETURN(f) fi;
	op_with_slope(d,v,0,[lift_rightfactor,ric,ext,2])
end:

# This procedure lifts the factor generated by make_rightfactor. It is rather
# slow, therefor we also have a faster lift method. We still need this
# procedure because the faster method does not work for the first few lifts.
# Meaning of the variables:
# ric: solution of the Riccati, i.e. 1st order factor over alg. clos. k((x))
# ram: ramification is a map x -> c * x^(ram) for a constant c.
# b[0], b[1], ..: these are used as an Ansatz
# This procedure generates linear equations stating that ric is a righthand
# factor. This righthand factor gets computed by solving these equations.
lift_rightfactor:=proc(dummy_laur,order,slope,ric,ext,acc,som,dummy_ac)
	global  value_laurent, accuracy_laurent, description_laurent;
	local s,r,ram,rp,i,b,dummy,fout,l,ld,extl;
	ram:=degree(ric[3],x);
	l:=coeff(ric[1],xDF,0);
	r:=-nmterms_laurent(coeff(ric[1],xDF,0),acc);
	ld:=min(0,ldegree(r,x));
	rp:=1;
	s:=b[0];
	for i from 1 to order do
		l:=(i-1)*ld+acc;
		rp:=truncate(x*diff(rp,x)+r*rp,l,x,ext)/ram+fout[i]*x^l;
		s:=s+b[i]*rp
	od;
	s:=g_expand(subs(b[order]=1,s),ric[2]);
	extl:=NULL;
	if ram>1 then
		s:=subs(x=g_zero_of(subs(x=xDF,ric[3])-x,xDF,'extl'),s);
		extl:=eval(extl);
#11jan: bug: g_expand does not work for negative powers of alg. expressions.
s:=expand(s/extl^ldegree(s,extl));
		s:=g_expand(s,[extl,op(ric[2])])
	fi;
	s:={ext_to_coeffs(s,ext)};
	assign(g_solve(s,{seq(b[dummy],dummy=0..order-1)},ext,0));
	for i from 0 to order-1 do
		l:=coeff(som,xDF,i);
		b[i]:=nterms_expression(0,g_normal(b[i]),maple,ext
		 ,accuracy_laurent[l]+3);
		b[i]:=collect(b[i],g_normal);
		s:=ldegree(b[i],x)-1;
		while indets(coeff(b[i],x,s)) intersect {seq(fout[dummy]
		 ,dummy=1..order)} = {} and s<accuracy_laurent[l]+3 do
			s:=s+1
		od;
		value_laurent[l]:=truncate(b[i],s,x,ext);
		accuracy_laurent[l]:=s;
		description_laurent[l]:=[lift_rightfactor,l,args[2..5]
		 ,acc+2*order,som]
	od;
	NULL
end:

#################################
#       Faster split method     #
#################################

# Input: f=left*right
# Output: l and r such that l=left and r=right, and such that l and r lift
# faster then left and right.
faster_riccati_split:=proc(f,left,right,ext,slope)
#       RETURN([left,right]);
	op_with_slope(degree(left,xDF),slope,0,[lift_rsplit,degree(right,xDF)
	 ,f,left,right,ext,2])
end:

lift_rsplit:=proc(laur,order_l,slope,order_r,f,left,right,ext,n_ind,v,acc)
	global  description_laurent, value_laurent, accuracy_laurent;
	local try,i,lau,aclau,mo,j;
	try:=lift_rsplit2(laur,f,slope,op(v),n_ind,ext,acc);
	if try=`n_ind is too small` then # I must use more indeterminates
		userinfo(6,diffop,`Increasing the coprime index of`,right,`to`,n_ind+1);
		for i in indets(v) minus {xDF} do
			description_laurent[i]:=
			 [lift_rsplit,i,args[2..8],n_ind+1,v]
		od;
	elif try=`too few terms are known` then
		# the slow procedure lift_rightfactor will compute more terms
#		mo:=modulus;
#		if mo<>0 then
#			`diffop/compute_modp`();
#			modulus:=0;
#			lau:=coeff(v[1],xDF,0);
#			nt(lau,accuracy_laurent[lau]+3);
#			modulus:=mo
# else
		for i from 0 to degree(v[1],xDF)-1 do
			lau:=coeff(v[1],xDF,i);
			aclau:=accuracy_laurent[lau]+1;
			value_laurent[lau]:=nmterms_laurent(coeff(left,xDF,i)
			 ,aclau);
			accuracy_laurent[lau]:=aclau
		od;
		for i from 0 to degree(v[2],xDF)-1 do
			lau:=coeff(v[2],xDF,i);
			aclau:=accuracy_laurent[lau]+1;
			value_laurent[lau]:=nmterms_laurent(coeff(right,xDF,i)
			 ,aclau);
			accuracy_laurent[lau]:=aclau
		od
# fi
	fi;
	NULL
end:

lift_rsplit2:=proc(llaur,ff,slope,left,right,n_ind,ext,acc)
	global  value_laurent, accuracy_laurent;
	local f,lr,l,r,l_extra,r_extra,l_low,r_low,
		left_ind,right_ind,
		n_known,i,v,all_ind,lau,n_lifts,mult_args;
	n_lifts:=acc-accuracy_laurent[llaur]; # number of lifts
	if n_lifts<=0 then RETURN() fi;
	n_known:=accuracy_laurent[coeff(right,xDF,0)]+degree(right,xDF)*slope;
	# number of computed coefficients
	if n_known<n_ind then
		RETURN(`too few terms are known`)
	fi;
	l:=expand(subsvalueslaurents(left));
	r:=expand(subsvalueslaurents(right));
	mult_args:=ceil(n_known-slope*degree(ff,xDF))
	 ,ceil(n_known+n_lifts+n_ind-2),ext;
	f:=nm_block(ff,n_known,n_known+n_lifts+n_ind,slope,ceil);
	lr:=nm_mult(l,r,mult_args)-f;
	left_ind:=indeterminates_op(slope,n_ind,left);
	right_ind:=indeterminates_op(slope,n_ind,right);
	all_ind:=indets([left_ind,right_ind]) minus {xDF,x};
	l_low:=nm_block(left,0,n_ind,slope,ceil);
	r_low:=nm_block(right,0,n_ind,slope,ceil);
	for i from 1 to n_lifts do
		l_extra:=expand(x^n_known*left_ind);
		r_extra:=expand(x^n_known*right_ind);
		v:=lr+nm_mult(l_extra,r_low,mult_args)
		 +nm_mult(l_low,r_extra,mult_args);
		v:=nm_block2(v,n_known,n_known+n_ind,slope,ceil,degree(ff,xDF));
		v:=g_solve({coeffs(v,[x,xDF])});
		l_extra:=g_expand(subs(v,nm_block2(l_extra,n_known,
		 1+n_known,slope,ceil,degree(left,xDF))),ext);
		r_extra:=g_expand(subs(v,nm_block2(r_extra,n_known,
		 1+n_known,slope,ceil,degree(right,xDF))),ext);
		if indets([l_extra,r_extra]) intersect all_ind <> {} then
			RETURN(`n_ind is too small`)
		fi;
		lr:=lr+nm_mult(l+l_extra,r_extra,mult_args)+nm_mult(l_extra,r,mult_args);
		l:=l+l_extra;
		r:=r+r_extra;
		n_known:=n_known+1
	od;
	for i from 0 to degree(left,xDF)-1 do
		lau:=coeff(left,xDF,i);
		value_laurent[lau]:=coeff(l,xDF,i);
		accuracy_laurent[lau]:=accuracy_laurent[lau]+n_lifts
	od;
	for i from 0 to degree(right,xDF)-1 do
		lau:=coeff(right,xDF,i);
		value_laurent[lau]:=coeff(r,xDF,i);
		accuracy_laurent[lau]:=accuracy_laurent[lau]+n_lifts
	od;
	NULL
end:

indeterminates_op:=proc(slope,number,f)
	local i,j,res;
	options remember;
	res:=0;
	for i from 0 to degree(f,xDF)-1 do for j from 0 to number-1 do
		res:=res+indeterminate()*x^(j+ceil(
		 (i-degree(f,xDF))*slope))*xDF^i
	od od;
	res
end:

indeterminate:=proc() local r;r end:

###############################################
#  Formal solutions of differential equations #--------------------------------
###############################################		diffop_formalsol

# This part of diffop computes solutions of f=xDF^n+laur*xDF^(n-1)+...
# in algclos( k((x)) )[log(x),exp( .. )]
# All solutions are given up to conjugation over k((x))
# The result is a list which contains a list of solutions for every characteristic
# class. Such a list for 1 class looks like:
# [ [a1,a2,..] , r] where a.i in k((x))[log(x)]
# The corresponding solutions are: a.i*ExpInt(r) where ExpInt(r)=exp(int(r/x,x))
# If any ramifications (then x is not really x but a root of a*t^n=x) have been
# used then the list is longer and looks like:
# [ [a1,a2,..] , r,ext,ram,`alg over k((x))` ] where ram=a*x^n
# The same form is used when algebraic extensions were needed.
# Note that a characteristic class as defined in Ron Sommeling's thesis is the
# characteristic polynomial of r over k((x)). In the implementation, however, I'll
# denote the characteristic polynomial only by r. (or [r,ext,ram,`alg over k((x))`]
# if extensions over k((x)) were needed.

macro(	formal_sol=`diffop/formal_sol`,
	charclass_right=`diffop/charclass_right`,
	solve_semiregular=`diffop/solve_semiregular`,
	integrate_logs=`diffop/integrate_logs`,
	lift_integral=`diffop/lift_integral`,
	sol_1order_eq=`diffop/sol_1order_eq`,
	lift_sol1=`diffop/lift_sol1`,
	test_result=`diffop/test_result`
): # :)

# Input: an operator f of order >= 1
# Output: the formal solutions
formal_sol:=proc(f,ext)
	local v,i,res,k,c,lp,ex;
	# f may not be a factor of some other operator, otherwise charclass_right
	# gets confused. The following is a fix for that case:
	if description_laurent[coeff(f,xDF,0)]=lift_newton then
		RETURN(formal_sol(eval_laurent(f+eval_laurent(0),ext,xDF),ext))
	fi;
	v:=factor_op(f,`all alg factors`,ext); # This gives 1 order one factor
	# for every characteristic class that f has.
	res:=NULL;
for i in v do
	if type(i,list) and nops(i)=4 # Then the list i contains info about
	# the extension on k((x))
		then k:=i[1];ex:=i[2]
	else
		k:=i;ex:=ext
	fi;
	if degree(f,xDF)>1 then
		c:=charclass_right(k)
	fi;
	k:=coeff(k,xDF,0);
	lp:=-nmterms_laurent(k,1);
	if c=`multiplicity 1 factor` or degree(f,xDF)=1 then
		c:=[sol_1order_eq(eval_laurent(k+lp),ex)]
	else
		c:=solve_semiregular(c,ex)
	fi;
	# Now c[i]*ExpInt(lp) are the solutions corresponding to this particular
	# characteristic class
	test_result(formal_sol,f,ext,c,lp,i);
	lp:=[c,lp];
	if type(i,list) and nops(i)=4 then
		# algebraic extensions over k((x)) have been made, these have to
		# be included in this list
		lp:=[op(lp),op(i[2..4])]
	fi;
	res:=res,lp
od;
	[res]
end:

# r=xDF + laurent series
# Output: a string `multiplicity 1 factor` if r is the only factor with
# this particular char class.
# Otherwise r+charclass is regular singular and is the right factor of
# a certain operator obtained during the factorization. From this operator
# we take the largest right factor which consists only of this char class.
charclass_right:=proc(r)
	local d,f,ext,np,v,i;
	d:=description_laurent[coeff(r,xDF,0)];
if d[1]=lift_newton then
	if not d[4]=0 then RETURN(`multiplicity 1 factor`) fi;
	# Now we are in the regular singular case.
	f:=d[6];
	ext:=d[11];
	d:=description_laurent[coeff(f,xDF,0)];
	while d[1]=lift_newton do
		f:=d[6];
		d:=description_laurent[coeff(f,xDF,0)]
	od;
	np:=Newtonpolygon(f,`include the Newton polynomials`)[1][4];
	np:=factors(g_expand(subs(x=x-nt(r,1)+xDF,np),ext));
	v:=1;
	for i in np[2] do if type(i[1]-x,integer) then
		v:=v*i[1]^i[2]
	fi od;
	v:=expand(v);
	if lcoeff(v,x)<>1 or degree(v,x)<1 then bug()
	elif degree(v,x)=1 then RETURN(`multiplicity 1 factor`)
	fi;
	i:=substitute(2*xDF-nt(r,1),f,0,0,ext);
	if degree(i,xDF)>degree(v,x) then
		i:=factor_newton2(i,v,0,ext)[2]
	fi;
	i:=factor_op(i,`split over k((x))`,ext);
	if nops(i)<>degree(v,x) then bug() fi;
	i
elif d[1]=lift_substitute then
	charclass_right(d[6])
else bug()
fi end:

# Input: a list of factors [xDF+r1,...xDF+rn] where
# the 0'th coefficients of r.i are in Z.
# Output: the solutions
solve_semiregular:=proc(v,ext)
	local s,i;
if v=[] then RETURN([]) fi;
	s:=solve_semiregular([seq(xDF+eval_laurent(v[i]-v[nops(v)]
	 ,ext),i=1..nops(v)-1)],ext);
	s:=[op(map(integrate_logs,s,ext)),1];
	[seq(eval_laurent(expand(i*sol_1order_eq(coeff(v[nops(v)],xDF,0)
	 ,ext)),ext,log(x)),i=s)]
end:

# used by solve_semiregular
# determine g such that x*d/dx (g)=ff where ff in k((x))[log(x)]
integrate_logs:=proc(ff,ext)
	local i,d,l,f;
	options remember;
	d:=degree(ff,log(x));
if d=0 then
	if has(ff,set_laurents) then
		f:=eval_laurent(ff,ext);
		# l becomes the integral of f/x:
		nmterms_laurent(f,1,0)*log(x)+new_laurent(1,expand(int(
		 nmterms_laurent(f,0)/x,x)),[lift_integral,f,ext])
	else
		g_expand(int(ff/x,x),ext)
	fi
else
	# Bugfix (oct 94)
	l:=integrate_logs(coeff(ff,log(x),d),ext);
	l:=coeff(l,log(x),0)+log(x)*coeff(l,log(x),1)/(d+1);
	expand(l*log(x)^d+integrate_logs(convert([seq(coeff(ff,log(x),i)
	*log(x)^i,i=0..d-1)],`+`)-d*log(x)^(d-1)*coeff(l,log(x),0),ext))
fi
end:

# Used by integrate_logs
# lifts LL which is int(aa/x , x)
lift_integral:=proc(LL,aa,ext,acc)
	value_laurent[LL]+expand(int(
	nmterms_laurent(aa,acc,accuracy_laurent[LL])/x,x))
end:

# used by solve_semiregular
# Input: a Laurent series a in Z + x k[[x]]
# Output: a solution L of a*L + x*d/dx L = 0
sol_1order_eq:=proc(a,ext)
	local n;
	options remember;
	n:=-nmterms_laurent(a,1,0); # n := coefficient of x^0 in a
	if not type(n,integer) then bug() fi;
	new_laurent(n+1,x^n,[lift_sol1,a,n,ext])
end:

# used by sol_1order_eq
lift_sol1:=proc(LL,aa,n,ext,acc)
	local L,a,t,i;
	L:=value_laurent[LL];
	a:=nmterms_laurent(aa,acc-n);
	for t from accuracy_laurent[LL] to acc-1 do
		# divide the coefficient of x^t in a*L by n-t:
		L:=L+g_expand(convert([seq(coeff(L,x,i)*coeff(a,x,t-i)
		 ,i=ldegree(L,x)..degree(L,x))],`+`)/(n-t)*x^t,ext)
	od;
	L
end:

###############################################
#  Solutions of matrix differential equations #--------------------------------
###############################################		diffop_matsol

macro(	solve_mateqn=`diffop/solve_mateqn`,
	matrix_mult=`diffop/matrix_mult`,
	cyclic_vector=`diffop/cyclic_vector`,
	apply_A=`diffop/apply_A`,
	left_solutions=`diffop/left_solutions`,
	xddx_log=`diffop/xddx_log`,
	lift_xddx_log=`diffop/lift_xddx_log`,
	factor_newton_LCLM=`diffop/factor_newton_LCLM`,
	adjoint=`diffop/adjoint`,
	lift_adjoint=`diffop/lift_adjoint`,
	rational_solutions=`diffop/rational_solutions`,
	left_sol_rational=`diffop/left_sol_rational`,
	solve_matrat=`diffop/solve_matrat`
):

# A matrix
# n: solve modulo x^n
# 3rd argument: specify `x*d/dx`
SolveMat:=proc(A,n)
	local v,i,j,res,ram,e,k;
	if n=rational then RETURN(solve_matrat(A)) fi;
	k:=linalg[rowdim](A);
	if not k=linalg[coldim](A) then
		ERROR(`wrong arguments`)
	fi;
	res:=NULL;
	# This map(..,A) yields a matrix without a name. Otherwise
	# options remember causes errors.
	# v:=solve_mateqn(map(i -> convert(i,RootOf),A),args[3..nargs]);
	# In the latter line options remember simply did not work, try again:
	v:=solve_mateqn([k,k,[seq(seq(convert(A[i,j],RootOf)
	,j=1..k),i=1..k)]],args[3..nargs],`not a matrix`);
for i in v do
	if nops(i)<4 then
		ram:=x
	else
		ram:=i[4]
	fi;
#bug:	e:=exp(int(i[2]/degree(ram,x)/x,x));
	e:=exp(int(i[2]/x,x));
	for j in i[1] do
		res:=res,[seq(subs(x=RootOf(subs(x=_Z,ram)-x),nt(j[k,1]
		,n*degree(ram,x))*e),k=1..linalg[rowdim](A))]
	od
od;
	subs(g_conversion2,[res])
end:

# Gives the solution of (d/dx + A)(y) = 0 in internal format.
# If D=`x*d/dx` then x*d/dx is used instead of d/dx
# The entries of A must be rational functions in x, Laurent series are
# not yet allowed in this version.
solve_mateqn:=proc(A,D)
	local i,f,cv_mat,ext,v,res,m,ex,fs,j,ram,k;
	options remember;
	if args[nargs]=`not a matrix` then
		RETURN(solve_mateqn(linalg[matrix](op(A)),args[2..nargs-1]))
	fi;
	if nargs=1 or D<>`x*d/dx` then
		RETURN(solve_mateqn(map(i -> x*i,A),`x*d/dx`))
	fi;
	f:=cyclic_vector(A,D);
	ext:=g_ext(f);
	f:=subs(g_conversion1,f);
	cv_mat:=map((i,ex) -> eval_laurent(i,ex),linalg[transpose](evalm(
	 f[2])),ext);
	res:=NULL;
	f:=eval_laurent(f[1],ext,xDF);
	fs:={[f,cv_mat]};
	# Now try to split the problem up using an LCLM factorization:
	res:={};
while fs<>{} do
	f:=fs[1];
	v:=factor_newton_LCLM(f[1],ext);
	fs:=fs minus {f};
	if v[1]=failed then
		# Sufficiently split up, now solve f:
		v:=left_solutions(f[1],ext);
		for i in v do
			if nops(i)>2 then ex:=i[3];ram:=i[4]
			else ex:=ext;ram:=x
			fi;
			m:=map(ramification_laur,f[2],ram,ex);
			m:=linalg[matrix](linalg[rowdim](m),linalg[coldim](m),
			 [seq(seq(eval_laurent(m[j,k]*degree(ram,x)^(k-1),ex)
			 ,k=1..linalg[coldim](m)),j=1..linalg[rowdim](m))]);
			res:=res union {[[seq(matrix_mult(m,j,ex)
			 ,j=i[1])],i[2],op(i[3..nops(i)])]}
		od
	else
		fs:=fs union {seq([i[1],matrix_mult(f[2],i[2],ext)],i=v)}
	fi
od;
	test_result(solve_mateqn,A,res,ext);
	res
end:

# r is a right factor of f in D/D*f, with it corresponds a
# basis of a subspace r, xDF*r, xDF^2*r, .. and a basis
# transition. This matrix transition is multiplied by m.
# If a 4th argument is given then DF is used instead of xDF.
matrix_mult:=proc(M,r,ext)
	local m,mm,i,j,d,D;
	if nargs=4 then D:=DF else D:=xDF fi;
	m:=linalg[coldim](M);
	mm:=m-degree(r,D);
	m:=linalg[multiply](M,linalg[matrix](m,mm,[seq([seq(
	 coeff(mult(seq(D,d=2..j),r,ext),D,i-1),j=1..mm)],i=1..m)]));
	if nargs=4 then
		m
	else
		map((i,j) -> eval_laurent(i,j),m,ext)
	fi
end:

# Input: a differential operator d/dx + A where A is a matrix
# Output: list with: matrix with columns v,(d/dx+A)v,(d/dx+A)^2v,..
#  and an operator.
# Specifying the cyclic vector is possible using a third argument
# If the second argument is `x*d/dx` then x*d/dx is used as a derivation
# instead of d/dx.
cyclic_vector:=proc(A,D,vv)
	local v,n,i,M,m,df,MM1,MM2,AA1,AA2;
	if nargs=1 then RETURN(cyclic_vector(A,`d/dx`)) fi;
	n:=linalg[rowdim](A);   # A is an n by n matrix
	if n<>linalg[coldim](A) then ERROR(`wrong input`) fi;
	if nargs<3 then v:=[1,seq(0,i=2..n)] else v:=vv fi;
	M:=NULL;
	for i from 1 to n do
		if i=1 then
			v:=evalm(v)
		else
			v:=apply_A(A,v,D)
		fi;
		M:=M,convert(v,list)
	od;
	m:=linalg[transpose](evalm([M]));
	m:=linalg[linsolve](m,apply_A(A,v,D));
	# if indets([op(op(m))]) minus indets([x,v,op(op(A))]) <> {} then
	# desperate attempt to get indets working:
	MM2:=evalm(m);
	MM1:=op(MM2);
	MM2:=[op(op(MM1))];
	MM1:=indets(MM2);
	AA2:=evalm(A);
	AA1:=op(AA2);
	AA2:=[op(op(AA1))];
	AA1:=indets(AA2);
	if MM1 minus indets([x,args[2..nargs],AA1]) <> {} then
		if nargs=2 then
			v:=[seq(randpoly(x,degree=0),i=1..n)]
		else
			v:=[seq(randpoly(x,degree=degree(vv[i],x)+1),i=1..n)]
		fi;
		RETURN(cyclic_vector(A,D,v))
	fi;
	if D=`x*d/dx` then df:=xDF else df:=DF fi;
	[convert([df^n,seq(normal(-m[i+1])*df^i,i=0..n-1)],`+`),[M]]
end:

# used by cyclic_vector
apply_A:=proc(A,v,D)
	local a;
	if D=`x*d/dx` then
	map(normal,evalm(linalg[multiply](A,v)+map(a -> x*diff(a,x),v)))
	else
	map(normal,evalm(linalg[multiply](A,v)+map(a -> diff(a,x),v)))
	fi
end:


# Input: an operator f with only 1 slope.
# Now consider the ring K[x*d/dx]/(K[x*d/dx] * f)
# where K is an extension field of k((x)) that contains all solutions
# of adjoint(f)(y)=0.
# A basis of this ring is: [ (x*d/dx)^0, (x*d/dx)^1, .. ]
# so elements of this ring look like polynomials in xDF (is x*d/dx) of
# degree < n. (n is order of f).
# The output of this procedure is a list of elements of K[x*d/dx]/(K[x*d/dx] * f)
# that become 0 when multiplied by x*d/dx.
left_solutions:=proc(f,ext)
	local i,adj_sol,sol,ff,ex,ram,g,res,tres,j;
	options remember;
	adj_sol:=formal_sol(adjoint(f,ext),ext);
	tres:=NULL;
for sol in adj_sol do
	# sol[1] is a list of solutions
	# sol[2] is the characteristic class, this appears in the ExpInt part
	# sol[3..] are the algebraic extensions
	if nops(sol)=2 then
		ram:=x;
		ex:=ext
	else
		ram:=sol[4];
		ex:=sol[3]
	fi;
	ff:=ramification_of(f,ram,ex);
	res:=NULL;
	for i in sol[1] do
		# Now we must solve the following problem: give an operator g
		# such that xDF*g is zero mod ff. Use: g=i*xDF^(n-1)+.. where
		# i is a solution of the adjoint of ff.
		g:=i*xDF^(degree(f,xDF)-1);
		for j from degree(f,xDF)-2 by -1 to 0 do
			g:=g+eval_laurent(expand(i*coeff(ff,xDF,j+1)- 
			 xddx_log(coeff(g,xDF,j+1),ex)
			 -coeff(g,xDF,j+1)*eval_laurent(sol[2],ex)
			),ex,log(x))*xDF^j
		od;
		test_result(left_solutions,ExpInt(normal(sol[2]),x)*g
		 ,ff,ram,ex);
		res:=res,g
	od;
	tres:=tres,[[res],op(sol[2..nops(sol)])]
od;
	[tres]
end:

adjoint:=proc(f)
	local i,ext,v,D;
	options remember;
	userinfo(7,diffop,`computing the adjoint of`,f);
	if nargs=1 then ext:=g_ext(f) else ext:=args[2] fi;
	if has(f,xDF) and has(f,set_laurents) then
		v:=lowerbound_val(f);
		convert([xDF^degree(f,xDF),seq(new_laurent(v,0,
		 [lift_adjoint,i,f,ext])*xDF^i,i=0..degree(f,xDF)-1)],`+`)
	else
		if has(f,xDF) then D:=xDF else D:=DF fi;
		v:=convert([seq(mult(D^i*(-1)^(i+degree(f,D)),coeff(f,D,i)
		 ,ext),i=0..degree(f,D))],`+`);
		if type(v,polynom(anything,x)) then
			g_expand(v,ext)
		else
			collect(v,D,g_normal)
		fi
	fi
end:

# L is the n'th coefficient in the adjoint of f.
# ext seems to be unused. This is to make sure that ext appears in the
# descriptions of the Laurent series.
lift_adjoint:=proc(L,n,f,ext,acc)
	local a,res,i;
	a:=acc,accuracy_laurent[L];
	res:=0;
	for i from degree(f,xDF)-1 by -1 to n do
		res:=expand((-1)^(i+degree(f,xDF))*nmterms_laurent(
		 coeff(f,xDF,i),a)+x*diff(res,x)*(i+1)/(i+1-n))
	od;
	value_laurent[L]+res
end:

# Input: a polynomial in log(x) with laurent series coefficients.
# Output: x times the derivative.
xddx_log:=proc(f,ext)
	local m,d,i;
	m:=lowerbound_val(f);
	d:=degree(f,log(x));
	m:=convert([seq(new_laurent(m,0,[lift_xddx_log,i,d,f,ext])*log(x)^i
	 ,i=0..d)],`+`);
	test_result(xddx_log,f,ext,m);
	m
end:
lift_xddx_log:=proc(L,i,d,f,ext,acc)
	local a,res;
	a:=acc,accuracy_laurent[L];
	res:=value_laurent[L];
	if i<d then
		res:=res+nmterms_laurent(coeff(f,log(x),i+1),a)*(i+1)
	fi;
	expand(res+x*diff(nmterms_laurent(coeff(f,log(x),i),a),x))
end:

# Input: operator f
# Output: 2 factorizations: f = L1*R1 and f = L2*R2 such that f=LCLM(R1,R2)
# and such that R1 and R2 have no characteristic classes in common.
# No algebraic extensions are made.
factor_newton_LCLM:=proc(f,ext)
	local np,v,i,j,k,d,e,ee,c;
	options remember;
	np:=Newtonpolygon(f,`include the Newton polynomials`);
	userinfo(7,diffop,`Newton method factorizing`,args);
	k:=1;
if nops(np)>2 then # more than 1 slope.
	k:=2;
	v:=np[k][4] # the whole Newton polynomial, so
	# the result is just a Newton polygon splitting.
else
	v:=g_factors(np[k][4],ext)[2];
	if nops(v)=1 then RETURN([failed,op(v),np[k][3]]) fi;
	if np[1][3]=0 then # regular singular case
		e:=g_expand(v[1][1]^v[1][2],ext);
		ee:=1;
		d:=degree(v[1][1],x);
		for i from 2 to nops(v) do
		 c:=(coeff(v[i][1],x,d-1)-coeff(v[1][1],x,d-1))/d;
		 if # if v[i] has the same charclass as
		    # v[1] then collect it in e otherwise in ee:
		  d=degree(v[i][1],x) and type(c,integer) and
		  g_expand(v[i][1]-subs(x=x+c,v[1][1]),ext)=0 then
			e:=g_expand(e*v[i][1]^v[i][2],ext)
		else
			ee:=g_expand(ee*v[i][1]^v[i][2],ext)
		fi od;
		if ee=1 then RETURN([failed,e,np[k][3]]) fi;
		RETURN([
		 factor_newton2(f,e,np[k][3],ext),
		 factor_newton2(f,ee,np[k][3],ext) ])
	else
		v:=g_expand(v[1][1]^v[1][2],ext)
	fi;
fi;
	j:=factor_newton2(f,v,np[k][3],ext,`swap multiplication in nm_mult`);
	j:=[[j[2],j[1]],factor_newton2(f,v,np[k][3],ext)];
	test_result(factor_newton_LCLM,f,ext,j);
	j
end:

`diff/ExpInt`:=proc(a,x) a/x*ExpInt(a,x) end:

# Input: an n by n matrix A with coefficients in Q_bar(x)
# Output: a list of the rational solutions of A*y=diff(y,x)
solve_matrat:=proc(A)
	local i,f,ext,v,res,m,j,n;
	n:=linalg[rowdim](A);
	if n<>linalg[coldim](A) then ERROR(`wrong arguments`) fi;
	f:=cyclic_vector(A);
	ext:=g_ext(f);
	f:=subs(g_conversion1,f);
	m:=map(i -> subs(g_conversion2,i),linalg[transpose](evalm(f[2])));
	v:=left_sol_rational(f[1],ext);
	res:=[seq(matrix_mult(m,i,ext,`use DF`),i=v)];
	# Now convert this to lists:
	res:=[seq([seq(i[j,1],j=1..n)],i=res)];
	test_result(solve_mateqn,A,res,ext);
	res
end:

# Computes g in k(x)[DF]/( k(x)[DF]f ) such that DF*g is zero mod f.
left_sol_rational:=proc(f,ext)
	local i,sol,g,res,j;
	options remember;
	sol:=rational_solutions(adjoint(f,ext),ext);
	res:=NULL;
	for i in sol do
		# Now we must solve the following problem: give an operator g
		# such that DF*g is zero mod f. Use: g=i*DF^(n-1)+.. where
		# i is a solution of the adjoint of f.
		g:=i*DF^(degree(f,DF)-1);
		for j from degree(f,DF)-2 by -1 to 0 do
			g:=g+g_normal(i*coeff(f,DF,j+1)-diff(coeff(g,DF,j+1)
			,x))*DF^j
		od;
		test_result(left_sol_rational,g,f,ext);
		res:=res,g
	od;
	[res]
end:

# Compute the rational solutions of a differential equation by solving
# linear equations.
rational_solutions:=proc(f,ext)
local v,i,a,ansatz,bound,den,j,k,m,np,s,singularities,sol,t,zero,var,ff;
	if nargs=1 then
		v:=g_ext(f);
		RETURN(subs(g_conversion2,
		rational_solutions(subs(g_conversion1,f),v)))
	fi;
	singularities:={seq([i[1],convert(f,diffop,x=i[1]),[op(i[3]),op(ext)]]
	,i=v_ext_m(g_factor(denom(g_normal(f/lcoeff(f,DF))),ext),x))};
	s:=0;
	den:=1;
	for k in singularities do
		np:=Newtonpolygon(k[2],`include the Newton polynomials`);
		if np[1][3]<>0 then RETURN([]) fi;
		v:=factors(collect(np[1][4]/lcoeff(np[1][4],x),x,g_normal))[2];
		m:=-infinity;
		for i in v do if type(i[1]-x,integer) then
			m:=max(m,i[1]-x)
		fi od;
		if m=-infinity then RETURN([]) fi;
		t:=m/(x-k[1]);
		if k[3]<>ext then
			t:=subs(g_conversion1,evala(Trace(op(subs(
			 g_conversion2,[t,k[3],ext])))))
		fi;
		s:=s+t;
		den:=den/denom(g_normal(t))^m
	od;
	s:=g_normal(s);
	ff:=substitute(DF-s,f,ext);
	k:=convert(ff,diffop,x=infinity);
	np:=Newtonpolygon(k,`include the Newton polynomials`);
	if np[1][3]<>0 then RETURN([]) fi;
	v:=factors(collect(np[1][4]/lcoeff(np[1][4],x),x,g_normal))[2];
	m:=-infinity;
	for i in v do if type(i[1]-x,integer) then
		m:=max(m,i[1]-x)
	fi od;
	if m=-infinity then RETURN([]) fi;
	bound:=m;
	ansatz:=convert([seq(a[i]*x^i,i=0..bound)],`+`);
	zero:=ansatz*coeff(ff,DF,0);
	t:=ansatz;
	for i from 1 to degree(ff,DF) do
		t:=diff(t,x);
		zero:=zero+coeff(ff,DF,i)*t
	od;
	t:=solve({coeffs(collect(numer(subs(g_conversion2,zero)),x),x)});
	sol:=subs(t,ansatz);
	if sol=0 then RETURN([]) fi;
	var:={seq(a[i],i=0..bound)} intersect indets(sol);
	i:=[seq(g_normal(subs({seq(i=0,i=var minus {j}),j=1},sol*den)),j=var)];
	test_result(rational_solutions,f,i,ext);
	i
end:

#############################
#  debugging tools	    #-------------------------------------------------
#############################			diffop_debug

# Not included. Can be obtained by e-mail request.

has_liouvillian_solutions:=proc(f)
	if degree(f,DF)<>2 or indets(f) minus {x,DF} <> {} then
		ERROR(`wrong arguments`)
	fi;
	evalb(nops(DFactor(symmetric_power(f,6),`one step`))>1)
end:

# Input: f in Q(RootOf,..)(x)[DF]
# Output: g such that f(y)=0 -> g(y^n)=0
symmetric_power:=proc(f,n)
	local i,zero,a,y,ypower,dummy,subs_rem,res,co;
	ypower:=y(x)^n;
	zero:=a[0]*ypower;
	subs_rem:=diff(y(x),seq(x,dummy=1..degree(f,DF)))=-convert([y(x)*
	 coeff(f,DF,0)/lcoeff(f,DF),seq(diff(y(x),seq(x,dummy=1..i))*
	 coeff(f,DF,i)/lcoeff(f,DF),i=1..degree(f,DF)-1)],`+`);
	co:=combinat[numbcomp](degree(f,DF)+n,degree(f,DF));
	for i from 1 to co do
		ypower:=g_normal(subs(subs_rem,diff(ypower,x)));
		zero:=expand(zero+a[i]*ypower)
	od;
	while i>co or indets(res) intersect {seq(a[dummy],dummy=0..co)}
	 <> {} do
		i:=i-1;
		res:=subs(solve({a[i]=1,seq(a[dummy]=0,dummy=i+1..co),coeffs(
		 zero,indets(zero) minus {x,seq(a[dummy],dummy=0..co)})},{seq(
		 a[dummy],dummy=0..co)}),convert([seq(a[dummy]*DF^dummy,dummy=
		 0..i)],`+`))
	od;
	res
end:

lprint(`for help type: ?diffop`);

#####################################
#    Pade Hermite approximations    #-----------------------------------------
#####################################		diffop_pade2

# The following is a variant on Harm Derksen's (hderksen@sci.kun.nl) algorithm
# for computing Pade expansions. You can find his algorithm in the share
# library under the name pade2. G. Labahn and him found indepent of each other
# a good method for computing these expansions.

# I made the following changes:
# 1) This pade2 accepts my syntax for elements of k((x))  (laur1, laur2 etc.)
# 2) Harm pointed out to me that pade2 can also work with vectors of power
#  series. I replaced the variable z in his pade2 by z[1], z[2], etc. to 
#  achieve this.
# 3) point is always x=0
# 4) ext is the coefficients field
# Note that it is useful to have the original pade2 in order to understand
# how the method works.

macro(	lift_pade2=`diffop/lift_pade2`,
	pade2=`diffop/pade2`,
	smaller_pade2=`diffop/smaller_pade2`,
	smallest_pade2=`diffop/smallest_pade2`,
	valuation_pade2=`diffop/valuation_pade2`,
	wipe_pade2=`diffop/wipe_pade2`
):

# functionlist is a list of lists of elements in k((x))
# pade2 looks for a k[x]-linear relation between these lists (interpret these
# lists as vectors). If all these lists contain only 1 element, then this is
# basically Harm Derksen's pade2 algorithm.
pade2:=proc(functionlist,accuracy,ext)
	local i,j,n,y,z,yvars,zvars,fl,appr,degrees;
	n:=min(seq(seq(valuation_laurent(j,0),j=i),i=functionlist));
	if n<0 then
		RETURN(pade2([seq([seq(eval_laurent(x^(-n)*j)
		 ,j=i)],i=functionlist)],accuracy,ext))
	fi;
	n:=nops(functionlist);
	degrees:=[seq(0,i=1..n)]; # Not used yet, for later generalizations
	zvars:=[seq(z[i],i=1..nops(functionlist[1]))];
	yvars:=[seq(y[i],i=1..n)];
	fl:=0,[seq(convert([seq(functionlist[i][j]*z[j],j=1..nops(zvars))]
	 ,`+`),i=1..n)],n,degrees,yvars,zvars,ext,0;
	if subs(infinity=0,accuracy)<>accuracy then RETURN([fl]) fi;
	appr:=lift_pade2(fl,accuracy);
	smallest_pade2(op(appr))
end:

# This procedure lifts the list appr
# Output: new value of appr
# appr, acc, n and degrees is the same as in Harm Derksen's pade2
# functionlist: same as Harm's but multiplied by z
# y is a list of variables [y[1],y[2],..] are the same as in Harm's algorithm
# zvars is a list of variables [z[1],z[2],..] which play the role of Harm's
# variable z.
# old_acc: 0 on the first call, and the previous acc otherwise
# appr: 0 on the first call, and the previous result of lift_pade2 otherwise
lift_pade2:=proc(appr,functionlist,n,degrees,y,zvars,ext,old_acc,extra_acc)
local app,i,k,j,z,acc;
	userinfo(9,diffop,`lifting`,extra_acc,`steps`);
	acc:=old_acc+extra_acc;
	if old_acc=0 then
		# Since I can't change appr, I'll call it app here.
		app:=[seq(expand(x^degrees[i]*y[i]+nt(
		 functionlist[i],acc)),i=1..n)]
	else
		app:=map(g_expand,subs({seq(y[i]=y[i]+x^(-degrees[i])
		 *(nt(functionlist[i],acc)-nt(functionlist[i],old_acc))
		 ,i=1..n)},appr),ext)
	fi;
	app:=modm(app);
	for i from old_acc to acc-1 do for z in zvars do
		k:=0;
		for j to n do if valuation_pade2(app[j],z,x,acc)=i
		 and (k=0 or smaller_pade2(app[j],app[k],x,y)) then
			k:=j
		fi od;
		if k>0 then
			app:=[seq(wipe_pade2(app[j],app[k],z,x,i,ext),j=1..k-1),
			 expand(x*app[k]),seq(wipe_pade2(app[j],app[k],
			 z,x,i,ext),j=k+1..n)]
		fi
	od od;
	userinfo(9,diffop,`done lifting`);
	[app,functionlist,n,degrees,y,zvars,ext,acc]
end:

smallest_pade2:=proc(appr,dummy,n,degrees,y,zvars,ext,dummy2)
local smallest,i;
	smallest:=1;
	appr;
	for i from 2 to n do if smaller_pade2(appr[i],appr[smallest],x,y) then
		smallest:=i;
	fi od;
	[seq(g_expand(coeff(appr[smallest],y[i],1)/x^degrees[i],ext),i=1..n)]
end:

valuation_pade2:=proc(f,z,y,acc)
	if coeff(f,z,1)=0 then acc else ldegree(coeff(f,z,1),y) fi
end:

wipe_pade2:=proc(f,g,z,y,i,ext)
	local ff,gg;
	ff:=coeff(coeff(f,z,1),y,i);
	gg:=g_normal(ff/coeff(coeff(g,z,1),y,i));
	modm(g_expand(f-gg*g,ext))
end:

smaller_pade2:=proc(f,g,x,vars)
local ff,gg,i,df,dg,n;
n:=nops(vars);
ff:=[seq(degree(coeff(f,vars[i],1),x),i=1..n)];df:=max(op(ff));
gg:=[seq(degree(coeff(g,vars[i],1),x),i=1..n)];dg:=max(op(gg));
if df<dg then RETURN(true)
elif df>dg then RETURN(false)
fi;
for i to n do
if ff[i]=df and coeff(f,vars[i],1)<>0 then RETURN(false) fi;
if gg[i]=dg and coeff(g,vars[i],1)<>0 then RETURN(true);fi;
od;
end:

#######################
#  mod p computation  #-----------------------------------------------------
#######################				diffop_modp

macro(	compute_modm=`diffop/compute_modm`,
	compute_modp=`diffop/compute_modp`,
	iratrecon=readlib('iratrecon'),
	modp_table=`diffop/modp_table`,
	modulus2=`diffop/modulus2`,
	reconstruct=`diffop/reconstruct`
):

compute_modp:=proc()
	global  accuracy_laurent, description_laurent, set_laurents, value_laurent, rem_lift, modp_table;
	local i,procedure,ac_old,de_old,va_old;
	if nargs=0 then # no modular computation
		accuracy_laurent:=table(modp_table[0][1]);
		description_laurent:=table(modp_table[0][2]);
		set_laurents:=modp_table[0][3];
		value_laurent:=table(modp_table[0][4]);
		rem_lift:=table(modp_table[0][5]);
		RETURN()
	fi;
	userinfo(6,diffop,`computing modulo`,modulus);
	modp_table[0]:=[op(op(accuracy_laurent)),op(op(
	 description_laurent)),set_laurents,op(op(value_laurent))
	 ,op(op(rem_lift))];
	if modp_table[modulus]<>evaln(modp_table[modulus]) then
		accuracy_laurent:=table(modp_table[modulus][1]);
		description_laurent:=table(modp_table[modulus][2]);
		value_laurent:=table(modp_table[modulus][4]);
		rem_lift:=table(modp_table[modulus][5]);
		ac_old:=table(modp_table[0][1]);
		de_old:=table(modp_table[0][2]);
		va_old:=table(modp_table[0][4]);
		for i in set_laurents minus modp_table[modulus][3] do
#			accuracy_laurent[i]:=subs(modp_table[0][1],i);
#			description_laurent[i]:=subs(modp_table[0][2],i);
#			value_laurent[i]:=subs(modp_table[0][4],i)
# these 3 lines were awfully slow, replacement:
			accuracy_laurent[i]:=ac_old[i];
			description_laurent[i]:=de_old[i];
			value_laurent[i]:=va_old[i]
		od;
		for i in modp_table[0][5] do if rem_lift[op(i)[1]]
		 =evaln(rem_lift[op(i)[1]]) then
			rem_lift[op(i)[1]]:=op(i)[2]
		fi od
	else
		compute_modp()
	fi;
	procedure:=args[1];
	i:=traperror(procedure(args[2..nargs]));
	if i=lasterror then
		userinfo(2,diffop,modulus,`gives error`,i);
		compute_modp();
		RETURN(`new prime needed`)
	else
		modp_table[modulus]:=[op(op(accuracy_laurent)),op(op(
		 description_laurent)),set_laurents,op(op(value_laurent))
		 ,op(op(rem_lift))]
	fi;
	compute_modp();
	i
end:


# Compute modulo prime power.
compute_modm:=proc(exponent)
	global  modulus, modulus2;
	local v;
	modulus:=modulus2^exponent;
	v:=compute_modp(args[2..nargs]);
	while v=`new prime needed` do
		modulus2:=nextprime(modulus2*5);
		modulus:=modulus2^exponent;
		v:=compute_modp(args[2..nargs])
	od;
	modulus:=0;
	v
end:

# Input: v is a list of polynomials, with coefficients in Z/(modl Z).
# Output: a corresponding list with rational number coefficients.
reconstruct:=proc(v,modl,N)
	local w,i,t,r;
if nargs=2 then
	reconstruct(args,floor(evalf(sqrt(modl/5))))
	# 5 instead of 2, because that way we can be almost sure that if
	# an answer is returned, it will be the right answer. That way we
	# don't compute a rightdivision in vain.
elif type(v,list) then
	w:=NULL;
	for i in v do
		w:=w,reconstruct(i,modl,N);
		if w[nops([w])]=failed then RETURN(failed) fi
	od;
	[w]
elif indets(v)<>{} then
	t:=[op(indets(v))];
	t:=t[1];
	w:=0;
	for i from ldegree(v,t) to degree(v,t) do
		r:=reconstruct(coeff(v,t,i),modl,N);
		if r=failed then RETURN(r) fi;
		w:=w+r*t^i
	od;
	w
elif iratrecon(v,modl,N,N,'r','t') then
	eval(r/t)
else
	failed
fi
end:

####################################
#      Global factorizations       #------------------------------------------
####################################			diffop_global

macro(	compute_bound=`diffop/compute_bound`,
	factor_global=`diffop/factor_global`,
	factor_minmult1=`diffop/factor_minmult1`,
	factor_order1=`diffop/factor_order1`,
	flist=`diffop/flist`,
	leftdivision=`diffop/leftdivision`,
	same_charclass=`diffop/same_charclass`,
	try_factorization=`diffop/try_factorization`,
	try_factorization2=`diffop/try_factorization2`,
	xDFn_modr=`diffop/xDFn_modr`
):

DFactor:=proc(ff)
	global  g_conversion1, g_conversion2, accuracy_laurent, description_laurent, set_laurents, value_laurent, rem_lift, modp_table, modulus, modulus2;
	local i,b,f,ext,eb,opt;
	f:=convert(ff,RootOf);
	if f<>ff then RETURN(DFactor(f,args[2..nargs])) fi;
if member(xDF,indets(ff)) then
	# Local operator as input: hence local factorization.
	RETURN(factor_op(ff,`split over k((x))`,g_ext([args])))
elif member(`clear all`,[args]) then
	# correct global variables messed up by a CTRL-C
	g_conversion1:={};
	g_conversion2:={};
	accuracy_laurent:=evaln(accuracy_laurent);
	description_laurent:=evaln(description_laurent);
	set_laurents:={};
	value_laurent:=evaln(value_laurent);
	rem_lift:=evaln(rem_lift);
	modp_table:=evaln(modp_table);
	for i in {g_gcdex,degree_ext,g_evala_rem,g_ext,eval_laurent,adjoint,
	  new_laurent2,valuation_laurent,series_val,mult,rightdivision,
	  factor_op,Newtonpolygon,factor_newton,skipped_factors,factor_newton2,
	  factor_riccati,ramification_of,indeterminates_op,xDFn_modr,
	  ramification_laur,integrate_logs,sol_1order_eq,solve_mateqn,
	  left_solutions,factor_newton_LCLM} do
		assign(i=subsop(4=NULL,op(i)))
	od
fi;
if degree(ff,DF)<=1 then
	RETURN([subs(g_conversion2,ff)])
fi;
	opt:=op({args[2..nargs]} minus {`clear all`});
	modulus:=0;
	modulus2:=503;
	ext:=g_ext([args]);
	f:=collect(subs(g_conversion1,ff),DF,g_normal);
	eb:=NULL;
	for i in [args[2..nargs]] do if type(i,`=`) and op(i)[1]=bound then
		eb:=op(i)[2]
	fi od;
	if member(`alg factor`,[args]) then
		if eb=NULL then ERROR(`must specify a bound`) fi;
		RETURN(subs(g_conversion2,factor_global(f,ext,eb,`alg factor`)))
	fi;
	b:=factor_global(f,ext,eb);
	b:=subs(g_conversion2,b);
if nops(b)=1 or member(`one step`,{args}) then
	b
elif member(`right factor`,{args}) then
	DFactor(b[2],opt)
elif member(LCM,{args}) and degree_ext(b[2],ext)>1 then
	b:=subs(g_conversion1,op(DFactor(b[2],opt,`right factor`)));
	b:=subs(g_conversion2,[op(DFactor(rightdivision(f,`diffop/LCM`(b,ext)
	 ,ext)[1],opt)),[`LCM of`,b,`over Q(`,op(ext),`x)`]]);
	if b[1]=1 then b:=b[2..nops(b)] fi;
	b
else
	[op(DFactor(b[1],opt)),op(DFactor(b[2],opt))]
fi
end:

# Input a global factor r
# Output: the LCM of r and its conjugates over Q(ext)
`diffop/LCM`:=proc(r,ext)
	local ansatz,a,i,sol,ind;
	ansatz:=convert([seq(a[i]*DF^i,i=0..degree(r,DF)
	 *degree_ext(r,ext))],`+`);
	ind:=indets(ansatz) minus {DF};
	sol:=solve(subs(g_conversion2,{seq(coeffs(i,DF),i=[ext_to_coeffs(
	 g_expand(numer(rightdivision(ansatz,r,ext)[2]),ext),ext)])}),ind);
	i:=degree(ansatz,DF);
	while nops(indets(subs(sol,ansatz)) minus {DF,x}) >1 do
		sol:=solve({a[i]=0,op(sol)},ind);
		i:=i-1
	od;
	sol:=solve({a[i]=1,op(sol)},ind);
	subs(g_conversion1,subs(sol,ansatz))
end:

# Computes xDF^n mod r.
xDFn_modr:=proc(n,r,ext)
	local a;
	options remember;
	if n<degree(r,xDF) then RETURN(xDF^n) fi;
	a:=xDFn_modr(n-1,r,ext);
	eval_laurent(expand(x*differentiate(a)
	 +a*xDF-coeff(a,xDF,degree(r,xDF)-1)*r),ext,xDF)
end:

# Gives the function list for pade2
# r is a local factor
flist:=proc(r,order,ext)
	local res,i,p,j;
	res:=NULL;
	for i from 0 to order do
		p:=xDFn_modr(i,r,ext);
		res:=res,[seq(coeff(p,xDF,j),j=0..degree(r,xDF)-1)]
	od;
	[res]
end:

# Input: local factor,maximal order to look for,Newton polygon bound,
#  point,global operator,extra singularities bound,algebraic extension
# Output: a string failed, or a global factorization.
# Sep 1994: Added the option that f=[operator,adjoint]. This means that
# r is a right factor of the adjoint of the operator.
try_factorization:=proc(r,max_order,bound,point,f,eb,ext,min_order)
	local i,flm,sr,j,re,fl,exponent,nstep;
	nstep:=4;
	userinfo(3,diffop,`trying local factor`,r,`in the point`,point,
	 `for global factors of order:`,seq(i,i=degree(r,xDF)..max_order));
for i from max(degree(r,xDF),args[8..nargs]) to max_order do
fl:=pade2(flist(r,i,ext),infinity,ext);
if compute_modm(1,try_factorization2,fl,bound,eb,nstep)<>failed then
	userinfo(3,diffop,`possible factor found modulo prime`,modulus2
	 ,`of order`,i);
	# Computing with higher accuracy:
	exponent:=4;
	flm:=compute_modm(exponent,try_factorization2,fl,bound,eb,nstep);
	while flm<>failed do
		sr:=expand(convert([seq(
		 flm[1][j+1]*xDF^j,j=0..nops(flm[1])-1)],`+`));
		sr:=expand(reconstruct(sr/lcoeff(sr,indets([x,sr])) mod
		 modulus2^exponent,modulus2^exponent));
		if sr<>failed then
			if type(f,list) and f[2]=adjoint then
			# Note: adjoint and substitute(x*DF, ) do not commute
				sr:=adjoint(sr)
			fi;
			sr:=substitute(x*DF,sr,ext);
			if point=infinity then
				sr:=substitute(-x^2*DF,subs(x=1/x,sr))
			else
				sr:=subs(x=x-point,sr)
			fi;
			if not(type(f,list)) then
				sr:=collect(sr/lcoeff(sr,DF),DF,g_normal);
				re:=rightdivision(f,sr,ext);
				if g_normal(re[2])=0 then
					userinfo(3,diffop,`found factor`,sr);
					RETURN([collect(re[1],DF,g_normal),sr])
				fi;
				userinfo(3,diffop,`unlikely failed attempt`)
			else
				# Because of f was made monic after localizing,
				# the adjoint of this local operator is not
				# the adjoint of f. To fix this sr must be
				# multiplied with the leading coefficient of
				# the localized f.
				if point=infinity then
					re:=x^(-degree(f[1],DF))
				else
					re:=(x-point)^(-degree(f[1],DF))
				fi;
				sr:=collect(re*lcoeff(f[1],DF)*sr,DF);
				sr:=mult(sr,1/lcoeff(sr,DF),ext);
				re:=leftdivision(f[1],sr,ext);
				if g_normal(re[2])=0 then
					userinfo(3,diffop,`found factor`,sr);
					RETURN([sr,collect(re[1],DF,g_normal)])
				fi;
				userinfo(3,diffop,`unlikely failed attempt`)
			fi
		fi;
		userinfo(4,diffop,`higher prime power needed`);
		exponent:=ceil(exponent*1.7);
		flm:=compute_modm(exponent,try_factorization2,fl,bound,eb
		,exponent)
	od
fi od;
	failed
end:

try_factorization2:=proc(fl,bound,eb,nstep)
	local i,v,sm,sm_old;
	v:=lift_pade2(op(fl),nstep);
	sm:=smallest_pade2(op(v));
	while sm<>sm_old or sm[nops(sm)]=0 do
		sm_old:=sm;
		v:=lift_pade2(op(v),nstep);
		sm:=smallest_pade2(op(v));
		for i in sm do if degree(i,x)>bound[nops(sm)-1]+eb then
			RETURN(failed)
		fi od
	od;
	[sm,v]
end:

# Input: a global operator f, the coefficients field and a bound for the
# number of extra singularities.
# Output: either [f] or f factored in 2 factors.
# Significant change (sep 1994)
# Bugfix (nov 1994)
factor_global:=proc(f,ext,extra_bound,what)
	local t,v,i,j,k,l,bound,eb,singularities,min_deg,done_s,all_one,R_left;
	if degree(f,DF)<=1 then RETURN([f]) fi;
	# point, localized f, algebraic extensions:
        singularities:={[infinity,convert(f,diffop,x=infinity),ext],seq([i[1],
          convert(f,diffop,x=i[1]),[op(i[3]),op(ext)]],i=v_ext_m(g_factor(
          denom(g_normal(f/lcoeff(f,DF))),ext),x))};
	userinfo(5,diffop,`done computing localizations`);
	for i from 1 to degree(f,DF)-1 do bound[i]:=0 od;
	eb:=0;
	for k in singularities do
		j:=Newtonpolygon(k[2]);
		v:=[seq(seq(j[i][2]+(l-j[i][1])*(j[i+1][2]-j[i][2])
		/(j[i+1][1]-j[i][1]),l=j[i][1]..j[i+1][1]-1),i=1..nops(j)-1)];
		for i from 1 to degree(f,DF)-1 do
			bound[i]:=bound[i]+degree_ext(k[3],ext)*v[i+1]
		od;
		eb:=eb+degree_ext(k[3],ext)
	od;
	# list of bounds for the coefficients of possible global factors
	bound:=[seq(-bound[degree(f,DF)-j]+j*eb,j=1..degree(f,DF)-1)];
	if nargs<3 then
		eb:=compute_bound(singularities,ext);
		if eb>20 then
			userinfo(1,diffop,
			`Computed bound number of extra singularities is`,
			eb,`Use option bound=computed to use this bound.`
			,`Now using a lower bound`);
		 eb:=15
		fi
	elif extra_bound=computed then
		eb:=compute_bound(singularities,ext)
	elif type(extra_bound,integer) then
		eb:=extra_bound
	else
		ERROR(`wrong arguments`)
	fi;
	userinfo(5,diffop,`bounds are`,bound,eb);
	all_one:=true;
	done_s:=[]; # The singularities we have considered
while singularities<>{} do
	min_deg:=min(seq(degree_ext(i[3],ext),i=singularities));
	for i in singularities do if degree_ext(i[3],ext)=min_deg then
		singularities:=singularities minus {i};
		v:=factor_op(i[2],`all right factors`,i[3]);
		done_s:=[op(done_s),[op(i),v]];
		for j in v do if skipped_factors(j)=0 then
			userinfo(3,diffop,cat(`Minimum multiplicity 1 in x=`,i[1]));
			RETURN(factor_minmult1(bound,i[1],f,i[2],eb,j,i[3]))
		fi od;
		all_one:=evalb(all_one and nops(v)=1 and degree(v[1],xDF)=1)
	fi od;
od;
	userinfo(3,diffop,`Minimum multiplicity >1`);
if all_one then # f has only 1 exponential part
	userinfo(3,diffop,`Looking for first order factors`);
	v:=factor_order1(f,done_s,ext);
	if v=[f] and degree(f,DF)>3 then
		userinfo(1,diffop,`factorization of`,f,`may be incomplete`)
	fi;
	RETURN(v)
fi;
	if degree(f,DF)<=3 then bug() fi;
	all_one:=true;
	for i in done_s do
		all_one:=evalb(all_one and nops(i[4])=1);
		for j in i[4] do
			t:=try_factorization(j,degree(f,DF)-1,bound,
			 i[1],f,eb,i[3]);
			if t<>failed then RETURN(t) fi
		od;
		for j in factor_op(adjoint(i[2],i[3]),`all right factors`,i[3])
		do
			t:=try_factorization(j,degree(f,DF)-1,bound,
			 i[1],[f,adjoint],eb,i[3]);
			if t<>failed then RETURN(t) fi
		od
	od;
if all_one then # only 1 char class in every singularity
	userinfo(3,diffop,`Trying with algebraic extensions`);
	for i in done_s do if degree(i[4][1],xDF)>1 then
		j:=factor_op(i[4][1],`alg factor`,i[3])[1];
		if degree(j[3],x)=1 then
			k:=j[1]
		elif degree_ext(j,i[3])=1 then
			next
		else
			k:=make_rightfactor(i[4][1],j,g_ext(j));
			# the slope:
			l:=-valuation_laurent(coeff(i[4][1],xDF,0),0)
			 /degree(i[4][1],xDF);
			k:=faster_riccati_split(i[4][1],rightdivision(i[4][1]
			,k,g_ext(j),l),k,g_ext(j),l)[1]
		fi;
		t:=try_factorization(k,degree(f,DF)-1,bound,i[1],f,0,g_ext(j));
		if t<>failed then RETURN(t) fi;
		v:=factor_op(adjoint(i[2],g_ext(j)),`all right factors`,ext);
		R_left:=0;
		for l in v while R_left=0 do if same_charclass(l
		,adjoint(k,g_ext(j)),g_ext(j))
			then R_left:=l
		fi od;
		if R_left=0 then bug() fi;
		# Now try if R gives a left hand global factor
		t:=try_factorization(R_left,degree(f,DF)-1,bound,i[1]
		 ,[f,adjoint],0,g_ext(j));
		if t<>failed then RETURN(t) fi
	fi od;
fi;
	userinfo(1,diffop,`factorization of`,f,`may be incomplete`);
	[f]
end:

# R is a local factor with multiplicity 1.
# The result of the procedure is a factorization. If f is returned unfactored
# it is irreducible (if the specified bound is correct).
factor_minmult1:=proc(bound,point,f,f_local,eb,R,ext)
	local t,w,fl,slope,k,l,R_left,i;
	fl:=floor(degree(f,DF)/2);
	t:=try_factorization(R,fl,bound,point,f,eb,ext);
	if t<>failed then RETURN(t) fi;
	w:=factor_op(adjoint(f_local,ext),`all right factors`,ext);
	R_left:=0;
	for i in w while R_left=0 do if same_charclass(i,adjoint(R,ext),ext)
		then R_left:=i
	fi od;
	if R_left=0 then bug() fi;
	# Now try if R gives a left hand global factor
	t:=try_factorization(R_left,fl,bound,point,[f,adjoint],eb,ext);
	if t<>failed then RETURN(t) fi;
	# Now try more terms:
	t:=try_factorization(R,degree(f,DF)-1,bound,point,f,eb,ext,fl);
	if t<>failed then RETURN(t) fi;
	t:=try_factorization(R_left,degree(f,DF)-1,bound,point,[f,adjoint],eb
	 ,ext,fl);
	if t<>failed then RETURN(t) fi;
	userinfo(5,diffop,`trying if algebraic extensions are needed`);
	if igcd(op(map(degree,w,xDF)))=1 then RETURN([f]) fi;
	slope:=-valuation_laurent(coeff(R,xDF,0),0)/degree(R,xDF);
	k:=op(factor_op(R,`alg factor`,ext));
	if degree(k[3],x)=1 then
		l:=k[1]
	elif degree_ext(k,ext)=1 then
		RETURN([f])
	else
		l:=make_rightfactor(R,k,g_ext(k));
		l:=faster_riccati_split(R,rightdivision(R,l,g_ext(k),slope)
		 ,l,g_ext(k),slope)[1]
	fi;
	t:=try_factorization(l,fl,bound,point,f,0,g_ext(k));
	if t<>failed then RETURN(t) fi;
	[f]
end:

# f and g irreducible.
# Output: true if f and g have the same characteristic classes.
same_charclass:=proc(f,g,ext)
	local gg,ff,r,c,d;
	if degree(f,xDF)<>degree(g,xDF) then RETURN(false) fi;
	if degree(f,xDF)=1 then RETURN(type(nt(f-g,1),integer)) fi;
	r:=map(Newtonpolygon,[f,g],`include the Newton polynomials`);
	if r[1][1][3]=0 then
		c:=r[1][1][4];
		d:=degree(c,x);
		c:=(coeff(c,x,d-1)-coeff(r[2][1][4],x,d-1))/d;
		if not type(c,integer) then RETURN(false) fi;
		r:=[[[op(r[1][1][1..3]),g_expand(subs(x=x-c,r[1][1][4]),ext)]
		 ,op(r[1][2..nops(r[1])])],r[2]]
	fi;
	if r[1]<>r[2] then RETURN(false) fi;
	gg:=factor_op(g,`alg factor`,ext)[1];
	r:=nt(gg[1],1);
	ff:=substitute(2*xDF-r,ramification_of(f,gg[3],gg[2]),ldegree(r,x),0,gg[2]);
	ff:=Newtonpolygon(ff,`include the Newton polynomials`);
	if ff[1][3]<>0 then RETURN(false) fi;
	ff:=factors(collect(ff[1][4]/lcoeff(ff[1][4],x),x,g_normal));
	for r in ff[2] do if degree(r[1],x)=1 and
	 type(coeff(r[1],x,0)/coeff(r[1],x,1),integer) then RETURN(true)
	fi od;
	false
end:

# Used by try_factorization
leftdivision:=proc(f,L,ext)
	local a,b;
	userinfo(5,diffop,`left division`,f,L);
	a:=0;
	b:=f;
	while degree(b,DF)>=degree(L,DF) do
		a:=a+lcoeff(b,DF)*DF^(degree(b,DF)-degree(L,DF));
		b:=collect(b-mult(L,lcoeff(b,DF),ext)*DF^
		 (degree(b,DF)-degree(L,DF)),DF,g_normal)
	od;
	userinfo(5,diffop,`done left division`);
	[a,b]
end:

# Search 1st left or right factors of an operator which has only 1
# characteristic classes of degree 1 in all singularities.
factor_order1:=proc(ff,singularities,ext)
	local f,s,a,i,zero,t,var,bound,ansatz;
	s:=0;
for i in singularities do
	if i[1]<>infinity then
		t:=subs(x=x-i[1],nt(factor_op(i[2],`split over k((x))`
		 ,i[3])[1]-xDF,1)/x)
	else
		t:=-subs(x=1/x,x*nt(factor_op(i[2],`split over k((x))`,i[3])[1],0))
	fi;
	if i[3]=ext then
		s:=s+t
	else
		s:=s+subs(g_conversion1,evala(Trace(op(subs(g_conversion2
		,[t,i[3],ext])))))
	fi
od;
	f:=substitute(DF-g_normal(s),ff,ext);
	bound:=nt(factor_op(convert(f,diffop,x=infinity),`split over k((x))`
	,ext)[1]-xDF,1);
	if not type(bound,integer) and bound>=0 then bug() fi;
	ansatz:=convert([seq(a[i]*x^i,i=0..bound)],`+`);
	zero:=ansatz*coeff(f,DF,0);
	t:=ansatz;
	for i from 1 to degree(f,DF) do
		t:=diff(t,x);
		zero:=zero+coeff(f,DF,i)*t
	od;
	var:={seq(a[i],i=0..bound)};
	t:=solve({coeffs(collect(numer(subs(g_conversion2,zero)),x),x)});
	if subs(t,ansatz)=0 then
		if nargs=4 or degree(ff,DF)=2 then RETURN([ff])
	else
		s:=factor_order1(adjoint(f),singularities,ext,ext);
		if nops(s)=1 then RETURN([ff]) fi;
		RETURN(map(adjoint,[s[2],s[1]]))
		fi
	fi;
	i:=bound;
	while nops(indets(subs(t,ansatz)) intersect var)>1 do
		t:=solve({op(t),a[i]=0},var);
		i:=i-1
	od;
	ansatz:=subs(g_conversion1,subs(t,ansatz));
	ansatz:=subs(op(indets(ansatz) intersect var)=1,ansatz);
	ansatz:=DF+g_normal(s-diff(ansatz,x)/ansatz);
	f:=rightdivision(ff,ansatz,ext);
	if g_normal(f[2])<>0 then bug() fi;
	[f[1],ansatz]
end:

# Bound for the number of extra singularities in a 1st order left or right factor
compute_bound:=proc(singularities,ext)
	local v,i,res,j,r,k,Z,ma,mi,c;
	userinfo(5,diffop,`computing number of extra singularities bound`);
	res:=0;
for i in singularities do
	v:=map(op,map(factor_op,factor_op(i[2],`all right factors`,i[3])
	,`alg factor`,i[3]));
	ma:=-infinity;
	mi:=infinity;
	for j in v do
		if type(j,list) then
			if j[3]<>x then next fi;
			k:=j[1]
		else
			k:=j
		fi;
		c:=nthterm_laurent(coeff(k,xDF,0),0);
		# Now compute the trace of this algebraic number divided by
		# the degree of this number over Q:
		c:=evala(Norm(Z-subs(g_conversion2,c)));
		c:=-coeff(c,Z,degree(c,Z)-1)/lcoeff(c,Z)/degree(c,Z);
		if not type(c,rational) then bug() fi;
		ma:=max(c,ma);
		mi:=min(c,mi);
		r:=charclass_right(k);
		if nops(r)=1 then next fi;
		ma:=max(ma,c+nt(r[1],1)-xDF)
	od;
#	if mi=infinity then RETURN(`no 1st order factor`) fi;
	if mi=infinity then RETURN(0) fi;
	res:=res+degree_ext(i[3],ext)*(ma-mi)/2
od;
	res:=floor(res);
	userinfo(5,diffop,`bound for extra singularities is`,res);
	res
end:

#save `diffop.m`;
#quit
