#
## <SHAREFILE=algebra/IntBasis/IntBasis.mpl >
## <DESCRIBE>
##        SEE ALSO: algebra/IntBasis/IntBasis.tex  (30K)
##
## Function: integral_basis
##        Computes an integral basis for an algebraic number or function.
##
## Function: genus
##        genus(f,x,y) computes the genus of an algebraic curve given
##        by the polynomial f in x and y.
##        This code is using a new method based on Puiseux series
##        expansions, and should be considerably faster than
##        the Maple library routine maxorder for large problems.
##        AUTHOR: Mark van Hoeij, hoeij@sci.kun.nl
##
## Function: puiseux
##        Computes a Puiseux series expansion of an algebraic function.
##        Especially intended for expansions about singular points where
##        the result contains more than one branch.
##        SEE ALSO: genus and integral_basis in ?share,calculus.
##        AUTHOR: Mark van Hoeij, hoeij@sci.kun.nl
## </DESCRIBE>

##########################
# standard input
###
# Convert from 5.2 to Release 3
# Author: Mark van Hoeij, e-mail address:    hoeij@sci.kun.nl
# If you change/improve this program, or if you find a bug, please let me know

# Version: Aug 18 1994
# Date previous version: Jan 11 1994
# Changes:
#   A few bug fixes.
#   I added an algorithm by Harm Derksen and me to make nice tubeplots of
#   singularities.
# Future changes:
#   Faster Puiseux expansion computation.
#   Better find_point algorithm. This should result in a faster parametrization
#   algorithm with a better (smaller degree algebraic numbers) output.
# References:
#   integral basis: "An algorithm for computing an integral basis in an
#     algebraic function field" to appear in JSC.
#   parametrization algorithm: ISSAC 94 Proceedings.

IntBasis := `See ?integral_basis, ?genus, and ?puiseux`:

lprint(`Note: this program requires Maple V Release 2`);
lprint(`For help about Puiseux expansions type:  ?puiseux`);
lprint(`For help about the integral basis type:  ?integral_basis`);
lprint(`For help about genus and parametrization computation type: ?genus`);

macro(  compute_x=`genus/compute_x`,
	cont_exp=`puiseux/cont_exp`,
	cont_exp_m1=`puiseux/cont_exp_m1`,
	degree_ext=`puiseux/degree_ext`,
	double_factors=`integral_basis/double_factors`,
	ext_needed=`integral_basis/ext_needed`,
	ext_to_coeffs=`integral_basis/ext_to_coeffs`,
	find_point=`genus/find_point`,
	function_with_one_pole=`genus/function_with_one_pole`,
	g_conversion1=`puiseux/g_conversion1`,
	g_conversion2=`puiseux/g_conversion2`,
	g_denom=`puiseux/g_denom`,
	g_evala=`puiseux/g_evala`,
	g_evala_rem=`puiseux/g_evala_rem`,
	g_expand=`puiseux/g_expand`,
	g_ext_r=`puiseux/g_ext_r`,
	g_ext=`puiseux/g_ext`,
	g_factor=`puiseux/g_factor`,
	g_normal=`puiseux/g_normal`,
	g_numer=`puiseux/g_numer`,
	g_solve=`puiseux/g_solve`,
	g_zero_of=`puiseux/g_zero_of`,
	homogeneous=`genus/homogeneous`,
	monic=`puiseux/monic`,
	singularities=`genus/singularities`,
	trancendental_ext=`puiseux/trancendental_ext`,
	truncate=`puiseux/truncate`,
	truncate_subs=`puiseux/truncate_subs`,
	v_ext_m=`puiseux/v_ext_m`
):

#############################
#  groundfield computation  #
#############################

# Most procedure are given twice. The first is meant to describe in Maple
# language what the second procedure does. The second procedures uses
# an own representation of algebraic numbers. I only wrote a replacement
# for evala(Expand()). For Normal and Factor I call the Maple procedures.

g_conversion1:={}: # RootOf syntax -> my own syntax
g_conversion2:={}: # my syntax -> RootOf syntax
trancendental_ext:={}: # set of trancendental extensions
# Note: this algorithm is written for constants field which are
# algebraic extensions of Q. For other constants fields it is not
# well tested, and certainly not optimized.

g_solve:=proc()
	if trancendental_ext={} then solve(args)
	else solve(args,indets(args) minus trancendental_ext)
	fi
end:

# Factorization over the groundfield
g_factor:=proc(ff,ext)
local f;
	f:=numer(g_normal(ff));
	if ext=[] then factor(f) else evala(Factor(f,op(ext))) fi
end:

g_factor:=proc(ff,ext)
local f;
	f:=numer(g_normal(ff));
	if ext=[] then factor(f)
	else
		subs(g_conversion1,evala(Factor(subs(g_conversion2,f)
		 ,op(subs(g_conversion2,ext)))))
	fi
end:

# g_ext: gives a list of the algebraic extensions.
g_ext_r:=proc(a)
local v,vv,i,tail;
options remember;
	v:=indets(a,RootOf);
	if nops(v)=0 then RETURN([]) fi;
	vv:={};
	for i in v do vv:=vv union indets(op(i),RootOf) od;
	tail:=g_ext(vv);
	v:=[op(v minus vv)];
	[op(v),op(tail)]
end:

# Gives the algebraic extensions appearing in aa.
g_ext:=proc(a)
global  g_conversion1, g_conversion2;
local v,i,result,ii,vv;
options remember;
	v:=g_ext_r(subs(g_conversion2,a));
	result:=NULL;
	for i from nops(v) to 1 by -1 do if member(subs(g_conversion1,v[i])
	 ,{seq(`puiseux/rootof`.ii,ii=0..nops(g_conversion2))}) then
		result:=subs(g_conversion1,v[i]),result
	else
		if g_conversion1={} then g_conversion1:=NULL fi;
		vv:=nops(g_conversion2);
		g_conversion1:=v[i]=`puiseux/rootof`.vv,g_conversion1;
		g_conversion2:={`puiseux/rootof`.vv=v[i],op(g_conversion2)};
		result:=subs(g_conversion1,v[i]),result
	fi od;
	[result]
end:

# Gives the algebraic degree of a over b
degree_ext:=proc(a,b)
local v,i,all,d,var;
options remember;
	v:=indets(a,RootOf) minus indets(b,RootOf);
	all:=[op(indets([a,b],RootOf))];
	all:={seq(all[i]=var[i],i=1..nops(all))};
	d:=1;
	for i in v do
		d:=d*degree(subs(all,op(i)),_Z)
	od;
	d
end:

degree_ext:=proc(aa,bb)
local a,b,v,i,all,d,var;
options remember;
	a:=subs(g_conversion2,aa);
	b:=subs(g_conversion2,bb);
	v:=indets(a,RootOf) minus indets(b,RootOf);
	all:=[op(indets([a,b],RootOf))];
	all:={seq(all[i]=var[i],i=1..nops(all))};
	d:=1;
	for i in v do
		d:=d*degree(subs(all,op(i)),_Z)
	od;
	d
end:

g_expand:=proc(a,ext) evala(Expand(a)) end:

g_expand:=proc(a,ext)
	local v;
	if trancendental_ext={} then g_evala(expand(a),ext)
	else subs(g_conversion1,evala(Expand(subs(g_conversion2,a))))
	fi
end:

g_evala:=proc(a,ext) evala(Expand(a)) end:

g_evala:=proc(a,ext)
	local dummy,e;
	if nops(ext)=0 then RETURN(a)
	elif nops(ext)=1 then
		e:=ext[1];
		expand(convert([seq(coeff(a,e,dummy)*g_evala_rem(e^dummy)
		 ,dummy=0..degree(a,e))],`+`))
	else
	e:=g_evala(a,ext[2..nops(ext)]);
	g_evala(expand(convert([seq(coeff(e,ext[1],dummy)*g_evala_rem(ext[1]^dummy)
	 ,dummy=0..degree(e,ext[1]))],`+`)),ext[[2..nops(ext)]])
	fi
end:

g_evala_rem:=proc()
	options remember;
	expand(subs(g_conversion1,evala(Expand(subs(g_conversion2,args)))))
end:

g_normal:=proc(a)
	if indets(a,RootOf)={} then
		normal(a,expanded)
	else
		evala(Normal(a));
		evala(Expand(numer(")))/evala(Expand(denom(")))
	fi
end:

# works for both syntaxes
g_normal:=proc(aa)
	local a;
	if indets(aa,RootOf)<>{} then 
		evala(Normal(aa));
		RETURN(evala(Expand(numer(")))/evala(Expand(denom("))))
	fi;
	a:=subs(g_conversion2,aa);
	if indets(a,RootOf)={} then
		normal(a,expanded)
	else
		evala(Normal(a));
		evala(Expand(numer(")))/evala(Expand(denom(")));
		subs(g_conversion1,")
	fi
end:

# Bugfix: numer and denom sometimes factor the numerator and denominator.
# This causes: "ERROR could not compute coefficient" in very rare cases.
# Also expanding won't be useful anymore in g_normal.
# The following two procedures are meant to save some typing:
g_numer:=proc() expand(numer(args)) end:
g_denom:=proc() expand(denom(args)) end:

# works for both syntaxes
g_normal:=proc(aa)
	local a;
	if indets(aa,RootOf)<>{} then 
		RETURN(evala(Normal(aa)))
	fi;
	a:=subs(g_conversion2,aa);
	if indets(a,RootOf)={} then
		normal(a)
	else
		subs(g_conversion1,evala(Normal(a)))
	fi
end:


# Input : an irreducible polynomial kk in x, not necessarily monic
# Output: a zero of k
# If an algebraic extension is needed it will be placed in ext.
g_zero_of:=proc(k,x,ext)
	if degree(k,x)=1 then
		ext:=NULL;
		RETURN(evala(-coeff(k,x,0)/coeff(k,x,1)))
	fi;
	ext:=RootOf(k,x)
end:

g_zero_of:=proc(k,x,ext)
	global  g_conversion1, g_conversion2;
	local a,vv;
	if degree(k,x)=1 then
		ext:=NULL;
		RETURN(g_normal(-coeff(k,x,0)/coeff(k,x,1)))
	fi;
	a:=RootOf(subs(g_conversion2,k),x);
	# Bugfix 7 Aug 1994: one must avoid having multiple
	# `puiseux/rootof`s for the same RootOf
	if not member(_Z,indets(op(subs(g_conversion1,a)))) then
		a:=subs(g_conversion1,a);
		ext:=a;
		RETURN(a)
	fi;
	if g_conversion1={} then g_conversion1:=NULL fi;
	vv:=nops(g_conversion2);
	g_conversion1:=a=`puiseux/rootof`.vv,g_conversion1;
	g_conversion2:={`puiseux/rootof`.vv=a,op(g_conversion2)};
	ext:=`puiseux/rootof`.vv
end:

# Gives the zeros of the factors, their multiplicities and algebraic extensions
v_ext_m:=proc(f,x)
local ext,nulp,i,result;
	if degree(f,x)=0 then RETURN({}) fi;
	if type(f,`^`) then
		nulp:=g_zero_of(op(f)[1],x,'ext');
		ext:=eval(ext);
		RETURN({[nulp,op(f)[2],[ext],degree(op(f)[1],x)]})
	fi;
	if type(f,`*`) then
		result:={};
		for i in {op(f)} do
			result:=result union v_ext_m(i,x)
		od;
		RETURN(result)
	fi;
	nulp:=g_zero_of(f,x,'ext');
	ext:=eval(ext);
	{[nulp,1,[ext],degree(f,x)]}
end:

# ext_to_coeffs does basically: coeffs(expression,RootOf( .. ))
# We need this procedure because coeffs doesn't always work this way.
# a must be evala'd
ext_to_coeffs:=proc(a,ext)
local r,i,j;
	r:=indets(a,RootOf) minus indets(ext,RootOf);
	if r={} then RETURN(a) fi;
	for i in r do if not member(i,indets(r minus {i},RootOf)) then
		r:=[coeffs(g_expand(subs(i=j,a)),j)];
		RETURN(seq(ext_to_coeffs(j,ext),j=r))
	fi od;
	ERROR(`?`)
end:

ext_to_coeffs:=proc(a,ext)
	local dummy,aa;
	aa:=(indets(a) minus indets(ext)) intersect
	{seq(`puiseux/rootof`.dummy,dummy=0..nops(g_conversion2))};
	coeffs(a,[op(aa)])
end:

########################
# Puiseux computation  #
########################

# Takes the lowest coefficients
truncate:=proc(aa,n,x,ext)
local dummy,a;
	# a:=expand(aa);
	a:=collect(aa,x);
	a:=expand(convert([seq(x^dummy*coeff(a,x,dummy),dummy=0..n-1)],`+`));
# replaced expand(g_normal by expand(collect(  :
	if trancendental_ext<>{} then a:=expand(collect(a,x,g_normal)) fi;
	g_evala(a,ext)
end:

truncate_subs:=proc(f,x,y,y_value,n,ext)
	truncate(subs(y=y_value,f),n,x,ext)
end:

truncate_subs:=proc(f,x,y,y_value,n,ext)
local ym,i,result;
	if degree(f,x)>=n and n>=1 then
		RETURN(truncate_subs(truncate(f,n,x,ext),x,y,y_value,n,ext))
	fi;
	result:=0;
	for i from 0 to degree(f,y) do
		if i=0 then ym:=1 else ym:=truncate(ym*y_value,n,x,ext) fi;
		result:=result+coeff(f,y,i)*ym
	od;
	truncate(result,n,x,ext)
end:

# Makes f monic, returns also q, the factor needed to multiply y by, to
# get it integral.
monic:=proc(f,y,ext,q)
	local dummy,ff,lc,qq;
	ff:=g_numer(g_normal(f));
	lc:=lcoeff(ff,y);
	if indets(lc,`name`) minus {op(ext)}={} then
		q:=1;
		RETURN(g_expand(g_normal(ff/lc),ext))
	fi;
	lc:=subs(g_conversion1,evala(Sqrfree(subs(g_conversion2,lc))));
	lc:=lc[2];
	lc:=g_expand(product(lc[dummy][1],dummy=1..nops(lc)),ext);
	ff:=monic(subs(y=y/lc,ff),y,ext,'qq');
	qq:=eval(qq);
	q:=g_expand(qq*lc,ext);
	ff
end:

# See ?puiseux
puiseux:=proc(aa::algfun(rational),eqn::name=algfun(rational),n::numeric)
global  trancendental_ext;
local x,v,a,f,y,ext,q,verz,i,result,ma,dummy,dummy2,qx;
	if not type(aa,RootOf) then ERROR(`First argument must be a RootOf`) fi;
	x:=eval(op(1,eqn));
	v:=op(2,eqn);
	a:=subs(x=x+v,aa);
	ext:=g_ext(op(a));
	f:=subs(g_conversion1,op(a));
	f:=subs(_Z=y,f);
	f:=monic(f,y,ext,'q');
	q:=eval(q);
	qx:=ldegree(q,x);
	q:=g_expand(q/x^qx,ext);
# Bugfix, I forgot the seq(rootof... which made it fail on Monagan's tests
	trancendental_ext:=indets(f) minus
		{x,y,seq(`puiseux/rootof`.i,i=0..nops(g_conversion2))};
	verz:=`puiseux/technical_answer`(f,x,y,n+qx,ext);
	ma:=ceil(max(seq(dummy[2]/dummy[3],dummy=verz)))+qx;
	q:=taylor(1/q,x=0,ma);
	q:=g_expand(sum('x^dummy2*coeff(q,x,dummy2)',dummy2=0..ma-1),ext);
	result:={};
	for i in verz do
# removed the g_normal
		result:=result union {subs(x=x^(1/i[3]),
		 truncate(subs(x=x^i[3],q)*i[1],i[2],x,ext))/x^qx}
	od;
	subs(g_conversion2,subs(x=x-v,result))
end:

# This procedure computes Puiseux expansions.
# It's input must be in internal format.
# Output: a list, see cont_exp
`puiseux/technical_answer`:=proc(f,x,y,n,ext)
local result,i,verz;
options remember;
	verz:=cont_exp([0,0,1,ext,degree(f,y),1,0],f,x,y);
	result:={};
	for i in verz do
		result:=result union cont_exp_m1(i,f,x,y,n)
	od;
result
end:

# This procedure continues an expansion until it has multiplicity 1
# Input is a list v
# v[1] = The expansion so far
# v[2] = This expansion is determined modulo x^(v[2]/v[3])
# v[3] = Least common multiple of the denominators of the powers of x
# v[4] = The algebraic extensions in this expansion
# v[5] = The multiplicity of this factor
# v[6] = The degree of the algebraic extensions above the groundfield
# It will also return (not needed in the input)
# v[7] = sum of the valuations ( v(x^(1/d))=1/d ) of the differences
# of this expansion with the other expansions. This sum is needed for
# computing an integral basis in an algebraic function field.
# f    = minimal polynomial of y over L(x), must be monic
cont_exp:=proc(v,f,x,y)
local t,n,a,i,ii,r,som,result,vv7;
	if v[5]=1 then RETURN({v}) fi;
	result:={};
	r:=v[1]+a*x^v[2];
	# we try to find an equation for the unknown a
	i:=1;
	ii:=0;
	if v[1]=0 then ii:=g_expand(subs({x=x^v[3],y=r},f),v[4])
	else
		while ii=0 do
		   i:=i+v[3]+2;
		   ii:=truncate_subs(subs(x=x^v[3],f),x,y,r,v[2]+
		    v[3]*v[7]+i,v[4])
		od
	fi;
	vv7:=(ldegree(ii,x)-v[2])/v[3];
	r:=tcoeff(ii,x);
	if degree(r,a)>0 then
		r:=g_expand(r*g_normal(1/lcoeff(r,a)),v[4]);
		if v[5]=0 then r:=expand(r/a^ldegree(r,a)) fi;
		if degree(r,a)>1 then
			r:=g_factor(r,v[4])
		fi;
		r:=v_ext_m(r,a);
		for i in r do
			result:=result union cont_exp([v[1]+x^v[2]
			 *i[1],v[2]+1,v[3],[op(i[3]),op(v[4])],i[2],v[6]*i[4]
			 ,vv7],f,x,y)
		od
	fi;
	som:=0;
	for ii in result do
		som:=som+ii[5]*ii[6]
	od;
	n:=2;
	while som<v[5] do
		for t from 1 to n-1 do if igcd(t,n)=1 then
			result:=result union cont_exp([subs(x=x^n
			 ,v[1]),(v[2]-1)*n+t,v[3]*n,v[4],0,v[6],vv7],f,x,y);
			som:=0;
			for i in result do som:=som+i[5]*i[6] od;
		fi od;
		n:=n+1;
	od;
	result
end:

# This procedure continues (= computes more terms of) expansions that have
# multiplicity 1. This could also be done with cont_exp, but this
# procedure is faster.
cont_exp_m1:=proc(v,f,x,y,nnk)
local nk,a,result,machtx,r,rr;
	nk:=ceil(nnk*v[3])/v[3];
	machtx:=v[2];
	if nk<=machtx/v[3] then RETURN({v}) fi;
	result:=v[1]+a*x^v[2];
	# r:=g_expand(subs({x=x^v[3],y=result},f),v[4]);
	# r:=truncate(r/x^ldegree(r,x),v[3]*nk-machtx,x,v[4]);
	r:=truncate_subs(subs(x=x^v[3],f),x,y,result,v[3]*(nk+v[7]),v[4]);
	r:=expand(r/x^ldegree(r,x));
	while nk>machtx/v[3] do
		rr:=coeff(r,x,0);
		if degree(rr,a)<>1 then ERROR(`degree is not 1`) fi;
		rr:=g_normal(-coeff(rr,a,0)/coeff(rr,a,1));
		result:=g_expand(subs(a=rr+x*a,result),v[4]);
		machtx:=machtx+1;
		if nk>machtx/v[3] then
			r:=subs(a=rr+x*a,r);
			r:=g_expand(r,v[4]);
			if ldegree(r,x)<>1 then ERROR(`degree is not 1`) fi;
			r:=truncate(r/x,v[3]*nk-machtx,x,v[4])
		fi
	od;
	{[subs(a=0,result),machtx,v[3],v[4],v[5],v[6],v[7]]}
end:

# There are 2 extra syntaxes now:
# integral_basis(f,x,y)
# and integral_basis(f,x,y,df) where df is a set of irreducible
# polynomials in x. In this case integral_basis will return a local
# integral basis for these factors in df.
integral_basis:=proc()
global  trancendental_ext;
local alfa,f,x,y,extl,disc,df,basis,pl_transl,k,ext,nulp,f_translated,max_v7
 ,places,d,i,b,found_something,dummy,dummy2,q,power_of_k,equations
 ,values_basis_in_places,value_new_one,kk,max_power_k;
options remember;
	alfa:=eval(args[1]);
	if nargs=1 then x:=op(indets(alfa,`name`) minus {_Z}) else x:=args[2] fi;
	if not (type(alfa,algfun(rational)) and type(x,`name`)) then 
		ERROR(`Wrong arguments`)
	fi;
	if nargs>=3 then
		f:=args[1];
		x:=args[2];
		y:=args[3];
		alfa:=y;
		extl:=g_ext(f);
		f:=subs(g_conversion1,f)
	else
		extl:=g_ext(op(alfa));
		f:=subs(g_conversion1,op(alfa));
		for i from 1 to nops(extl) do
			f:=subs(extl[i]=b[i],f)
		od;
		f:=subs(_Z=y,f);
		for i from 1 to nops(extl) do
			f:=subs(b[i]=extl[i],f)
		od
	fi;
	f:=monic(f,y,extl,'q');
	trancendental_ext:=indets(f) minus
		{x,y,seq(`puiseux/rootof`.i,i=0..nops(g_conversion2))};
	q:=eval(q);
	 # f is the minimal polynomial of y over L(x), we made it monic
	if printlevel>1 then lprint(`Computing the discriminant ...`) fi;
if nargs<=3 then
	disc:=g_expand(discrim(f,y),extl);
	if printlevel>1 then
		lprint(`disc=`,disc);
		lprint();
		lprint(`Computing multiple factors of the discriminant  ...`)
	fi;
	df:=double_factors(disc,x,extl);
	if printlevel>1 then
		lprint(`Factors with multiplicity > 1 are:`,df);
		lprint()
	fi;
	 # df contains those factors k that appear more than once in the
	 # discriminant discrim(f,y)
else
	df:=args[4]
fi;
for k in df do
	power_of_k[k]:=1;
	# Later we will search for integrals of the shape (..)/k^power_of_k[k]
	nulp:=g_zero_of(k,x,'ext');
	ext:=eval(ext);
	ext:=[ext,op(extl)];
	 # the zero of k is now in nulp, if an algebraic extension is needed
	 # it is placed in ext
	f_translated:=g_expand(subs(x=x+nulp,f),ext);
	if printlevel > 1 then
		lprint(`Computing the Puiseux expansions for the factor`,k)
	fi;
	pl_transl:=[op(`puiseux/technical_answer`(f_translated,x,y,0,ext))];
	 # Now pl_transl contains the puiseux expansions, but
	 # translated x=x+nulp
	max_v7:=max(seq(dummy[7],dummy=pl_transl));
	 # this number is the maximum internal intersection multiplicity
	 # (however, the way we count it x^(1/2) and -(x^(1/2)) have
	 # intersection multiplicity 1/2), see also in the file puiseux the
	 # procedure cont_exp
	max_power_k[k]:=floor(max_v7);
	if printlevel >1 then
		lprint(`Maximum number of factors`,k,
		 `in the denominator is`,max_power_k[k]);
		 lprint()
	fi;
	pl_transl:=[seq(op(cont_exp_m1(dummy,f_translated,x,y,min(max_v7
	 -dummy[7]+dummy[2]/dummy[3],max_power_k[k]))),dummy=pl_transl)];
	   # Here I computed more terms of the puiseux expansions
	places[k]:=[seq([dummy[1],nulp,dummy[3],
	  # The following ext_needed(..) checks if we really need a certain
	  # algebraic extension. If we don't we get rid of it.
	  # In the previous versions of IntBasis we had the following line
	  # which was a bug.
	  # ext_needed(dummy[7],max_power_k[k],dummy[4],dummy[1],x)
	 ext_needed(dummy,max_v7,x)
	],dummy=pl_transl)];
	  # places[k][dummy] contains: (dummy=1..number of puiseux expansions)
	  # 1) The puiseux expansion (but translated x=x+nulp)
	  # 2) Contains the zero of k   (= this number nulp)
	  # 3) The least common multiple of the denominators of the powers
	  # of x, so x here must be interpreted as x^(1/d) if this lcm is d.
	  # If we take into account the translation it would be x^(1/d)-nulp.
	  # So in order to interpret a normal x we substitute
	  # subs(x=x^d+nulp,..)
	  # 4) The algebraic extension belonging to this puiseux expansion
	values_basis_in_places[k,1]:=[seq(1,dummy=pl_transl)]
	   # Because the first basis-element will be 1.
od;
i:=df;
for k in i do if max_power_k[k]=0 then
	df:=df minus {k}
fi od;
basis:=[1];
for d from 2 to degree(f,y) do
	if printlevel>1 then
		lprint(`Integral elements of degree less than`,d-1,`are:`);
		lprint(`the previous elements and`,basis[d-1]);
		lprint()
	fi;
	basis:=[op(basis),y*basis[d-1]];
	  # This basis is our first guess
	for k in df do
		  # Now we compute the values of this new basis-element in all
		  # places (that is, we substitute y=puiseux expansions)
		values_basis_in_places[k,d]:=[seq(truncate
		 (values_basis_in_places[k,d-1][dummy]*places[k][dummy][1]
		 ,places[k][dummy][3]*max_power_k[k],x
		 ,places[k][dummy][4]),dummy=1..nops(places[k]))]
	od;
	for k in df do
	  # using the values of this new basis-element in the places,
	  # we will see if we can find an integral, with a bigger denominator
	found_something:=true;
	while found_something and power_of_k[k]<=max_power_k[k] do
		for i from 1 to d-1 do b[i]:=evaln(b[i]) od;
		b[i]:=1;
		 # Now we compute the values of basis[1]*b[1]+..+basis[d]*b[d]
		 # in the places
		 # and we try to put an extra factor k in the denominator
		value_new_one:=[seq(truncate(sum('b[dummy2]
		 *values_basis_in_places[k,dummy2][dummy]',dummy2=1..d)
		 ,places[k][dummy][3]*power_of_k[k],x
		 ,places[k][dummy][4]),dummy=1..nops(places[k]))];
		 # All coefficients of powers of x less than power_of_k must
		 # be zero, in order for this new_one to be dividable by
		 # k^power_of_k[k]. So we find equations by taking the
		 # remainder
		equations:={seq(ext_to_coeffs(dummy,[places[k][1][2],extl]
		 ),dummy=value_new_one)};
		equations:={seq(coeffs(dummy,x),dummy=equations)};
		equations:=subs(g_conversion1,evala({g_solve(
		subs(g_conversion2,equations))}));
		 # Now we know what values b[1] .. b[d] must have
		if equations={} then
			found_something:=false
		else
			if degree_ext(places[k][1][2],extl)>1 then
				equations:=subs(places[k][1][2]=x,equations)
			fi;
			assign(op(equations));
			  # Now basis[1]*b[1]+..basis[d]*b[d] is dividable by k
			  # In the following for_do_od we compute the values of
			  # basis[1]*b[1]+..+basis[d]*b[d] in all places.
			for kk in df do
			 values_basis_in_places[kk,d]:=[seq(
			   truncate(sum('subs(x=x^places[kk][dummy
			   ][3]+places[kk][dummy][2],eval(b[dummy2]))*
			   values_basis_in_places[kk,dummy2][dummy]',dummy2=
			   1..d),places[kk][dummy][3]
			   *max_power_k[kk],x,places[kk][dummy][4])
			   ,dummy=1..nops(places[kk]))]
			od;
			 # Now we will put (basis[1]*b[1]+..+basis[d]*b[d])/k
			 # in the basis
			basis:=[seq(basis[dummy],dummy=1..d-1),normal(sum(
			'eval(b[dummy2])*basis[dummy2]',dummy2=1..d)/k)];
			  # Now we should divide values_basis_in_places[.. ,d]
			  # by k, but instead we multiply all other
			  # values_basis_in_places[.., <>d] by k, this will
			  # be done in the following nested for_do_od
			for i from 1 to d-1 do for kk in df do
			 values_basis_in_places[kk,i]:=[seq(
			   truncate(values_basis_in_places[kk,i][
			   dummy]*subs(x=x^places[kk][dummy][3]+places[kk][
			   dummy][2],k),places[kk][dummy][3]*max_power_k[kk]
			   ,x,places[kk][dummy][4]),dummy=1..nops(places[kk]))]
			od od;
			  # Now we increase power_of_k, so we will try to put
			  # more factors k in the denominator
			power_of_k[k]:=power_of_k[k]+1
		fi
	od od od;
	subs(g_conversion2,subs(y=alfa*q,basis))
end:

# This procedure finds out if we can miss a certain alg. extension.
# It's not major importance.
ext_needed:=proc(a,b,c,d,x)
local dummy;
	if a<b or nops(c)=0 then RETURN(c) fi;
	if lcoeff(d,x)<>c[1] then RETURN(c) fi;
	if member(c[1],indets(eval(d-c[1]*x^degree(d,x)))) then RETURN(c) fi;
	[seq(c[dummy],dummy=2..nops(c))]
end:

# The previous ext_needed contains a bug, here is the fix
ext_needed:=proc(v,max_v7,x)
local dummy;
RETURN(v[4]);
	if v[7]<max_v7 or nops(v[4])=0 or
	coeff(v[1],x,v[2]-1)<>v[4][1] or
	member(v[4][1],indets(eval(v[1]-v[4][1]*x^degree(v[1],x))))
		then RETURN(v[4])
	fi;
	[seq(v[4][dummy],dummy=2..nops(v[4]))]
end:

# This procedure gives those factors which appear more than once
double_factors:=proc(ff,x,ext)
local f,i,ii,i2,result,v,vv,frem,j;
	f:=ff;
	if nops(ext)>0 then
		f:=subs(g_conversion1,evala(Norm(subs(g_conversion2,f))))
	fi;
	f:=sqrfree(numer(normal(f)))[2];
	v:={};
	for i in f do if i[2]>1
# Bugfix:
	and degree(i[1],x)>=1 then
		vv:=readlib(factors)(i[1]);
		v:={op(v),seq(ii[1],ii=vv[2])}
	fi od;
	result:={};
	for i in v do
		if ext=[] then
			result:=result union {g_expand(i/lcoeff(i,x),ext)}
		else
			i2:=g_expand(i*i,ext);
			i2:=g_expand(i2*g_normal(1/lcoeff(i2,x)),ext);
			frem:=ff;
			while degree(frem,x)>=degree(i2,x) do
				frem:=g_expand(frem-lcoeff(frem,x)*i2*x
				 ^(degree(frem,x)-degree(i2,x)),ext)
			od;
			if frem=0 then frem:=i2 fi;
			frem:=evala(Sqrfree(subs(g_conversion2,frem)))[2];
			f:=1;
			for j in frem do if j[2]>1 then
			    f:=f*evala(Factor(j[1],subs(g_conversion2,ext)))
			fi od;
			if type(f,`*`) then vv:={op(f)} else vv:={f} fi;
			for j in vv do if evala(Rem(subs(g_conversion2,i)
			 ,j,x))=0 then
				result:={op(result),j}
			fi od
		fi
	od;
	v:={};
	for i in result do
		if degree(i,x)>0 then v:=v union {i} fi
	od;
	v
end:

##########################################
# genus and parametrizations computation #
##########################################

# f is a polynomial in x and y
# f must be irreducible over Qbar
# If f is reducible this procedure does not work properly.
genus:=proc()
	global  trancendental_ext;
	local f,x,y,d,t,p,v,result,i,point;
	options remember;
	if args[nargs]=`irr check` then
		f:=evala(AFactor(args[1]));
		if not type(f,`+`) then
			i:=2;
			if type(f,`*`) then
				i:=0;
				for t in [op(f)] do if indets(t) intersect
				 {args[2..3]} <> {} then i:=i+1
				fi od;
			fi;
			if i>1 then RETURN(args[1],`factored as`,f) fi;
		fi;
		RETURN(genus(args[1..nargs-1]))
	fi;
	if nargs<3 or (not type(args[2],name)) or (not type(args[3],name))
	or (not type(f,polynom(anything,[args[2..3]])))
	or (nargs>=4 and not member(args[4],{`parameter`,`parametrization`}))
	or (nargs>=4 and args[4]=`parametrization` and (nargs=4 or not
	 type(args[5],name)))
	or indets(args[1]) intersect {args[2..3]}={}
	then
		ERROR(`wrong number or type of arguments`)
	fi;
	f:=expand(args[1]);
	x:=args[2];
	y:=args[3];
	trancendental_ext:=indets(f) minus
		{x,y,seq(`puiseux/rootof`.i,i=0..nops(g_conversion2))};
	if nargs>3 and args[4]=`parametrization` then
		t:=args[5];
		p:=genus(args[1..3],`parameter`,args[6..nargs]);
		if p=`genus is not zero` then RETURN(p) fi;
		RETURN([compute_x(p,f,x,y,t),compute_x(p,f,y,x,t)])
	fi;
	d:=degree(f,{x,y});
	if coeff(f,y,d)=0 then
		if nargs=5 then
			v:=args[5];
			if nops(v)=3 then
				# v is a point
				v:=[v[1]-v[2],op(v[2..3])]
			else
				# v[1] is a function with 1 pole in v[2],
				# where v[2]=`finite` or `infinity`
				v:=[evala(Expand(subs(x=x+y,v[1]))),v[2]]
			fi
		else
			# No point, or function was specified yet.
			v:=NULL
		fi;
		RETURN(subs(x=x-y,
		 genus(evala(Expand(subs(x=x+y,f))),args[2..min(4,nargs)],v)))
	fi;
	# Now coeff(f,y,d)<>0, so [0,1,0] is not a point on the curve anymore
	result:=(d-1)*(d-2)/2;
	v:=singularities(f,x,y);
	for i in v do
		result:=result-i[3]*degree_ext(i,f)
	od;
	if nargs=3 then RETURN(result) fi;
	if result<>0 then RETURN(`genus is not zero`) fi;
	if nargs>4 then
		point:=args[5]
	else
		point:=find_point(f,x,y)
	fi;
	if nops(point)=3 then
		# Now we compute a function that has a pole in point. If the
		# point is finite, this function has no other poles in finite
		# otherwise it has no other poles in infinity.
		# It may still have other poles, they will be removed by
		# the procedure function_with_one_pole
		if point[3]=0 then
			point:=[(homogeneous(evala(Quo(subs({t=0,x=1}
			 ,homogeneous(f,x,y,t,`polynom`)),y-point[2]/point[1],y))
			 ,y,t,x,`polynom`)),`infinity`];
		else
			point:=[evala(Quo(evala(Expand(subs(x=point[1]/point[3],f)))
			 ,y-point[2]/point[3],y))/(x-point[1]/point[3]),`finite`]
		fi
	fi;
	function_with_one_pole(f,x,y,point)
end:

# f is a genus 0 curve in x and y. p is the parameter, i.e. p generates
# the function field
# Output: a rational function in t that gives x as a function of p.
compute_x:=proc(p,f,x,y,t)
	local i,X,np,dp,R,ansatz,C,ar;
	if not member(y,indets(f)) then RETURN(g_solve(f)) fi;
	dp:=denom(p);
	np:=numer(p);
	X[1]:=-1;
	for i from 1 to 3 do
		R[i]:=0;
		while degree(R[i],t)<degree(f,y) do
			X[i]:=X[i]+1;
			ar:=evala(subs(x=X[i],[dp*t-np,f]));
			if member(t,indets(ar)) then
				R[i]:=evala(Resultant(op(ar),y))
			fi
		od;
		X[i+1]:=X[i]
	od;
	ansatz:=(-X[1]*C*R[2]+X[2]*R[1])/(R[1]-C*R[2]);
	ansatz:=g_normal(subs(g_solve({coeffs(collect(evala(Rem(collect(numer(
	 ansatz)-X[3]*denom(ansatz),t),R[3],t)),t),t)}),ansatz));
	if member(C,indets(ansatz)) then ERROR(`something is wrong`) fi;
	ansatz
end:
	
# The following procedure computes a parameter. A parameter is a function
# that has only 1 pole, with multiplicity 1. Then k(p)=k(x,y), p generates
# the function field. The pole will be in point. In fact, as the syntax
# is now, point is not a point, but a function with a pole in a certain point.
#
# point=[function,`finite`] or [function,`infinity`]. This function must
# have one pole in finite or infinity. This function is the start for our
# search for a parameter. We add an ansatz to this point. Since this function
# is already good in a part of the plane (either `finite` or `infinity`)
# we will change no poles in that part of the plane, and try to remove the
# poles in the other part.
#
# The point [0,1,0] must not be a point on the curve. The reason is that
# we have divided the projective plane in 2 affine parts, a line and
# a plane. Doing this, there remains 1 point, and therefor that point must
# not be on the curve.
#
# Note that the way we divide the plane in parts is not really relevant
# for the method. The fact that one part is only a line does not give much
# asymmetry in the algorithm. The important thing about these parts is that
# they are affine, and not projective. We need this to apply integral basis.
#
# We will compute an integral basis for the finite part of the plane, and a
# local integral basis for the line `infinity`. Using these integral basis
# we can find linear equations stating that an ansatz, or an ansatz + a
# given function is integral (i.e. no poles) in a part of the plane (the
# part `finite` and `infinity`.
function_with_one_pole:=proc(f,x,y,point)
	local d,j,z,f_infty,ib1,ib2,den1,den2,d1,d2,ansatz,a,f1,f2,equations,
	 ext,i,zero,point1,deg_inf;
	d:=degree(f,{x,y});
	f_infty:=subs(x=1,homogeneous(f,x,y,z,`polynom`));
	ib1:=integral_basis(f,x,y);
	ib2:=integral_basis(f_infty,z,y,{z});
	den1:=g_denom(ib1[d]);
	den2:=g_denom(ib2[d]);
	if point[2]=`infinity` then
		deg_inf:=degree(den1,x)-degree(point[1],{x,y})+1;
		point1:=x^deg_inf*point[1]/den1;
		if deg_inf<0 then den1:=evala(Expand(den1*x^(-deg_inf))) fi
	else
		point1:=point[1]
	fi;
	d1:=degree(den1,x);
	d2:=ldegree(den2,z);
	d2:=max(d2,degree(g_numer(point1),{x,y})-degree(g_denom(point1),{x,y}));
	ansatz:=convert([seq(seq(a[i,j]*x^i*y^j,i=0..d1+d2-j)
	 ,j=0..d-1)],`+`)/den1;
	if point[2]=`finite` then
		f1:=ansatz;
		f2:=ansatz+point1
	else
		f1:=ansatz+point1;
		f2:=ansatz
	fi;
	# Now we solve the following set of linear equations:
	# {f1 has no poles in finite, and f2 has no poles in infinity}
	ext:=g_ext([f,f1,f2]);
	f1:=g_normal(f1);
	den1:=g_denom(f1);
	f1:=subs(g_conversion1,g_numer(f1));
	f2:=g_normal(subs(x=1,homogeneous(f2,x,y,z,`ratfunction`)));
	den2:=g_denom(f2);
	den2:=z^ldegree(den2,z);
	f2:=subs(g_conversion1,g_numer(f2));
	ib1:=subs(g_conversion1,[seq(g_expand(g_normal(i*den1),ext),i=ib1)]);
	ib2:=subs(g_conversion1,[seq(g_expand(g_normal(i*den2),ext),i=ib2)]);
	zero:=f1;
	for i from d-1 by -1 to 0 do while degree(coeff(zero,y,i),x)
	 >=degree(lcoeff(ib1[i+1],y),x) and coeff(zero,y,i)<>0 do
		zero:=g_expand(zero-ib1[i+1]*
		g_normal(lcoeff(coeff(zero,y,i),x)/lcoeff(lcoeff(ib1[i+1],y),x))
		*x^(degree(coeff(zero,y,i),x)-degree(lcoeff(ib1[i+1],y),x))
		,ext)
	od od;
	equations:={a[d1,0]=0,coeffs(zero,[x,y])};
	# These equations state that f1 is integral. We've added and 1 extra
	# linear condition, to make the solution unique. We still have to
	# add the equations for infinity:
	zero:=f2;
	for i from d-1 by -1 to 0 do while degree(coeff(zero,y,i),z)
	 >=degree(lcoeff(ib2[i+1],y),z) and coeff(zero,y,i)<>0 do
		zero:=g_expand(zero-ib2[i+1]*
		g_normal(lcoeff(coeff(zero,y,i),z)/lcoeff(lcoeff(ib2[i+1],y),z))
		*z^(degree(coeff(zero,y,i),z)-degree(lcoeff(ib2[i+1],y),z))
		,ext)
	od od;
	equations:=subs(g_conversion2,{op(equations),coeffs(zero,[y,z])});
	d:=g_normal(subs(g_solve(equations),ansatz)+point1);
	if indets(d) minus indets(f) <> {} then
		ERROR(`can not find parameter`,f=evala(AFactor(f)))
	fi;
	d
end:

# Input: f is a polynomial in x and y
# Output is a list of the following lists:
# [ [x,y,z], multiplicity, contribution in the genus ]
singularities:=proc()
	local df,ext,extl,j, nulp,pl_transl,f_translated
	 ,f,x,y,z,n,k,ff,i,mult,contr,result;
	options remember;
	f:=args[1];
	x:=args[2];
	y:=args[3];
	if nargs=3 or (nargs=4 and args[4]=`points`) then
		ff:=[op(singularities(f,x,y,`finite`)),
		 op(singularities(f,x,y,`infinity`))];
		result:=NULL;
		for i in ff do if (nargs=3 and i[2]>1) or (nargs=4 and i[2]=1)
			then result:=result,i
		fi od;
		RETURN([result])
	fi;
n:=degree(f,{x,y});
if args[4]=`infinity` then
	k:=-1;
	while k=-1 or degree(subs(z=0,ff),x)<n do
		# Making sure that we may assume y=1
		k:=k+1;
		ff:=homogeneous(subs(y=y+k*x,f),x,y,z,`polynom`)
	od;
	RETURN([seq([[i[1][2],1+k*i[1][2],0],i[2],i[3]],
	 i=singularities(subs(y=1,ff),z,x,`around zero`,args[5..nargs]))])
fi;
	extl:=g_ext(f);
	f:=subs(g_conversion1,f);
	if args[4]=`around zero` then
		df:={x}
	else
		df:=double_factors(g_expand(discrim(f,y),extl),x,extl)
	fi;
	result:=NULL;
	for k in df do
		nulp:=g_zero_of(k,x,'ext'); # is the x value
		ext:=eval(ext);
		ext:=[ext,op(extl)];
		f_translated:=g_expand(subs(x=x+nulp,f),ext);
		pl_transl:=[op(`puiseux/technical_answer`(f_translated,x,y
		 ,0,ext))];
		ff:={seq(subs(x=0,i[1]),i=pl_transl)}; # the y values
		for i in ff do # i is y value
			mult:=0;
			contr:=0;
			for j in pl_transl do if subs(x=0,j[1])=i then
				mult:=mult+degree_ext(j[1],[i,op(ext)])*min(
				 max(ldegree(j[1]-subs(x=0,j[1]),x),1)/j[3],1);
				contr:=contr+degree_ext(j[1],[i,op(ext)])
				 *(j[7]-(j[3]-1)/j[3])/2
			fi od;
			if not (type(mult,integer) and type(contr,integer))
				then ERROR(`something is wrong`)
			fi;
			result:=result,[[nulp,i,1],mult,contr]
		od
	od;
	subs(g_conversion2,[result])
end:

# f is a polynomial or a rational function in x and y.
# Output: the corresponding homogeneous rational function or polynomial
homogeneous:=proc(ff,x,y,z,what)
local i,j,n,f;
	f:=expand(g_normal(ff));
	if what=`ratfunction` then
		i:=g_numer(f);
		j:=g_denom(f);
		n:=degree(i,[x,y,z])-degree(j,[x,y,z]);
		if n<0 then i:=i*z^(-n) elif n>0 then j:=j*z^n fi;
		RETURN(homogeneous(i,x,y,z,`polynom`)
		 /homogeneous(j,x,y,z,`polynom`))
	fi;
	n:=degree(f,{x,y});
	convert([seq(seq(coeff(coeff(f,x,i),y,j)*x^i*y^j*z^(n-i-j)
	,i=0..n-j),j=0..n)],`+`)
end:

# Searches a point in a small algebraic extension using the information
# gathered during the Puiseux series computation.
find_point:=proc(f,x,y)
local vp,i,m,j;
	vp:=singularities(f,x,y,`points`);
	m:=min(degree(f,{x,y}),seq(degree_ext(i,f),i=vp));
	if m=degree(f,{x,y}) and m>2 then
		m:=singularities(f,x,y);
		if nops(m)=1 and m[1][2]=degree(f,{x,y})-1 then
		# This is not a regular point, but it has mult n-1
		# and also a tangent line in the right direction
			m:=m[1][1];
			if m[3]=0 then RETURN([y-x*m[2]/m[1],`infinity`])
			else RETURN([(y-m[2])/(x-m[1]),`finite`])
			fi
		# a function with 1 pole is also allowed instead of a point
		fi;
		vp:={seq(i[1][1],i=m)};
		i:=0;
		while member(i,vp) do i:=i+1 od;
		vp:=readlib(factors)(subs(x=i,f));
		m:=min(seq(degree(j[1],y),j=vp[2]));
		for j in vp[2] do if degree(j[1],y)=m then
			RETURN([i,RootOf(j[1]),1])
		fi od
	fi;
	for i in vp do if degree_ext(i,f)=m then RETURN(i[1]) fi od;
	ERROR(`?`)
end:

lprint(`For plotting type: ?plot_knot`);
macro(  element_of=`plot_knot/element_of`,
	equal_to_zero=`plot_knot/equal_to_zero`,
	all_substitutions=`plot_knot/all_substitutions`,
	curve=`plot_knot/curve`
):

# Test if subs(x^(1/d) = any d-th root of unity * x^(1/d),el) is
# an element of verz.
# x stands for x^(1/d) here
element_of:=proc(el,d,verz,x) local i,elz,j;
	# Remove the branches that don't go through 0:
	if coeff(el,x,0)<>0 then RETURN(true) fi;
	if d=1 then RETURN(false) fi;
	for i from 1 to d-1 do
		elz:=evala(evalf(subs(x=evalc(exp(i*2.0*Pi*I/d))*x,el)));
		for j in verz do if equal_to_zero(j[1]-elz,x) then
			RETURN(true)
		fi od
	od;
	false
end:

equal_to_zero:=proc(a,x) local som,i;
	som:=0.0;
	for i in {coeffs(a,x)} do som:=evalf(som+abs(i)^2) od;
	if som<0.000001 then RETURN(true) fi;
	false
end:

# All possible floating point substitutions for algebraic numbers.
# v: a list of RootOf's
all_substitutions:=proc(v) local i,a,b,result,j;
	if v=[] then RETURN({[]}) fi;
	a:=v[nops(v)];
	result:={seq([a=i],i=[fsolve(op(a),_Z,complex)])};
	b:=v[1..nops(v)-1];
	{seq(seq([op(j),op(i)],j=all_substitutions(subs(i,b))),i=result)}
end:

# f is a polynomial in x en y
plot_knot:=proc(f,x,y)
	local ext_f,pui,ext_pui,i,pui2,curven,true_or_not,t,opt,eps,cols;
	opt:={args[4..nargs]};
	if not member(numpoints,indets(opt)) then opt:=opt union {numpoints=150} fi;
	if not member(radius,indets(opt)) then opt:=opt union {radius=0.05} fi;
	if not member(tubepoints,indets(opt)) then opt:=opt union {tubepoints=5} fi;
	if not member(scaling,indets(opt)) then opt:=opt union {scaling=CONSTRAINED} fi;
	if not member(style,indets(opt)) then opt:=opt union {style=PATCH} fi;
	eps:=1;
	if member(epsilon,indets(opt)) then
		eps:=subs(opt,epsilon);
		opt:=opt minus {epsilon=eps}
	fi;
	cols:=[];
	if member(colours,indets(opt)) then
		cols:=subs(opt,colours);
		opt:=opt minus {colours=cols}
	fi;
	ext_f:=g_ext(f);
	pui:=`puiseux/technical_answer`(g_expand(subs(g_conversion1,f),ext_f)
	 ,x,y,0,ext_f);
	ext_f:=subs(g_conversion2,ext_f);
	ext_pui:=subs(g_conversion2,g_ext(pui));
	ext_pui:=ext_pui[1..nops(ext_pui)-nops(ext_f)];
	ext_f:=all_substitutions(ext_f);
	ext_f:=ext_f[1];
	ext_pui:=all_substitutions(subs(ext_f,ext_pui));
	pui:=subs(ext_f,subs(g_conversion2,pui));
	# Now replace the RootOf's by all corresponding floating point numbers
	pui2:={};
	for i in ext_pui do
		pui2:=pui2 union subs(i,pui)
	od;
	pui:=pui2;
	pui2:={};
	for i in pui do
		true_or_not:=element_of(i[1],i[3],pui2,x);
		if (not true_or_not) then pui2:=pui2 union {i} fi
	od;
	curven:={};
	for i in pui2 do curven:=curven union {curve(i[1],i[3],t,eps,x)} od;
	if printlevel>1 then print(`Number of branches:`,nops(curven)) fi;
	if cols=[] then
		plots[tubeplot](curven,t=0..2*Pi,op(opt))
	else
		curven:=[op(curven)];
		plots[display]([seq(plots[tubeplot](curven[i],t=0..2*Pi,op(opt),
		 color=cols[1+irem(i,nops(cols))]),i=1..nops(curven))])
	fi
end:

curve:=proc(functie_in_x,d,t,eps,x) local f,fre,fim,factor;
	f:=expand(subs(x=eps*(cos(t)+I*sin(t)),functie_in_x));
	fre:=coeff(f,I,0)/eps;fim:=coeff(f,I,1)/eps;
	factor:=1/(sqrt(1+fre^2+fim^2)-fre);
	[factor*cos(d*t),factor*sin(d*t),factor*fim];
end:

# Here are some more nice test examples for parametrization computation:

# f:=expand( (y-x)*(y-2*x)*(y-3*x)*(y-4*x) + x^3);
# Has only 1 singularity, so parameter computation is trivial

# f:=113/2592000*x-19/2592000*y+119/129600*y^2-2/405*y*x-271/129600*x^2+7/18*y^2*x
# -17/18*y*x^2-10*y^3*x+35*y^2*x^2-50*y*x^3+24*x^4+307/180*x^3-1/20*y^3+y^4;

# f:=25+1326*y*x+3740*x^3*y+3252*x^2*y+3582*y^3*x^2+4030*x^3*y^2+1476*x^2*y^4+102*
# y^6*x+546*y^5*x+1590*y^4*x+244*x+184*y+518*y^2+720*y^3+854*x^2+1338*x^3+576*y^4
# +1101*x^4+282*y^5+14*y^7+y^8+84*y^6+4770*y^2*x^2+2706*y^3*x+2646*y^2*x+18*x^7+
# 508*x^5+132*x^6+x^8+2264*x^4*y+2124*x^3*y^3+1716*x^4*y^2+738*y*x^5+354*y^2*x^5+
# 570*x^4*y^3+550*x^3*y^4+122*y*x^6+28*y^6*x^2+318*x^2*y^5+8*y^7*x+56*y^3*x^5+56*
# y^5*x^3+70*y^4*x^4+28*y^2*x^6+8*y*x^7;
# Parameter computation does not take much time, but parametrization takes 500
# seconds on a fast machine. Probably evala(Resultant()) is not nearly as
# efficient as resultant().

# f:=-89325/138014+413549/138014*y-185984/69007*y^2+1518117/552056*y^3-19259/69007
# *y^4+y^5-155466/69007*x+153719/69007*x*y-392407/138014*x*y^2+31903/138014*x*y^3
# -66141/69007*x*y^4-314712/69007*x^2+165666/69007*x^2*y-392063/276028*x^2*y^2-
# 50807/138014*x^2*y^3-81846/69007*x^3-50807/138014*x^3*y+244125/138014*x^3*y^2-
# 688135/552056*x^4*y;

# f:=-y-4*y^2-y^2*x-5*y*x-2*x^2+3*x^2*y-6*y^3-2*y^4+x^3+3*x*y^4+6*y^5*x^2+4*x*y^7+12
# *y^5*x+3*y^6+y^9+x^2*y^2+6*x^2*y^3+x*y^3-2*y^5+12*x*y^6+4*y^8+6*y^7+4*x^3*y^3+4
# *y^2*x^3+12*y^4*x^2+x^4*y;

#save `IntBasis.m`;
#quit
