#
## <SHAREFILE=combinat/perm/perm.mpl >
## <DESCRIBE>
##                SEE ALSO: combinat/perm/perm.tex  (30K)
##                A package of routines for computing with permutation groups,
##                including routines for computing orbits, subgroups, testing
##                for conjugacy. There are also procedures used to represent and
##                manipulate combinatorial structures.
##                AUTHOR: Yves Chiricota, chiricot@firmin.greco-prog.fr
## </DESCRIBE>


#      --------oOo--------
#         perm package 
#      --------oOo--------

#        Author: Yves Chiricota  May 1993.
#        email: chiricot@geocub.greco-prog.fr
#        LaBRI,
#        Universite Bordeaux I,
#        351, cours de la Liberation,
#        33405, Talence, Cedex,
#        FRANCE.


# This file contains the definition for the procedure of
# the "perm" package. Moreover, it create the table that 
# contain the "readlib" call to ".m" files (one file for 
# each  procedure).



# Initialisation procedure. The initialisation consist to
# read representative of subgroup conjugacy class of S7.
# This is a table generated by gentabsg and called "tabsg".
# It try to read that table in the library. 
`perm/init` := proc()
   local rd;
   # On definit rd pour que le read soit "trappable"...
   rd:=proc() readlib(tabsg) end:
   if traperror(rd()) = lasterror 
	then printf(`%s`,`* The table "tabsg" is not defined *\n `);

	else printf(`%s`,`* The table "tabsg" is loaded  *\n `);
	fi
end:

# ----------- beginning of proc definitions --------------------


# All permutations are given as word on {1,2,...,n}.

# Inverse of p.
`perm/inv` := proc(p)
   local i,q;
   for i from 1 to nops(p) do	
	q[p[i]]:=i; od;
   [seq(q[i],i=1..nops(p))];
end:

# This procedure check whether p < q in lexicographic order on word .
`perm/minlex` := proc(p,q)
   local i,egaux;
   egaux := true;
   i:=1;
   while (i<=nops(p) and egaux) do 
      if p[i]=q[i] 
         then i:=i+1 
         else egaux := false 
         fi
      od;
   if egaux=true 
      then RETURN(false)
      elif p[i]>q[i]
         then RETURN(false)
         else RETURN(true)
      fi;
end:

# Product of p and q.
`perm/prod` := proc(p,q) local i;
	[seq(p[i],i=q)];
end:

# Sign of p.
`perm/sgn` := proc(p)
   local cyc;
   cyc:=perm[decomp](p);
   RETURN(convert(map(proc(c) (-1)^(nops(c)-1) end,cyc),`*`));
end:


# Conjugate p by g.
`perm/conj` := proc(p,g)
   perm[prod](g,perm[prod](p,perm[inv](g)))
end:


# This procedure returns the successor of p with respect to
# lexicographic order.
`perm/nextperm` := proc(p)
    local pos,n,k,i,pastrouve,U,V;
    n:=nops(p);
    pos:=n; pastrouve:=true;
    while (pastrouve and pos>1) do
        if p[pos]>p[pos-1]
             then pastrouve:=false 
             else pos:=pos-1 fi;
        od;
    if pos=1 then RETURN(perm[Id](n)) fi;
    k:=min(op(map(proc(a,b) 
                     if a>b then a fi end,
                  {op(p[pos..n])},p[pos-1]))); 
    V:= {op(p[pos..n])} minus {k};
    RETURN([op(p[1..pos-2]),k,op(sort([p[pos-1],op(V)]))])
end:

# Identity permutation on n points.
`perm/Id` := proc(n)
   [$1..n]
end:

# Cyclic group generated by p.
`perm/grcyc` := proc(p)
   local H,q,Id0;
   Id0:=perm[Id](nops(p));
   if p=Id0    
     then RETURN({p})
     else
        q:=p; H:={p};
        while q<>Id0 do
           q:=perm[prod](q,p);  H:=H union {q}  od;
        RETURN(H)
        fi
end:

# Dihedral group of order n.
`perm/dihedral`:=proc(n) 
   local Cn,i; option remember;
   Cn:=perm[grcyc]([$2..n,1]);
 RETURN(Cn union map(proc(p,q) perm[prod](q,p) end,Cn,[1,seq(n-i,i=0..n-2)]));
end:

# Give the cycle that contain i in p.
`perm/cyc` := proc(i,p)
		local k,cycle;
		cycle:=[i];
		k:=p[i];
		while (i<>k) do cycle:=[op(cycle),k];
																		k:=p[k];
															od;
  RETURN(cycle);
end:

# Return p in disjoints cycle form.
`perm/decomp` := proc(mot)
  if type(mot,table)
    then
       RETURN(map('perm[decomp]',mot));
    fi;
  if type(mot,{set,listlist})
    then map(proc(m) [`perm/decomp/decomp_1`(m,{op(m)})] end,mot)
    else	[`perm/decomp/decomp_1`(mot,{op(mot)})]
    fi;
end:
`perm/decomp/decomp_1` := proc(mot,restant)
   local c;
   if restant<>{} then
	c:=perm[cyc](min(op(restant)),mot);
     `perm/decomp/decomp_1`(mot,restant minus {op(c)}),c
     fi;
end:

# Cyclic type of q.					
`perm/ctype` := proc(q)
			local i,t,lo,M,p;
   p:=perm[decomp](q);
			if nops(p)>0 then
      M:=max(op(map('nops',p)));
						for i from 1 to M do t[i]:=0; od;
						for i from 1 to nops(p) do	lo:=nops(p[i]);	t[lo]:=t[lo]+1;	od;
		 		 RETURN(convert(t,list));
						fi;
end:

# Inject p in Sn (where p is a permutation in Sk with k<=n).
`perm/complete` := proc(p,n)
   [op(p),$nops(p)+1..n]
end:

# Subgroup generated by X.
`perm/gensg` := proc(X,n)
   local Ident,pile,H,sigma,gamma,i,Xn,t;
   Ident:=perm[Id](n);
   Xn:=map(proc(p,n) perm[complete](p,n) end,X,n);
   H:={Ident,op(Xn)};
   pile:=Xn;
   while pile<>[] do
      sigma:=op(1,pile);
      pile:=subsop(1=NULL,pile);
      for i from 1 to nops(Xn) do
         gamma:=[seq(sigma[t],t=op(i,Xn)),$nops(sigma)+1..n];
         if not member(gamma,H)
            then
                pile:=[gamma,op(pile)];
               H:=H union {gamma};
            fi;
         od;
      od;
   RETURN(H);
end:
`perm/gensg` := proc(X,n)
   local Ident,pile,H,sigma,gamma,i,Xn,t,temp;
   Ident:=perm[Id](n);
   Xn:=map(proc(p,n) perm[complete](p,n) end,X,n);
   H:={Ident,op(Xn)};
   pile:=Xn;
   while pile<>[] do
      sigma:=pile[1];
      # pile:=[$('pile[j]','j'=2..nops(pile))];
      # pile:=subsop(1=NULL,pile);
      temp:=NULL;
      for i in Xn do
         # gamma:=perm[complete](perm[prod](sigma,op(i,Xn)),n);
         gamma:=[seq(sigma[t],t=i),$nops(sigma)+1..n];
         if not member(gamma,H)
            then
               temp:=gamma,temp;
               H:=H union {gamma};
            fi;
      od;
      pile:=subsop(1=temp,pile);
   od;
   RETURN(H);
end:


# Lateral class of g with respect to K.
`perm/cllat` := proc(g,K)
   convert(map(proc (x,g) perm[prod](g,x) end,K,g),set);
end:

# Minimum permutation (in the lex. order) of the list lPerm.
`perm/pmin` := proc(lPerm)
   local i,pm;
   pm:=lPerm[1];
   for i from 2 to nops(lPerm) do
      if perm[minlex](lPerm[i],pm)
         then pm:=lPerm[i]
         fi;
      od;
   RETURN(pm);
end:

# Return the set whose elements are orbits of H 
# acting on {1,2,..,|H|}.
`perm/orbits` := proc(K)
    local reste,orbs,classe,H;
    H:=convert(K,set); 
    reste:={$1..nops(op(1,H))};

    orbs:={};
    while reste <> {} do
        classe:=map(proc(p,reste) 
                        perm[action](p,op(1,reste)) 
                     end,
                     H,reste);
        orbs:=orbs union {classe};
        reste:=reste minus classe;
        od;
    RETURN(orbs);
end:

# List of inversions of p.
`perm/inversions` :=proc(p)
    local i,j,L;
    L:=NULL;
    for i from 1 to nops(p) do
        for j from i+1 to nops(p) do
            if p[i]>p[j] then L:=L,[i,j] fi;
            od;
        od;
    RETURN([L]);
end:    

# List of descents ofp.
`perm/desc` := proc(p)
   map(proc(i,p) if p[i]>p[i+1] then i fi end,[$1..nops(p)-1],p);
end:

# Procedure to manipulate combinatorial structures (type ?struct).

# Do a permutation act on a structure.
`perm/action` := proc(sigma,struct)
   eval(`perm/action/action_1`(perm[inv](sigma),struct))
end:
`perm/action/action_1` := proc(sigma,struct)
   map(proc(fact,sigma) 
           if type(fact,integer) 
             then RETURN(sigma[fact])
             else RETURN(`perm/action/action_1`(sigma,fact))
             fi
           end,
        struct,sigma);
end:

# Under lying set of a structure.
`perm/ulset` := proc(s)
   { `perm/ulset/epluche`(s) }
end:
`perm/ulset/epluche` := proc(s)
   if type(s,numeric) 
      then RETURN(s)
      else op(map('`perm/ulset/epluche`',s))
      fi;
end:

# Cardinality or the under lying set of a structure.
`perm/ulcard` :=proc(s)
  nops(perm[ulset](s));
end:

# Use to generate the orbit of s with respect to the action of
# the symmetric group Sn (where n=ulcard(s)).
`perm/genstruct` := proc(s)
   local E,Sn,i;   
   Sn:=combinat[permute](perm[ulcard](s));
   E:={};
   for i from 1 to nops(Sn) do
      E:=E union {perm[action](Sn[i],s)};
      od;
   RETURN(E);
end:

# This procedure is used to "regroup" a structure s with respect to
# a subgroup H of Sn. This operation consists to calculate H.s. This 
# is a way for create new structures from old ones.
`perm/regroup`:=proc(H,s)
    map(proc(g,s) perm[action](g,s) end,convert(H,set),s)
end:

# To create a p-structure (type ?makestruct).
`perm/makestruct` := proc(nom,H) 
   option remember; # A structure don't can be created twice.
  assign(nom,subs({stab=H,nomstruct=nom},
	 proc() 
    	local  sigma,tau,K,i;
	   	   sigma:=`perm/makestruct/permInd`(args);
   		   K:=stab;
   		   tau:=perm[pmin](perm[cllat](sigma,K));
   		   sigma:=perm[prod](perm[inv](sigma),tau);
   		   RETURN(evaln(nomstruct(seq(args[sigma[i]],i=1..nargs))))
		end));
  nom;
  # op(nom);
end:

`perm/makestruct/permInd` := proc()
   local i,m,f,g;
   m:=[seq(min(op(perm[ulset](args[i]))),i=1..nargs)];
   f:=sort(m);
   for i from 1 to nargs do
      g[f[i]]:=i
      od;
   RETURN([seq(g[i],i=m)]);
end:


# Cycle on n points.
`perm/C/makecycle` := proc(n)
   perm[makestruct](C.n,perm[grcyc]([$2..n ,1]))
end:

# Polygon on n points.
`perm/P/makepolyg` :=proc(n)
   perm[makestruct](P.n,perm[dihedral](n))
end:

# Generic cycle.
`perm/C` := proc() option remember;
   if not assigned(C.nargs) 
	then `perm/C/makecycle`(nargs) fi; 
   (C.nargs)(args);
   subs(C.nargs=C,");
end:
   
# Generic polygon.
`perm/P` := proc() option remember;
   if not assigned(P.nargs)
   	then `perm/P/makepolyg`(nargs) fi;
   (P.nargs)(args);
   subs(P.nargs=P,");
end:

# Stabilizer of a structure.
`perm/stab` := proc(s)
   local i,Sn,ensStab;  option remember;
   Sn:=combinat[permute](perm[ulcard](s));
   ensStab:={};
  for i from 1 to nops(Sn) do
      if perm[action](Sn[i],s)=s
         then ensStab:=ensStab union {Sn[i]}
         fi
      od;
   RETURN(ensStab)
end:

# Stabilizer of a structure (second method).
`perm/stabit` := proc(s)
   local p,ident,ensStab;
   ident:=perm[Id](perm[ulcard](s));
   ensStab:={ident};
   p:=perm[nextperm](ident);
   while p<>ident do
      if perm[action](p,s)=s
         then ensStab:=ensStab union {p}
         fi;
      p:=perm[nextperm](p);
      od;
   RETURN(ensStab)
end:


# Conjugacy procedures.

# Subprocedure of grconj
`perm/grconj/ctype` := proc(p)
			local i,t,lo,M;
			if nops(p)>0 then
      M:=max(op(map('nops',p)));
						for i from 1 to M do t[i]:=0; od;
						for i from 1 to nops(p) do	lo:=nops(p[i]);	t[lo]:=t[lo]+1;	od;
		 		 RETURN(convert(t,list));
						fi;
end:

# The next procedure returns a table whose indexes are cyclic type
# of elements of H.
# Notation: t(g) = cyclic type of g (t(g)[i] = number of cycle 
#                  of length i in g).
# - clconj(H)[t(g)] will contain all permutations of type t(g).
# - H is given as a set or permutation.
`perm/grconj/clconj` := proc(H)
   local C,types,i;
   # C = table with type as indexes: C[t(g)] = {perm. of type t(g)}
      # Initialisation.
   types:=map(proc(p) perm[ctype](p) end,H);
   C:=table();
   
   for i from 1 to nops(types) do
      C[op(types[i])] := {}
      od;
   for i from 1 to nops(H) do
      C[op(perm[ctype](H[i]))]:=
         C[op(perm[ctype](H[i]))] union {H[i]}
      od;
   RETURN(op(C));
end:


# Take a set of conjugaison classes and return the set of couples
# [t(C),|C| for each class C.
# Cl is a table whose indexes are cyclic types.
`perm/grconj/indcard` := proc(Cl)
   map(proc(t,card) [t,card[op(t)]] end,
           [indices(Cl)],
           map('nops',Cl));
   RETURN(convert(",set))
end:

# This implement a total order on couples [t(C),|C|].
# I1 and I2 are couples of form [t(C),|C|], (where C is a 
# permutation conjugaison class). Classes are ordered 1) by
# cardinality and 2) by inverse lex. order of cyclic type.
`perm/grconj/ordrecl` := proc(I1,I2)
   local t1,t2,maxl;
   if op(2,I1)<op(2,I2) 
      then RETURN(true)
      elif op(2,I1)>op(2,I2)
         then RETURN(false)
         else
            maxl:=max(nops(op(1,I1)),nops(op(1,I2)));
            t1:=[op(op(1,I1)),0$(maxl-nops(op(1,I1)))];
            t2:=[op(op(1,I2)),0$(maxl-nops(op(1,I2)))];
            RETURN(not minlex(t1,t2));
      fi;
   
end:

# Return index of the smaller class in lindcard, where
# the parameter lindcard is a list of couples (t,c).
`perm/grconj/minlexcl` := proc(lindcard)
  local i,pp;
  pp:=lindcard[1];
  for i from 2 to nops(lindcard) do
     if `perm/grconj/ordrecl`(lindcard[i],pp)
        then pp:=lindcard[i]
        fi
     od;
   RETURN(pp);
end:


# Return true if H1 and h2 are conjugated by g, if not,
# false is returned.
`perm/grconj/permconj` := proc(H1,H2,g)
   local i;
   if nops(H1)<>nops(H2)
      then RETURN(false)
      else i:=1;         
          while i<=nops(H1) do
            if not member(perm[conj](H1[i],g),H2)
               then RETURN(false) fi;
            i:=i+1;
            od;
         RETURN(true);
         fi;
end:


# This procedure find if H1 and H2 are conjugated.
`perm/grconj` := proc(H1,H2)
   local Cl1,Cl2,i,j,indC,clmin,Sn,conjp;
   if H1=H2 then RETURN(true) fi;
   if nops(H1)<>nops(H2) then RETURN(false) fi;
   if `perm/grconj/ctype`(perm[orbits](H1))<>`perm/grconj/ctype`(perm[orbits](H2))
     then RETURN(false) fi;
     
   Cl1:=`perm/grconj/clconj`(H1);
   Cl2:=`perm/grconj/clconj`(H2);

   # If the indexes set associed to H1 and H2 are not identical
   # then H1 and H2 are not conjugated.
   indC:=`perm/grconj/indcard`(op(Cl1)); 
   if {indC}<>{`perm/grconj/indcard`(op(Cl2))}
      then RETURN(false) fi;
   
   # Next, we check if there is a permutation that conjugate
   # the smaller class.   
   indC:=indC minus { `perm/grconj/minlexcl`(indC) };
   Sn:=combinat[permute](nops(H1[1]));
   clmin:=`perm/grconj/minlexcl`(indC);
   indC := convert(indC minus { `perm/grconj/minlexcl`(indC) },list);

   # We search a permutation that conjugate the smaller class of
   # Cl1 and Cl2. If there is a such permutation, we check that
   # it conjugate the others class.
   for i from 1 to nops(Sn) do
      if `perm/grconj/permconj`(Cl1[op(clmin[1])],Cl2[op(clmin[1])],Sn[i])
         then
            conjp:=true;
            j:=1;
            while (conjp and (j<=nops(indC))) do 
               if not `perm/grconj/permconj`(Cl1[op(op(1,indC[j]))],
                               Cl2[op(op(1,indC[j]))],Sn[i])
                  then conjp:=false
                  fi;
               j:=j+1;
               od; # while
             if conjp then RETURN(true) fi;
         fi;
      od; # If

   RETURN(false);
end:


        
# Multiplication in group algebra of Sn.

`convert/and`:=proc(l) 
   local i,A; 
   A:=l[1]; 
   for i from 2 to nops(l) do A:=A and l[i] od;
end:

# Scalar type. 
`type/scal` := proc(p)
    if type(p,{numeric,name,indexed}) then RETURN(true) fi;
    if type(p,`^`) and type(op(1,p),{numeric,name,indexed}) then RETURN(true) fi;
    if type(p,{`*`,`+`}) and
#    if type(p,`*`) and 
	convert(map(proc(x) if type(x,{numeric,name,indexed}) 
		then true else false fi end,[op(p)]),`and`)
	then RETURN(true) fi;
    RETURN(false)
end:

# perm type.
`type/perm` := proc(p)
    if type(p,list) then true
    elif type(p,`^`) and type(op(1,p),list)
    then true else false fi
end:

# Multiplication operator.
`perm/&X` := proc()
local _AA;
#options remember;
     _AA := [args];
    _AA := map(proc(x,pn)
                   if type(x,function) and (op(0,x) = pn) then op(x) else x fi
               end,
        _AA,procname);
    if nops(_AA) = 1 then RETURN(_AA[1]) fi;

    if type(_AA[1],`+`) 
      then RETURN(map(proc(mon,q) mon &X q end,_AA[1],_AA[2])) fi;
    if type(_AA[2],`+`) 
      then RETURN(map(proc(mon,q) q &X mon end,_AA[2],_AA[1])) fi;

    if nargs = 2 then RETURN(eval(`perm/&X/pprod`(_AA[1],_AA[2]))) fi;
    'procname'(op(_AA))
end:

# Product of two permutations.
`perm/&X/pprod`:=proc(p,q) local p1,q1,a1,b1;
    if type(p,perm) and type(q,perm)
        then RETURN(perm[prod](p,q)) fi;
    if (type(p,scal) and type(q,perm)) or (type(p,perm) and type(q,scal))
        or (type(p,scal) and type(q,scal)) then RETURN(p*q) fi;    
    if type(p,`*`) 
        then a1:=select(`type/scal`,p);
             p1:=select(`type/perm`,p)
        elif type(p,scal) 
            then a1:=p; p1:=[]
            else a1:=1; p1:=p;
        fi;
    if type(q,`*`) 
        then b1:=select(`type/scal`,q);
             q1:=select(`type/perm`,q)
        elif type(q,scal) 
            then b1:=q; q1:=[]
            else b1:=1; q1:=q
        fi;
    RETURN(a1*b1*perm[prod](p1,q1))
end:

# Permutation that appears in the polynom P.
`perm/supp` :=proc(P) local Q;
  #Q:=expand(P);
  Q := P;
  if not type(Q,`+`)
      then if type(Q,perm) 
	        then RETURN({Q}) 
	        else RETURN({select(`type/perm`,Q)})
	        fi;
     fi;
  map(proc(m) if type(m,perm)
                  then RETURN(m) 
                  else RETURN(select(`type/perm`,m))
                  fi
           end,
      convert(Q,set)); 

end:


# The following table contains generator of representative 
# of conjugacy class of S7 subgroups.

`perm/tab_gen`:=table();

# On 7 points.

`perm/tab_gen`[C6c]:=[[2,3,1,5,4,7,6]]:

`perm/tab_gen`[C6C2a]:=[[2,3,1,5,4,6,7],[1,2,3,4,5,7,6]]:

`perm/tab_gen`[S3e]:=[[2,1,3,5,4,7,6],[3,2,1,5,4,7,6]]:

`perm/tab_gen`[D6c]:=[[2,3,1,5,4,6,7],[2,1,3,4,5,6,7]]:

`perm/tab_gen`[D6d]:=[[2,3,1,4,5,7,6],[2,1,3,5,4,6,7]]:

`perm/tab_gen`[D6e]:=[[2,3,1,5,4,7,6],[2,1,3,5,4,6,7]]:

`perm/tab_gen`[S3Ka]:=[[2,1,3,4,5,6,7],[3,2,1,4,5,6,7],[1,2,3,5,4,6,7],[1,2,3,4,5,7,6]]:

`perm/tab_gen`[C10a]:=[[2,3,4,5,1,7,6]]:

`perm/tab_gen`[D5b]:=[[2,3,4,5,1,6,7],[1,5,4,3,2,7,6]]:

`perm/tab_gen`[D10a]:=[[2,3,4,5,1,6,7],[1,5,4,3,2,6,7],[1,2,3,4,5,7,6]]:

`perm/tab_gen`[H5b]:=[[2,3,4,5,1,6,7],[1,3,5,2,4,7,6]]:

`perm/tab_gen`[H5C2a]:=[[2,3,4,5,1,6,7],[1,3,5,2,4,6,7],[1,2,3,4,5,7,6]]:

`perm/tab_gen`[A5C2a]:=[[2,3,4,5,1,6,7],[2,3,1,4,5,6,7],[1,2,3,4,5,7,6]]:

`perm/tab_gen`[S5c]:=[[2,3,4,5,1,6,7],[2,3,4,1,5,7,6]]:

`perm/tab_gen`[S5C2a]:=[[2,3,4,5,1,6,7],[2,3,4,1,5,6,7],[1,2,3,4,5,7,6]]:

`perm/tab_gen`[C12a]:=[[2,3,1,5,6,7,4]]:

`perm/tab_gen`[C6C2b]:=[[2,3,1,5,4,7,6],[1,2,3,6,7,4,5]]:

`perm/tab_gen`[D4C3a]:=[[2,3,1,5,6,7,4],[1,2,3,6,5,4,7]]:

`perm/tab_gen`[A4c]:=[[2,3,1,5,6,4,7],[1,2,3,5,4,7,6]]:

`perm/tab_gen`[A4C3a]:=[[2,3,1,4,5,6,7],[1,2,3,5,6,4,7],[1,2,3,5,4,7,6]]:

`perm/tab_gen`[S4C3a]:=[[2,3,1,5,6,7,4],[1,2,3,5,6,4,7]]:

`perm/tab_gen`[Q6a]:=[[2,3,1,5,4,7,6],[2,1,3,6,7,5,4]]:

`perm/tab_gen`[C4S3a]:=[[2,3,1,5,6,7,4],[2,1,3,4,5,6,7]]:

`perm/tab_gen`[D6f]:=[[2,3,1,5,4,7,6],[2,1,3,6,7,4,5]]:

`perm/tab_gen`[S3Kb]:=[[2,1,3,4,5,6,7],[3,2,1,4,5,6,7],[1,2,3,6,7,4,5]]:

`perm/tab_gen`[D12a]:=[[2,3,1,5,6,7,4],[2,1,3,6,5,4,7]]:

`perm/tab_gen`[D4S3_a]:=[[2,3,1,5,4,7,6],[2,1,3,6,5,4,7]]:

`perm/tab_gen`[D4S3_b]:=[[2,3,1,5,4,6,7],[2,1,3,6,7,4,5]]:

`perm/tab_gen`[D4S3a]:=[[2,3,1,5,6,7,4],[2,1,3,6,5,4,7],[2,1,3,4,5,6,7]]:

`perm/tab_gen`[A4S3a]:=[[2,3,1,5,4,7,6],[2,1,3,5,6,4,7]]:

`perm/tab_gen`[S4e]:=   [[2,1,3,5,6,7,4],[3,1,2,5,6,4,7]]:

`perm/tab_gen`[S4S3_a]:=[[2,1,3,5,6,7,4],[3,1,2,5,6,4,7],[2,3,1,4,5,6,7]]:

`perm/tab_gen`[S4S3a]:= [[2,1,3,5,6,7,4],[3,1,2,5,6,4,7],[2,1,3,4,5,6,7]]:

`perm/tab_gen`[C7a]:=[[2,3,4,5,6,7,1]]:

`perm/tab_gen`[D7a]:=[[2,3,4,5,6,7,1],[1,7,6,5,4,3,2]]:

`perm/tab_gen`[H7_a]:=[[2,3,4,5,6,7,1],[1,3,5,7,2,4,6]]:

`perm/tab_gen`[H7a]:=[[2,3,4,5,6,7,1],[1,4,7,3,6,2,5]]:

`perm/tab_gen`[PSL27a]:=[[2,3,4,5,6,7,1],[2,3,7,5,1,4,6]]:

`perm/tab_gen`[A7a]:=[[2,3,4,5,6,7,1],[2,1,4,3,5,6,7]]:

`perm/tab_gen`[S7a]:=[[2,3,4,5,6,7,1],[2,1,3,4,5,6,7]]:




`perm/tab_gen`[I]:=[[1,2,3,4,5,6,7]]:


# On 6 points.

`perm/tab_gen`[C2c]    :=    [[2,1,4,3,6,5,7]]:

`perm/tab_gen`[Kc]     :=    [[2,1,3,4,5,6,7],[1,2,4,3,6,5,7]]:

`perm/tab_gen`[Kd]     :=    [[2,1,4,3,5,6,7],[2,1,3,4,6,5,7]]:

`perm/tab_gen`[C23a]   :=    [[2,1,3,4,5,6,7],[1,2,4,3,5,6,7],[1,2,3,4,6,5,7]]:

`perm/tab_gen`[C4b]    :=    [[2,3,4,1,6,5,7]]:

`perm/tab_gen`[C4C2a]  :=    [[2,3,4,1,5,6,7],[1,2,3,4,6,5,7]]:

`perm/tab_gen`[Ke]     :=    [[2,1,4,3,5,6,7],[3,4,1,2,6,5,7]]:

`perm/tab_gen`[C23b]   :=    [[2,1,4,3,5,6,7],[3,4,1,2,5,6,7],[1,2,3,4,6,5,7]]:

`perm/tab_gen`[D4b]    :=    [[3,4,2,1,5,6,7],[2,1,3,4,6,5,7]]:

`perm/tab_gen`[D4c]    :=    [[3,4,2,1,6,5,7],[2,1,3,4,5,6,7]]:

`perm/tab_gen`[D4d]    :=    [[3,4,2,1,6,5,7],[2,1,3,4,6,5,7]]:

`perm/tab_gen`[D4C2a]  :=    [[3,4,2,1,5,6,7],[2,1,3,4,5,6,7],[1,2,3,4,6,5,7]]:

`perm/tab_gen`[A4C2a]  :=    [[2,3,1,4,5,6,7],[2,1,4,3,5,6,7],[1,2,3,4,6,5,7]]:

`perm/tab_gen`[S4b]    :=    [[2,3,4,1,6,5,7],[2,3,1,4,5,6,7]]:

`perm/tab_gen`[S4C2a]  :=    [[2,3,4,1,5,6,7],[2,3,1,4,5,6,7],[1,2,3,4,6,5,7]]:

`perm/tab_gen`[C3b]    :=    [[2,3,1,5,6,4,7]]:

`perm/tab_gen`[C32a]   :=    [[2,3,1,4,5,6,7],[1,2,3,5,6,4,7]]:

`perm/tab_gen`[S3C3a]  :=    [[2,1,3,4,5,6,7],[3,2,1,4,5,6,7],[1,2,3,5,6,4,7]]:

`perm/tab_gen`[S3c]    :=    [[2,1,3,5,4,6,7],[3,2,1,6,5,4,7]]:

`perm/tab_gen`[S32_a]  :=    [[2,3,1,4,5,6,7],[1,2,3,5,6,4,7],[2,1,3,5,4,6,7]]:

`perm/tab_gen`[S32a]   :=    [[2,1,3,4,5,6,7],[3,2,1,4,5,6,7],[1,2,3,5,4,6,7],
                   [1,2,3,6,5,4,7]]:

`perm/tab_gen`[C6b]    :=    [[2,3,4,5,6,1,7]]:

`perm/tab_gen`[S3d]    :=    [[4,6,5,1,3,2,7],[5,4,6,2,1,3,7]]:

`perm/tab_gen`[D6b]    :=    [[2,3,4,5,6,1,7],[1,6,5,4,3,2,7]]:

`perm/tab_gen`[A4b]    :=    [[2,3,1,5,6,4,7],[4,5,3,1,2,6,7]]:

`perm/tab_gen`[A4C2b]  :=    [[2,3,1,5,6,4,7],[4,5,3,1,2,6,7],[4,5,6,1,2,3,7]]:

`perm/tab_gen`[S4c]    :=    [[5,1,3,2,4,6,7],[2,3,1,5,6,4,7]]:

`perm/tab_gen`[S4d]    :=    [[6,5,1,3,2,4,7],[2,3,1,5,6,4,7]]:

`perm/tab_gen`[S4C2b]  :=    [[5,1,3,2,4,6,7],[2,3,1,5,6,4,7],[4,5,6,1,2,3,7]]:

`perm/tab_gen`[S3C3b]  :=    [[4,6,5,1,3,2,7],[5,4,6,2,1,3,7],[2,3,1,4,5,6,7],[1,2,3,5,6,4,7]]:

`perm/tab_gen`[S32b]   :=    [[4,6,5,1,3,2,7],[5,4,6,2,1,3,7],[2,1,3,6,5,4,7],
                   [3,2,1,5,4,6,7]]:

`perm/tab_gen`[C2S3_a] :=    [[4,5,6,1,3,2,7],[4,5,6,3,2,1,7]]:

`perm/tab_gen`[C2S3a]  :=    [[2,1,3,4,5,6,7],[3,2,1,4,5,6,7],[4,5,6,1,2,3,7]]:

`perm/tab_gen`[A5b]    :=    [[2,3,4,5,1,6,7],[4,1,5,2,6,3,7]]:

`perm/tab_gen`[S5b]    :=    [[2,3,4,5,1,6,7],[3,2,6,1,5,4,7]]:

`perm/tab_gen`[A6a]    :=    [[2,3,4,5,1,6,7],[2,3,4,1,6,5,7]]:

`perm/tab_gen`[S6a]    :=    [[2,3,4,5,1,6,7],[6,2,3,4,5,1,7]]:

# On 5 points.

`perm/tab_gen`[C6a]    :=    [[2,3,1,5,4,6,7]]:

`perm/tab_gen`[S3b]    :=    [[2,1,3,5,4,6,7],[3,2,1,5,4,6,7]]:

`perm/tab_gen`[D6a]    :=    [[2,3,1,5,4,6,7],[2,1,3,4,5,6,7]]:

`perm/tab_gen`[C5a]    :=    [[2,3,4,5,1,6,7]]:

`perm/tab_gen`[D5a]    :=    [[2,3,4,5,1,6,7],[1,5,4,3,2,6,7]]:

`perm/tab_gen`[H5a]    :=    [[2,3,4,5,1,6,7],[1,3,5,2,4,6,7]]:

`perm/tab_gen`[A5a]    :=    [[2,3,4,5,1,6,7],[2,3,1,4,5,6,7]]:

`perm/tab_gen`[S5a]    :=    [[2,3,4,5,1,6,7],[2,3,4,1,5,6,7]]:

# On less than 4 points.

`perm/tab_gen`[S4a]    :=    [[2,3,4,1,5,6,7],[2,3,1,4,5,6,7]]:

`perm/tab_gen`[A4a]    :=    [[2,3,1,4,5,6,7],[2,1,4,3,5,6,7]]:

`perm/tab_gen`[D4a]    :=    [[3,4,2,1,5,6,7],[2,1,3,4,5,6,7]]:

`perm/tab_gen`[Kb]     :=    [[2,1,4,3,5,6,7],[3,4,1,2,5,6,7]]:

`perm/tab_gen`[C4a]    :=    [[2,3,4,1,5,6,7]]:

`perm/tab_gen`[Ka]     :=    [[2,1,3,4,5,6,7],[1,2,4,3,5,6,7]]:

`perm/tab_gen`[C2b]    :=    [[2,1,4,3,5,6,7]]:

`perm/tab_gen`[S3a]    :=    [[2,1,3,4,5,6,7],[3,2,1,4,5,6,7]]:

`perm/tab_gen`[C3a]    :=    [[2,3,1,4,5,6,7]]:

`perm/tab_gen`[C2a]    :=    [[2,1,3,4,5,6,7]]:


# The next procedure can be used to generate the representatives
# of conjugacy class of Sn with a table containing generators (like
# `perm/tab_gen`). If each element T[gen] is a list of generator,
# then gentabsg(T,n) return a table whose index gen contain the
# set of permutation of the sungroup <gen> (in Sn).
# This procedure can be used to generate once for all the list of subgroup.
`perm/gentabsg` :=proc (T,n)
   local tabPerms;
   map(proc(gen,tabPerm,T,n)
            tabPerm[op(gen)]:=perm[gensg](T[op(gen)],n)
            end,
        [indices(T)],tabPerms,T,n);
   RETURN(op(tabPerms));
end:


# ----------- end of proc definitions --------------------


# ----------- definition of "perm" table ----------------

perm:='perm':
perm['inv']:=eval(`perm/inv`):
perm['prod']:=eval(`perm/prod`):
perm['conj']:=eval(`perm/conj`):
perm['sgn']:=eval(`perm/sgn`):
perm['nextperm']:=eval(`perm/nextperm`):
perm['Id']:=eval(`perm/Id`):
perm['grcyc']:=eval(`perm/grcyc`):
perm['dihedral']:=eval(`perm/dihedral`):
perm['cyc']:=eval(`perm/cyc`):
perm['minlex']:=eval(`perm/minlex`):
perm['cllat']:=eval(`perm/cllat`):
perm['pmin']:=eval(`perm/pmin`):
perm['orbits']:=eval(`perm/orbits`):
perm['inversions']:=eval(`perm/inversions`):
perm['desc']:=eval(`perm/desc`):

perm['decomp']:=eval(`perm/decomp`):
perm['ctype']:=eval(`perm/ctype`):
perm['complete']:=eval(`perm/complete`):
perm['gensg']:=eval(`perm/gensg`):
perm['action']:=eval(`perm/action`):
perm['ulset']:=eval(`perm/ulset`):
perm['ulcard']:=eval(`perm/ulcard`):
perm['genstruct']:=eval(`perm/genstruct`):
perm['regroup']:=eval(`perm/regroup`):
perm['posmin']:=eval(`perm/posmin`):
perm['makestruct']:=eval(`perm/makestruct`):
perm['C']:=eval(`perm/C`):
perm['P']:=eval(`perm/P`):
perm['stab']:=eval(`perm/stab`):
perm['stabit']:=eval(`perm/stabit`):

perm['grconj']:=eval(`perm/grconj`):

perm['`&X`']:=eval(`perm/&X`):
perm['supp']:=eval(`perm/supp`):

perm['gentabsg']:=eval(`perm/gentabsg`):
perm['tab_gen']:=eval(`perm/tab_gen`):

# save `perm.m`;
# quit;
