## <SHAREFILE=analysis/calcvar/calcvar.mpl >
## <DESCRIBE>
## A Maple package for the calculus of variations.
##                The present package is intended to help the user solve
##                certain classes of problems from the calculus of variations,
##                and in particular is not intended as a `black box' for
##                solving problems for the user.  The package implements some
##                necessary conditions, but most of the interest for this
##                context is from the partial implementation of some
##                sufficient conditions.
##                AUTHOR: Robert M. Corless, rcorless@uwo.ca
## SEE ALSO: calcvar1.ms calcvar2.ms calcvar.tex
## </DESCRIBE>
## <UPDATE=R4 >

# Calculus of Variations Package

# (c) Copyright Nov 30, 1993 by Robert M. Corless

# Maintenance History
#    Feb 28, 1994
#       --- finished help files.
#
#    Dec 10, 1993
#       --- added algebraic conjugate-point finder

# This package is a complete, end-to-end rewriting of
# the package `Variations' by Yanan Wu.  Each subroutine
# has been re-written from scratch, some routines have been
# deleted, and new facilities (such as testing for convexity)
# have been added.

# The reason for this re-writing was to extend the functionality
# from scalar problems to vector problems, and to add convexity
# and the Royal Road of Caratheodory to the tests. It is not
# intended as a criticism of Yanan Wu's work, and indeed his 
# program can still do things that this package cannot.

#                                    Robert M. Corless, 1993
calcvar[Euler_Lagrange] := proc(f,t::name,xoft)
  local xlist,eq,locf,shrt,x,xp,i,n,nconsts,lvars,origvars; 

  if assigned(K) and not type(K,string) then 
    ERROR(`The global variable K has a value,\nand this routine uses it as a generic constant.`); 
  fi;
   
  # Convert x(t) to [x(t)] for consistency with the 
  # parametric case, where the unknowns are [x(t),y(t),...,z(t)].
  
  if type(xoft,list(anyfunc(identical(t)))) then    
    n := nops(xoft); 
    xlist := xoft;
  elif type(xoft,anyfunc(identical(t))) then
    n := 1;
    xlist := [xoft];
  else
    ERROR(`Unrecognized type for unknown`,xoft)
  fi;
  
  # Set up the dummy shorthand variables for use with unapply.
    
  x := array(1..n);
  xp:= array(1..n);  
   
  # shrt is a list of the equivalences of the shorthand variables
  # with the original variables.
  
  shrt := NULL;
  
  # Some of the Euler-Lagrange equations may return with arbitrary
  # constants in them.  The variable nconsts counts the number of
  # such constants already used, K[1], K[2], ... , K[nconsts].
  
  nconsts := 0;
  
  # Convert the expression f to an operator, locf

  locf := f;
  for i to n do
    locf := subs(diff(xlist[i],t)=xp[i],xlist[i]=x[i],locf);
    shrt := shrt,x[i]=xlist[i],xp[i]=diff(xlist[i],t);
  od:
  lvars := t,seq(x[i],i=1..n),seq(xp[i],i=1..n);
  origvars := op(subs(shrt,[lvars]));
  locf := unapply(locf,lvars);
  
  # Now compute the Euler-Lagrange equations.

  eq := {}:
  for i to n do

     # This is the 2nd order Euler-Lagrange equation, in one line.    
 
     {D[1+i](locf)(origvars) - diff( D[1+i+n](locf)(origvars), t)};
     eq := eq union ";

     # Note that the 2nd order equation will in fact be an algebraic
     # equation, not a differential equation, if the integrand is 
     # independent of diff(x[i],t)---however, that fact will appear
     # automatically (the person or program solving the equations
     # will have to take note of this fact). 

     # If the integrand is independent of x[i], then we can find
     # a first integral.  Here we compute it and add it to the set.

     if (not has(locf(lvars),x[i]))  then
       nconsts := nconsts+1;
       {D[1+n+i](locf)(origvars) = K[nconsts]};
       eq := eq union ";
     fi;
  od;
  # If the integrand is independent of t, then again we can find a 
  # first integral.  However, this first integral is often trivial if
  # n > 1, since F = x'F_x' + y'F_y' if F is a homogeneous function
  # (which it must be if the integral is to depend on the curve and
  # not on the parameterization).
  #
  # This first integral may be nontrivial for some Hamiltonian
  # problems, though, so include it in the list anyway.
  #     
  if not has(locf(lvars),t) then
    nconsts := nconsts+1;
    locf(origvars);
    for i to n do
      " - diff(xlist[i],t)*D[1+n+i](locf)(origvars);
    od;
    eq := eq union {" = K[nconsts]};
  fi;

  # Return the set of (no doubt algebraically dependent) Euler-Lagrange
  # equations.

  eq
end:


calcvar[Weierstrass] := proc(f,t::name,xoft,p::name)
  local Lp,pvars,xlist,locf,shrt,x,xp,i,n,lvars;
   
  # Convert x(t) to [x(t)] for consistency with the 
  # parametric case, where the unknowns are [x(t),y(t),...,z(t)].
  
  if type(xoft,list(anyfunc(identical(t)))) then    
    n := nops(xoft); 
    xlist := xoft;
  elif type(xoft,anyfunc(identical(t))) then
    n := 1;
    xlist := [xoft];
  else
    ERROR(`Unrecognized type for unknown`,xoft)
  fi;
  
  # Set up the dummy shorthand variables for use with unapply.
    
  x := array(1..n);
  xp:= array(1..n);  
  p := array(1..n);
   
  # shrt is a list of the equivalences of the shorthand variables
  # with the original variables.
  
  shrt := NULL;
  
  # Convert the expression f to an operator, locf

  locf := f;
  for i to n do
    locf := subs(diff(xlist[i],t)=xp[i],xlist[i]=x[i],locf);
    shrt := shrt,x[i]=xlist[i],xp[i]=diff(xlist[i],t);
  od:
  lvars := t,seq(x[i],i=1..n),seq(xp[i],i=1..n);
  pvars := t,seq(xlist[i],i=1..n),seq(p[i],i=1..n);
  locf := unapply(locf,lvars);

  # Tell Maple that each p is assumed real.
  # This hack should be replaced as soon as possible.

  for i to n do signum(p[i]^2) := 1; od;
  
  # Now compute the Weierstrass excess function

  Lp:= array(1..n);

  for i to n do
    Lp[i] := D[1+i+n](locf)(pvars)
  od;

  f - locf(pvars) 
        - linalg[dotprod](linalg[matadd](convert(map(diff,xlist,t),vector),p,1,-1),Lp);
end:


calcvar[Convex] := proc(f,t::name,xoft)
  local H,convex,xlist,locf,shrt,x,xp,i,n,lvars;
   
  # Convert x(t) to [x(t)] for consistency with the 
  # parametric case, where the unknowns are [x(t),y(t),...,z(t)].
  
  if type(xoft,list(anyfunc(identical(t)))) then    
    n := nops(xoft); 
    xlist := xoft;
  elif type(xoft,anyfunc(identical(t))) then
    n := 1;
    xlist := [xoft];
  else
    ERROR(`Unrecognized type for unknown`,xoft)
  fi;
  
  # Set up the dummy shorthand variables for use with unapply.
    
  x := array(1..n);
  xp:= array(1..n);  
   
  # shrt is a list of the equivalences of the shorthand variables
  # with the original variables.
  
  shrt := NULL;
  
  # Convert the expression f to an operator, locf

  locf := f;
  for i to n do
    locf := subs(diff(xlist[i],t)=xp[i],xlist[i]=x[i],locf);
    shrt := shrt,x[i]=xlist[i],xp[i]=diff(xlist[i],t);
  od:
  lvars := t,seq(x[i],i=1..n),seq(xp[i],i=1..n);
  locf := unapply(locf,lvars);
  
  # Now compute the Hessian matrix:

  H := linalg[hessian](locf(lvars),[seq(x[i],i=1..n),seq(xp[i],i=1..n)]);


  # If an error occurs, simply return NULL for the convexity 
  # conditions.  (There is a bug in Release 2 linalg[definite]).

  convex := traperror(linalg[definite](H,'positive_semidef'));
  if convex=lasterror then 
    convex := NULL
  else
    convex := subs(shrt,convex)
  fi;

  if convex <> NULL then
    if type(convex,`and`) then
      traperror(map(e->map(Normalizer,e),convex));
      if "<>lasterror then convex := " fi 
    elif type(convex,`<=`) then
      traperror(map(Normalizer,convex));
      if "<>lasterror then convex := " fi
    else
      traperror(Normalizer(convex));
      if "<>lasterror then convex := " fi
    fi;
  fi;
  # Return an expression sequence: the matrix and the conditions
  # for a minimum.  (Of course, if an error in linalg[definite] 
  # occurred, then the following will just return H).

  subs(shrt,eval(H)),convex
end:

calcvar[Jacobi] := proc(f,t::name,xoft,extremal,h::name,a)
  local soln,Php,Jeq,P,Q,Fyyp,exlist,xlist,locf,shrt,x,xp,i,j,n,lvars;
   
  # Convert x(t) to [x(t)] for consistency with the 
  # parametric case, where the unknowns are [x(t),y(t),...,z(t)].
  
  if type(xoft,list(anyfunc(identical(t)))) then    
    n := nops(xoft); 
    xlist := xoft;
    exlist:= extremal;   # No error checking for extremal
  elif type(xoft,anyfunc(identical(t))) then
    n := 1;
    xlist := [xoft];
    exlist:= [extremal]; # No error checking for extremal
  else
    ERROR(`Unrecognized type for unknown`,xoft)
  fi;
  
  # Set up the dummy shorthand variables for use with unapply.
    
  x := array(1..n);
  xp:= array(1..n);  
   
  # shrt is a list of the equivalences of the shorthand variables
  # with the original variables.
  
  shrt := NULL;
  
  # Convert the expression f to an operator, locf

  locf := f;
  for i to n do
    locf := subs(diff(xlist[i],t)=xp[i],xlist[i]=x[i],locf);
    shrt := shrt,x[i]=exlist[i],xp[i]=diff(exlist[i],t);
  od:
  lvars := t,seq(x[i],i=1..n),seq(xp[i],i=1..n);
  locf := unapply(locf,lvars);

  P := linalg[hessian](locf(lvars),[seq(xp[i],i=1..n)]);
  Q := linalg[hessian](locf(lvars),[seq(x[i],i=1..n)]);
  Fyyp := linalg[jacobian](
            linalg[grad](locf(lvars),[seq(x[i],i=1..n)]),
                             [seq(xp[i],i=1..n)]);
  P := subs(shrt,eval(P));
  Q := evalm(subs(shrt,eval(Q)) - map(diff,subs(shrt,eval(Fyyp)),t) );

  if n=1 then
   Jeq := -diff(P[1,1]*diff(h(t),t),t) + Q[1,1]*h(t);
   soln := traperror(dsolve({Jeq,h(a)=0},h(t)));
  else
   Php := array(1..n);
   for i to n do
     Php[i] := 0;
     for j to n do
       Php[i] := Php[i] + P[i,j]*diff(h.j(t),t);
     od;
   od;
   Jeq := array(1..n);
   for i to n  do
     Jeq[i] := -diff(Php[i],t);
     for j to n do
       Jeq[i] := Jeq[i] + Q[i,j]*h.j(t);
     od;
    od;
    soln := traperror(dsolve({seq(Jeq[i],i=1..n),seq(h.i(a)=0,i=1..n)},{seq(h.i(t),i=1..n)}));
   fi;
  Jeq,soln;
end:
calcvar[Conjugate_eqn] := proc(
  extremals,parms::list(name),particular_parms::list,
  t::name,t0) local lex,J,n,`2n`,i,j;
  
  if type(extremals,algebraic) then
    lex := [extremals];
    n   := 1;
  elif type(extremals,list(algebraic)) 
    or type(extremals,set(algebraic)) then
    lex := extremals;
    n   := nops(lex);
  else
    ERROR(`Unknown type of extremals`,extremals);
  fi;
 
  `2n` := nops(parms);
  if 2*n <> `2n` then
    ERROR(`2n-parameter family of extremals needed`,`2n`,`<>`,`2*`,n);
  fi;

  J := array(1..`2n`,1..`2n`);

  for i to n do
    for j to `2n` do
      J[2*i-1,j] := diff(lex[i],parms[j]);
      J[2*i,j]   := subs(t=t0,J[2*i-1,j]);
    od;
  od;

  subs(seq(parms[j]=particular_parms[j],j=1..`2n`),eval(J));
  linalg[det](");
end:

calcvar[depends] := proc(x::{string,list(string),set(string)},t::name) local u;
  if type(x,string) then
    macro(`dd`.x=diff(x(t),t,t),`d`.x=diff(x(t),t));
    alias(x.`''`=diff(x(t),t,t),x.`'`=diff(x(t),t),x=x(t));
  else
    for u in x do depends(u,t) od;
  fi;
end:

calcvar[free] := proc(x,t::name) local u,a;
  if type(x,anyfunc(identical(t))) then
    alias(op(0,x)=op(0,x));
    a := {alias()};
    for u in a do
      if substring(u,1..1)=op(0,x) and substring(u,2..2)=`'` then
        alias(u=u)
      fi
    od;
    macro(`dd`.(op(0,x))=`dd`.(op(0,x)),`d`.(op(0,x))=`d`.(op(0,x)));
  else
    for u in x do free(u,t) od;
  fi;
alias();
end:

#save calcvar,`calcvar.m`;
#quit
