#
## <SHAREFILE=analysis/tzsolve/tzsolve.mpl >
## <DESCRIBE>
##        A PDE solver for hyperbolic and parabolic 1D problems
##                AUTHOR: Antonio Almeida, aalmeida@lemac18.lemac.ist.utl.pt
## </DESCRIBE>
## <UPDATE=R4 >

########## Release 3 version ##################################################

################################################################################
#                                                                              #
#    Description: PDE solver  solver for hyperbolic and parabolic 1D problems  #
#    Module: tzsolveR3.p    (for MapleV R3)                                    #
#    Author: Antonio Almeida                                                   #
#    Creation Date: Dec 13 1994                                                #
#    Last Revision Date: Dec 13 1994                                           #
#    Revised by: Antonio Almeida                                               #
#    Author Adress: IST, Lisbon                                                #
#    Revisor Adress: IST, Lisbon                                               #
#    Author e-mail adress: aalmeida@lemac18.lemac.ist.utl.pt                   #
#    Revisor e-mail adress: aalmeida@lemac18.lemac.ist.utl.pt                  #
#    Copyright by: A.Almeida                                                   #
#                                                                              #
################################################################################




##########################
# standard input
###
# Convert from 5.2 to Release 3

# Global variable that takes the values true or false.
# If true then the solver returns a table with the results.
# In this case the convergence rate can be computed.  

TestPDESolver:=false:

# Global variable that takes the value true or false.
# If set to true then the results are printed and a 
# PLOT3D structure of the numerical solution is displayed.

print_and_plot_PDE:=true:


# This procedure computes the Stiffness matrix and load vector for arbitrary
# basis functions (1D problems) for a second order elliptic differential 
# equation of the form -k*diff(u(x),x$2)+c*u(x)=f(x).
# Note: a is the coefficient of the time derivative  

ComputeKeFeMe:=proc(BasisFunctions::table,a::constant,k::constant,c::constant,
                    f::algebraic)

 local i, j, n,  Ke, Me, Fe, J, Te:

 n:=nops([entries(BasisFunctions)]):# getting the element d.o.f
 Ke:=array(symmetric,1..n,1..n):    # declaring the elementary stiffness matrix
 Me:=array(symmetric,1..n,1..n):    # declaring the elementary mass matrix
 Fe:=array(1..n):                   # declaring the elementary load vector

 Te:=sum('x.i'*BasisFunctions[i],i=1..2):#  the map Te: x |-> x(xi)
 
 # The jacobian

 J:='h'/2:
 
 # for loop in i and j where the elementary stiffness 
 # matrix, mass matrix and load vector entries are computed 

 for i to n do
  for j to n do
   BasisFunctions[i]:
   BasisFunctions[j]:
   Ke[i,j]:=collect(J*int(k*diff("",'xi')*diff(",'xi')*(1/J)^2+c*""*",
                          'xi'=-1..1),'h'):
   Me[i,j]:=collect(a*J*int("""*"",'xi'=-1..1),'h')
  od:   
  Fe[i]:=collect(int(J*subs('x'=Te,f)*BasisFunctions[i],'xi'=-1..1),'h')
 od:

 # returning the elementary stiffness matrix and load vector
 
 RETURN(map(eval,[Ke,Fe,Me]))

 end:

# ComputeKeFeMe


# This procedure assembles the global stiffness matrix, mass matrix,
# load vector and "global" initial condition (i.e. the vector that 
# corresponds to the discretized initial condition) 

AssembleGlobal_K_F_M:=proc(Ke::array,Fe::array,Me::array,Nelem::integer,
                           Delta::numeric,left_point::numeric,
                           InitialCond::operator,InitialVel::anything)

 local i, j, e, Ne, Nnodes, H, GlobalK, GlobalF, GlobalM,
       GlobalInit, GlobalInitVel: 	 

 Ne:=linalg[vectdim](Fe):# getting the element number of nodes

 # Getting the number of nodes 

 Nnodes:=(Ne-1)*Nelem+1:

 H:=evalhf(Delta*(Ne-1)/(Nnodes-1)):

 # declaring the global stiffness matrix  

 GlobalK:=array(sparse,1..Nnodes,1..Nnodes):

 # declaring the global mass matrix
  
 GlobalM:=array(sparse,1..Nnodes,1..Nnodes):

 # declaring the global load vector  

 GlobalF:=array(sparse,1..Nnodes):

 # declaring the initial condition vector

 GlobalInit:=array(sparse,1..Nnodes):

 # declaring the initial velocity vector

 GlobalInitVel:=array(sparse,1..Nnodes):

 # assembling the global stiffness matrix and load vector

 if Ne=2 then # Assembling the Global stiffness matrix for linear elements 

  for e to Nelem do # e is the element number (from 1 to Nelem)  
   for i from 0 to Ne-1 do  # i and j are the loop variables in e-th element
    for j from 0 to Ne-1 do
     GlobalK[e+i,e+j]:=GlobalK[e+i,e+j]+Ke[i+1,j+1]:
     GlobalM[e+i,e+j]:=GlobalM[e+i,e+j]+Me[i+1,j+1]
    od:
    GlobalF[e+i]:=GlobalF[e+i]+
                  subs({'x1'=left_point+(e-1)*H,'x2'=left_point+e*H},Fe[i+1])
   od 
  od

 else # Assembling the Global stiffness matrix for quadratic elements 

  for e to Nelem do # e is the element number (from 1 to Nelem)  
   for i from 0 to Ne-1 do  # i and j are the loop variables in e-th element
    for j from 0 to Ne-1 do
     GlobalK[2*e-1+i,2*e-1+j]:=GlobalK[2*e-1+i,2*e-1+j]+Ke[i+1,j+1]:    
     GlobalM[2*e-1+i,2*e-1+j]:=GlobalM[2*e-1+i,2*e-1+j]+Me[i+1,j+1]
    od:
    GlobalF[2*e-1+i]:=GlobalF[2*e-1+i]+
                      subs({'x1'=left_point+(e-1)*H,'x2'=left_point+e*H},Fe[i+1]):
   od 
  od
 
 fi:

 # Filling the "global" initial condition data 

 for i to Nnodes do
  GlobalInit[i]:=evalhf(InitialCond(left_point+(i-1)*H/(Ne-1))) 
 od:

 # Filling the "global" initial velocity data 

 for i to Nnodes while type(InitialVel,'operator') do
  GlobalInitVel[i]:=evalhf(InitialVel(left_point+(i-1)*H/(Ne-1))) 
 od: 

 # replacing the value h in the elementary stiffness 
 # matrix and load vector

 (x,y)-> subs('h'=y,x) :

 map(",GlobalM,H):

 map("",GlobalF,H):

 map(""",GlobalK,H):

 # returnig the assembled stiffness matrix and load vector as result

 if not type(GlobalInitVel,'array') then 
    RETURN(map(eval,[","",""",GlobalInit])) # for the heat equation
 else   # for the d'Alembert equation
    RETURN(map(eval,[","",""",GlobalInit,GlobalInitVel])) 
 fi

 end: 

# AssembleGlobal_K_F_M


# This procedure imposes the boundary condition (Neumann or Dirichlet type)
# modifying the stifness matrix and/or the load vector 

ImposeBC:=proc(BoundCond::set,k::algebraic,K::array,F::array,left_point::numeric)
 
 local i, j, GlobalF, GlobalK, n:

 GlobalF:=copy(F):
 
 GlobalK:=copy(K):

 n:=linalg[vectdim](F):

 for i to nops(BoundCond) do

  # If it is a Neumann type boundary condition then modify the load vector 

  if has(BoundCond[i],'D') then
   if op(1,lhs(BoundCond[i]))=left_point then # left point
    GlobalF[1]:=GlobalF[1]-k*rhs(BoundCond[i])
   else
    GlobalF[n]:=GlobalF[n]+k*rhs(BoundCond[i]) # right point
   fi  
  else # otherwise is Dirichlet type boundary condition and uses the penalty 
       # method 
   if op(1,lhs(BoundCond[i]))=left_point then # left point
    # Modifying the stiffness matrix 
    GlobalK[1,1]:=1:
    for j from 2 to n do 
     GlobalK[1,j]:=0:
     GlobalK[j,1]:=0	
    od:
    # Modifying the load vector
    GlobalF[1]:=rhs(BoundCond[i])
   else # right point
    # Modifying the stiffness matrix 
    GlobalK[n,n]:=1:
    for j to n-1 do 
     GlobalK[n,j]:=0:
     GlobalK[j,n]:=0 
    od:
    # Modifying the load vector
    GlobalF[n]:=rhs(BoundCond[i]) 
   fi
  fi
 od:
  
 RETURN(map(eval,[GlobalK,GlobalF]))

 end:

# ImposeBC


# This procedure "lumps" the mass(capacitance) matrix, i.e.,
# diagonalises the matrix (sum each row and assign it to the diagonal term)


LumpMatrix:=proc(Matrix::array)

 local i, n, LumpedMatrix;

 n:=linalg[rowdim](Matrix):   
 LumpedMatrix:=array('diagonal',1..n,1..n):

 for i to n do
  LumpedMatrix[i,i]:=sum(Matrix['j',i],'j'=1..n)
 od:
  
 RETURN(eval(LumpedMatrix))

 end:

# LumpMatrix


# This procedure solves hyperbolic and/or parabolic PDE with constant 
# coefficients (heat equation and/or d'Alembert's equation), i.e., 
# equations of the following type:
# A*diff(u(x,t),t)-B*diff(u(x,t),x$2)+C*u(x,t)=0 or
# A*diff(u(x,t),t$2)-B*diff(u(x,t),x$2)+C*u(x,t)=0.
# A trapezoidal method is used in both type of equations. A plot of
# the analytical solution (using surfdata) is given.

tzsolve:=proc(diffeq::set(equation),unknown::function,N::integer, 
	      M::integer,Final_t::numeric,
              ElementType::integer,
	      theta::numeric)
 

 local i, A, B, C, bcond, leftmost_point, rightmost_point, dequation,  
       Kg, Fg, Mg, f, init_cond, init_vel, d, v, a, Delta_t,
       temp, bf1, bf2, Energy, Predictor_d, Predictor_v, indyVar1, indyVar2:

 # otherwise proceed normally 

 # Basis Functions (Lagrange polynomials in xi)
 
 # linear

 bf1:=table([(1)=(1-'xi')/2,(2)=(1+'xi')/2]):

 # quadratic

 bf2:=table([(1)=-'xi'*(1-'xi')/2,(2)=(1-'xi'^2),(3)='xi'*(1+'xi')/2]):

 # getting the independent variables 

 indyVar1:=op(1,unknown):

 indyVar2:=op(2,unknown):

 # getting the differential equation

 for i while not has(diffeq[i],diff) do od;  

 dequation:=diffeq[i]:

 # getting the boundary/initial conditions

 bcond:=diffeq minus {dequation}:

 # getting the initial velocity (for the d'Alembert equation)
 
 for i while not has(bcond[i],'D') and nops(bcond)=4 do od;

 if i>1 then init_vel:=bcond[i] fi:

 # removing the initial velocity from the bcond set

 if type(init_vel,'equation') then bcond:=bcond minus {init_vel} fi:

 # getting the initial condition

 for i while not op(1,op(1,bcond[i]))=indyVar1 do od;
  
 init_cond:=bcond[i]:

 # removing the initial condition from the bcond set

 bcond:=bcond minus {init_cond}:

 # Checking if the initial condition is given as a operator (in order to be 
 # possible to have if's or as an algebraic expression. In that case convert 
 # to an operator 

 if not type(rhs(init_cond),'operator') then
  init_cond:=unapply(rhs(init_cond),indyVar1) 
 else
  init_cond:=rhs(init_cond)
 fi:

 # for the wave equation one has also the same for the initial
 # velocity 

 if type(init_vel,'equation') then  
  if not type(rhs(init_vel),'operator') then
   if type(rhs(init_vel),'constant') then
    proc() options operator, arrow; RETURN(value) end:
    init_vel:=subs('value'=rhs(init_vel),"):
   else
    init_vel:=unapply(rhs(init_vel),indyVar1)
   fi 
  else
   init_vel:=rhs(init_vel)
  fi
 fi:

 # getting the interval extreme points

 leftmost_point:=min(op(map(x->(op(1,op(1,x))),bcond))):
 rightmost_point:=max(op(map(x->(op(1,op(1,x))),bcond))):

 # getting the A, B, C and f from the given differential equation

 if nops(diffeq)=4 then  # for a parabolic equation
  A:=coeff(lhs(dequation),diff(unknown,indyVar2))
 else  # for a hyperbolic equation
  A:=coeff(lhs(dequation),diff(unknown,indyVar2$2))
 fi:
 
 B:=-coeff(lhs(dequation),diff(unknown,indyVar1$2)): 
 C:=coeff(subs({""=0,-"=0},lhs(dequation)),unknown):
 f:=rhs(dequation):

 # invoking the procedure that computes the elementary 
 # stiffness matrix and load vector 

 if ElementType=1 then
  ComputeKeFeMe(eval(bf1),A,B,C,f) # linear elements
 elif ElementType=2 then
  ComputeKeFeMe(eval(bf2),A,B,C,f) # quadratic elements
 else # else generate an error
  ERROR(`Wrong type of elements`)  
 fi:

 # assembling the global stifness matrix, mass/capacitance matrix and load 
 # vector

 AssembleGlobal_K_F_M("[1],"[2],"[3],N,rightmost_point-leftmost_point,
		      leftmost_point,eval(init_cond),eval(init_vel)):

 temp:=": # assigning the previous result to a temporary variable

 Mg:=temp[3]:   # assigning Mg to the global mass matrix 
 d[0]:=temp[4]: # assigning InitialCondg to the "global" initial condition

 # if the init_vel is an operator then initialize the velocity
 # (d'Alembert's equation) 

 if type(init_vel,'operator') then v[0]:=temp[5] fi:

 # imposing the boundary conditions

 ImposeBC(bcond,B,temp[1],temp[2],leftmost_point):

 Kg:="[1]:   # assigning Kg to the global stiffness matrix
 Fg:=""[2]:  # assigning Fg to the global load vector  

 # Lumping the mass/capacitance matrix (diagonalizing) 

 Mg:=LumpMatrix(eval(Mg)):

 # Solving the problem in time and space

 # getting the time step

 Delta_t:=evalhf(Final_t/M):
 
 # Checking the stability condition for the explicit/implicit
 # algorithm (theta=[0,1/2) ) for both parabolic (heat equation) and
 # hyperbolic (d'Alembert's equation)

 if CheckStability(theta,
                   (rightmost_point-leftmost_point)/(linalg[vectdim](Fg)-1),
                   Delta_t,ElementType) 
 then 
  ERROR(`Stability condition for finite difference method is not fullfiled`)
 fi: 

 # evolving in time (trapezoidal method)

 if nops(diffeq)=4 then # trapezoidal method for the heat equation 
  for i to M do
   d[i]:=linalg[linsolve](linalg[matadd](Mg,Kg,1,theta*Delta_t),
	                  linalg[multiply](
                                linalg[matadd](Mg,Kg,1,-(1-theta)*Delta_t),
                                d[i-1]))
  od                                    
 else # trapezoidal method for the d'Alembert equation (Hughes pag. 490) 
 
 # computing the initial total energy (Kinetic+Strain)

 Energy[0]:=linalg[dotprod](d[0],linalg[multiply](Kg,d[0]))/2 +
            linalg[dotprod](v[0],linalg[multiply](Mg,v[0]))/2:

 # computing the initial acceleration

 a[0]:=linalg[linsolve](Mg,-linalg[multiply](Kg,d[0])): 

  for i to M do
   # computing a predictor for d[i]
  
   Predictor_d:=linalg[matadd](d[i-1],v[i-1],1,Delta_t*(1-theta)):
 
   # computing a predictor for v[i]

   Predictor_v:=linalg[matadd](v[i-1],a[i-1],1,Delta_t*(1-theta)): 

   # computing the acceleration at time step i
   
   a[i]:=linalg[linsolve](linalg[matadd](Mg,Kg,1,theta^2*Delta_t^2),
                          -linalg[multiply](Kg,linalg[matadd](Predictor_d,
                                                           Predictor_v,
                                                           1,theta*Delta_t))):
   # computing the velocity at time step i
   v[i]:=linalg[matadd](Predictor_v,a[i],1,theta*Delta_t):
   # computing the displacement at time step i using the predictor
   d[i]:=linalg[matadd](Predictor_d,v[i],1,theta*Delta_t):
   # computing the energy at time step i 
   Energy[i]:=linalg[dotprod](d[i],linalg[multiply](Kg,d[i]))/2+
              linalg[dotprod](v[i],linalg[multiply](Mg,v[i]))/2
  od 
 fi: 

 # printing and ploting the numerical solution 
 # if the variable prind_and_plot_PDE is set to true
  
 if print_and_plot_PDE then
  PlotPrintSolution(eval(d),leftmost_point,rightmost_point,unknown,ElementType,
                    Delta_t,theta,eval(Energy))
 fi:

 # if the global variable TestPDEsolver is set to true then 
 # return the table d (results)

 if TestPDESolver then RETURN(eval(d)) fi
 
 end:

# tzsolve


# This procedure plots and prints the numerical solution 

PlotPrintSolution:=proc(Solution::table,left_point::numeric,right_point::numeric,
                        unknown::function,ElementType::integer,DeltaT::numeric,
                        Theta::numeric,Energy::anything)

 local i, j, H, n, m, PlotListT, PlotListX, ActualPoint,
       indyVar1, indyVar2:

 n:=linalg[vectdim](Solution[1]): # getting the number of nodes (space)
 
 m:=nops([entries(Solution)]): # getting the number of time points

 # getting the independent variables 

 indyVar1:=op(1,unknown):

 indyVar2:=op(2,unknown):

 # computing the mesh parameter (uniform mesh)

 H:=evalhf((right_point-left_point)/(n-1)):

 # initializing the plot list in "T" (time)

 PlotListT:=[]:

 # printing the results in Maple top level

 print(`Numerical Solution`):
 print(`FEM in space and a finite difference method in time`):
 
 printf(`Number of Nodes per element: %u\n`,ElementType+1): 
 printf(`Number of Elements: %u\n`,(n-1)/ElementType):
 printf(`Number of points in time: %u\n`,m):
 printf(`Theta parameter: %f\n`,Theta):
 printf(`\n`):

 for i to m do
  printf(`-----------------------------------------\n`): 
  printf(`|               %a=%10.6g            |\n`,indyVar2,(i-1)*DeltaT):
  printf(`-----------------------------------------\n`):
  printf(`       %a              |       %a   \n`,indyVar1,unknown): 
  printf(`-----------------------------------------\n`):
 
  # printing the results in space and filling the plot list for surfdata
  
  PlotListX:=[]:# initializing the plot list in "X" (space)
  for j to n do
   # computing the actual point
   ActualPoint:=[left_point+(j-1)*H,(i-1)*DeltaT,Solution[i-1][j]]:
   # filling the plot list in X
   PlotListX:=[op(PlotListX),ActualPoint]:
   printf(`| %10.6g          |       %10.6g\n`,ActualPoint[1],ActualPoint[3]):
  od: # for loop in j
  # printing the energy for a hyperbolic problem
  if type(Energy,'table')then
   printf(`+---------------------------------------+\n`): 
   printf(`| Total Energy: %10.6g Joules       |\n`,Energy[i-1]):
   printf(`+---------------------------------------+\n`) 
  fi: 
  # filling the plot list in T 
  PlotListT:=[op(PlotListT),PlotListX]
 od: # for loop in i 
 
 printf(`-----------------------------------------\n`): 

 # ploting the numerical Solution
 
 print(plots[surfdata](PlotListT,
                       title=cat(`Numerical Solution NxM=`,n,'x',m),
                       axes=FRAME,labels=[op(unknown),convert(unknown,'string')],
                       style='PATCHCONTOUR'))
 end: 

# PlotPrintSolution


# case 1: Linear elements
# case 2: Quadratic elements

CheckStability:=proc(theta::numeric,h::numeric,DeltaT::numeric,ElemType::integer)

 false; 

 # Checking the several stability conditions (see previous comment)

 if theta<evalhf(1/2) then
  evalb((ElemType=1 and DeltaT/h^2 >evalhf(1/(2*(1-2*theta)))) 
        or
        (ElemType=2 and DeltaT/h^2 >evalhf(1/(12*(1-2*theta))))) 
 fi:

 RETURN(")

 end:

# CheckStability
  
# This procedure computes the convergence data log(max(abs(u_numerical-u_ex)))

GetConvData:=proc(Solution::table,h::numeric,
                  DeltaT::numeric,left_point::numeric,
                  AnalyticalSolution::operator)  

 local n, m, i, j, MaxTemp;

 # getting the number of points in space

 n:=linalg[vectdim](Solution[1]):

 # getting the number of entries of the table Solution 
 # (number of time steps) 

 m:=nops([entries(Solution)]):

 # getting the middle point in space

 if not type(n/2,'integer') then i:=(n+1)/2 else i:=n/2 fi:

 MaxTemp:=0:
 for j from 0 to m-1 do
  abs(evalhf(AnalyticalSolution(left_point+(i-1)*h,j*DeltaT))-Solution[j][i]): 
  MaxTemp:=max(",MaxTemp)
 od:
 
 # returning the logarithm of the L-infinity norm of the error  

 RETURN(abs(log(MaxTemp)))

 end:

# GetConvergenceData   


#save `tzsolve.m`;
#quit
