#
## <SHAREFILE=science/units/units.mpl >
## <DESCRIBE>
##   This file contains a a list of over two dozen common
##                physical constants.  It is based on E. R. Cohen and
##                B. N. Taylor, Rev Mod Phys 59, 1121 (1987).
##                Converts expressions automatically to MKSA (SI) units.
##                SEE ALSO: "Theoretical Methods in the Physical Sciences,
##                an Introduction to Problem-Solving with Maple", W. E. Baylis,
##                Birkhauser Boston, 1994.
##                AUTHOR: William. E. Baylis, baylis@uwindsor.ca
## SEE ALSO: units.ms
## </DESCRIBE>
## <UPDATE=R4 >

# LIST OF UNITS AND CONSTANTS
#
# Converts expressions automatically to MKSA (SI) units. Contains over
# two dozen common physical constants. More can be easily added.
# Prepared by W. E. Baylis for use in his text "Theoretical Methods in the
#     Physical Sciences, an Introduction to Problem-Solving with Maple",
#     Birkhauser Boston, about July 1994.
# Based on E. R. Cohen and B. N. Taylor, Rev Mod Phys 59, 1121 (1987).
#
# W. E. Baylis, Dept. Physics, U. Windsor, Windsor, Ont., Canada N9B 3P4.
# Internet Address: baylis@uwindsor.ca
#

units := `_units_`:

#  Metric prefixes
Exa    := 10^18    :
Peta   := 10^15    :
Tera   := 10^12    :
Giga   := 10^9     :
Mega   := 10^6     :
kilo   := 10^3     :
milli  := 10^(-3)  :
micro  := 10^(-6)  :
nano   := 10^(-9)  :
pico   := 10^(-12) :
femto  := 10^(-15) :
atto   := 10^(-18) :
#
#Common conversions
deg_   := Pi*rad_/180     :#angular measure
cycle_ := 2*Pi*rad_       :# relation between cycle and radian
rad_   := 1               :# radian (caution: some authors use cycle_=1)
Hz_    := cycle_/s_       :# Hertz (frequency)
hr_    := 60*min_         :
min_   := 60*s_           :
day_   := 24*hr_          :
yr_    := 365.24219*day_  :#tropical year
mm_    := milli*m_        :
cm_    := 0.01*m_         :
km_    := kilo*m_         :
in_    := 2.54*cm_        :
ft_    := 12*in_          :
yd_    := 3*ft_           :
mi_    := 5280*ft_        :#statute mile
ly_    := 94605282*Giga*m_:#light-year
pc_    := 3.261633*ly_    :#parsec
hectare_ := (100*m_)^2    :# 100 ares, unit of area
acre_  := 0.40468564*hectare_:#acre, 160 sq.rods
l_     := m_^3/1000       :# liter
L_     := l_              :# alternative liter symbol
ml_    := milli*l_        :
cc_    := ml_             :# cubic centimeter
gal_   := 3.785412*l_     :# gallon (US, liquid)
gal_imp:= 4.54609*l_      :# Imperial gallon
qt_    := gal_/4          :# quart(US, liquid)
pt_    := qt_/2           :# pint(US, liquid)
g_     := kg_/1000        :# gram (cgs unit of mass)
Tonne_ := 1000*kg_        :# metric ton
lb_    := 0.453592*kg_    :# pound mass (do not confuse with lb_force)
ton_   := 2000*lb_        :# short ton
oz_    := lb_/16          :# ounce (US, fluid)
N_     := kg_*m_/s_^2     :# Newton (metric unit of force)
dyne_  := g_*cm_/s_^2     :# cgs unit of force
lb_force := 4.44822*N_    :# caution: lb is used both for force and mass
P_     := g_/(cm_*s_)     :# poise, unit of viscosity
J_     := N_*m_           :# Joule (metric unit of energy)
erg_   := dyne_*cm_       :# cgs unit of energy
cal_   := 4.1868*J_       :# calorie (cgs heat unit)
Cal_   := kilo*cal_       :# dieticians calorie
Btu_   := 1055.056*J_     :# British thermal unit
eV_    := 1.60217733*10^(-19)*J_:# electon Volt (energy)
Hartree_:=27.21139613*eV_ :# atomic unit of energy = 2 Rydbergs of energy
W_     := J_/s_           :# Watt
hp_    := 745.7*W_        :# horsepower
Pa_    := N_/m_^2         :# Pascal, the metric unit of pressure
atm_   := 101325*Pa_      :# atmosphere = 29.9213 in_ of mercury
bar_   := 100000*Pa_      :# bar of pressure
torr_  := atm_/760        :# mm_ height of mercury
V_     := J_/C_           :# metric unit of Volt
StatV_ := 299.792458*V_   :# cgs (esu) unit of potential
C_     := A_*s_           :# Coulomb (metric unit of electric charge)
StatC_ := C_/2997924580   :# cgs (esu) unit of electric charge
StatA_ := A_/2997924580   :# cgs (esu) unit of current
F_     := C_/V_           :# Farad (metric unit of capacitance)
Wb_    := V_*s_           :# Weber (metric unit of magnetic flux)
H_     := V_*s_/A_        :# Henry (metric unit of inductance)
ohm_   := V_/A_           :# metric unit of resistance
T_     := Wb_/m_^2        :# Tesla (metric unit of magnetic field B)
gauss_ := T_/10000        :# cgs (emu) unit of magnetic induction B
Oe_    := gauss_          :# oersted, cgs (emu) unit of magnetic field H
#
#  Physical Constants
mol_   := 6.0221367*10^(-23)    :#Avogadro constant, 1 mole
alpha_ := 1/137.036             :#fine-structure constant
c_     := 299792458*m_/s_       :#speed of light (exact)
h_     := 6.6260755*10^(-34)*J_*s_:#Planck constant
hbar_  := 1.05457266*10^(-34)*J_*s_:#Planck constant/2 Pi
e_     := 1.60217733*10^(-19)*C_:#elementary charge
Ryd_inf:= 10973731.57/m_        :#Rydberg constant for infinite mass
mu_B   := 9.2740154*10^(-24)*J_/T_:#Bohr magneton
mu_0   := 4*Pi*10^(-7)*N_/A_^2  :#permeability of vacuum
eps_0  := 1/(mu_0*c_^2)         :#permittivity of vacuum
G_     := 6.6726*10^(-11)*m_^3/(kg_*s_^2):#Newtonian gravitational constant
k_     := 8.31451*J_/(K_*mol_)  :#Boltzmann constant
m_e    := 9.1093897*10^(-31)*kg_:#electron mass
m_p    := 1836.1527*m_e         :#proton mass
m_n    := 1838.68366*m_e        :#neutron mass
u_     := 1.66054*10^(-27)*kg_  :#unified atomic mass unit (amu) M(C12)/12
m_sun  := 1.989*10^30*kg_       :#mass of sun
m_earth:= 5.98*10^24*kg_        :#mass of earth
m_moon := 7.34*10^22*kg_        :#mass of moon
bohr_  := 0.52917725*10^(-10)*m_:#bohr radius a0 (atomic unit of length)
R_moon := 1.74*Mega*m_          :#average radius of moon
R_earth:= 6.37*Mega*m_          :#average radius of earth
R_sun  := 696*Mega*m_           :#average radius of sun
r_earth:= 149.59787*Giga*m_     :#average radius of earth's orbit about sun
AU_    := r_earth               :#Astronomical Unit
r_moon := 384.0*Mega*m_         :#average radius of moon's orbit about earth
g_acc  := 9.80665*m_/s_^2       :#acceleration of gravity on earth's surface
