#
## <SHAREFILE=combinat/posets/posets.mpl >
## <DESCRIBE>
##                SEE ALSO: combinat/posets.tex
##                A package of routines for manipulating partially ordered sets
##                including routines to enumerate all non-isomorphic Posets and
##                Lattices on n vertices.
##                AUTHOR: John Stembridge, jrs@math.lsa.umich.edu
## </DESCRIBE>

# COPYLEFT NOTICE:
# Copyleft (c) 1991 by John R. Stembridge.
#  
# Permission is granted to anyone to to use, modify, or redistribute this
# software freely, subject to the following restrictions:
# 
# 1. The author accepts no responsibility for any consequences of this
# software and makes no guarantee that the software is free of defects.
# 2. The origin of this software must not be misrepresented, either by
# explicit claim or by omission.
# 3. This notice and the copyleft must be included in all copies or
# modified versions of this software.
# 4. This software may not be included or redistributed as part of any
# package to be sold for profit unless the author has given explicit written
# permission to do so.
# 
# John Stembridge
# Department of Mathematics
# University of Michigan
# Ann Arbor, MI 48109-1003
# Internet:  jrs@math.lsa.umich.edu
#
###############################################################################
#
#  J(P,X) returns the poset of order ideals of the poset P with vertex set X.
#    If X=an integer n, then vertex set={1,...,n}. The poset returned will be
#    naturally labeled. If the second argument can be omitted if there are
#    no isolated points.
#
`posets/J`:=proc() local P,n,i,j,X,anti,DD,d,res,k;
  if nargs=1 then
    X:=map(op,args[1])
  elif type(args[2],'integer') then
    X:={$1..args[2]} else X:=args[2]
  fi;
  P:=posets['closure'](args[1]);
  anti:=table(posets['antichains'](P,X,0));
  n:=nops([indices(anti)]); res:=NULL;
  for i to n do;
    anti[i]:= anti[i] union
      map(proc(x,y) if member(x[2],y) then x[1] fi end,P,anti[i]);
    for j to i-1 do;
      d:=nops(anti[i])-nops(anti[j]);
      if d=1 then
        DD:=anti[i] minus anti[j];
        if nops(DD)=1 then res:=res,[j,i] fi;
      elif d=-1 then
        DD:=anti[j] minus anti[i];
        if nops(DD)=1 then res:=res,[i,j] fi;
      fi;
    od;
  od;
  X:=map(op,posets['filter']({res}));
  subs({seq(X[k]=k,k=1..n)},{res});
end:

#
# Lattices(n) returns a list of all nonisomorphic lattices on n vertices.
#  NOTE: The lattices in this list are represented by their COVERING relations;
#  i.e., a set of ordered pairs [i,j] with no instance of [i,k] and [k,j] for
#  any k. The posets are naturally labeled; i.e., every relation [i,j]
#  satisfies i<j. To obtain the actual transitive relation, use closure().
#
`posets/Lattices`:=proc(n) option remember; 
  if not type(n,posint) then ERROR(`positive integer expected`,n)
  elif n>4 then `posets/gen_Lattices`(n)
  elif n=4 then [posets['chain'](4),{[1,2]} &* {[1,2]}]
  else [posets['chain'](n)]
  fi;
end:
#
# gen_Lattices(n)=create a list of naturally labeled, nonisomorphic lattices
#   on the vertex set {1..n}.
#
# Remark: It is actually faster (for n up to 9) to create this list by
#  taking the list of Posets(n-2), adding a minimal and maximal element to
#  each member of the list, and then using the lattice() procedure in the
#  examples directory to remove the posets that are not lattices.
#
`posets/gen_Lattices`:=proc(n)
local res,P,down,e,A,new,legal,j,i,lbs,m;
  res:=[];
  for P in posets[Lattices](n-1) do;
    down:=table([seq({i},i=1..n-2)]);
    for e in posets[closure](P,n-1) do down[e[2]]:={op(down[e[2]]),e[1]} od;
    for A in posets[antichains](P,n-1) minus {{},{n-1}} do;
      new:=subs(n-1=n,P) union map((x,y) -> [x,y],A,n-1) union {[n-1,n]};
      down[n-1]:=`union`(seq(down[A[i]],i=1..nops(A)),{n-1});
      legal:=true;
      for j to n-2 while legal do;
        lbs:=down[n-1] intersect down[j]; m:=max(op(lbs));
        if lbs minus down[m] <> {} then legal:=false fi;
      od;
      if legal then res:=[op(res),posets[covers](new)] fi;
    od;
  od;
  posets[rm_isom](res);
end:

#
# Posets(n) returns a list of all nonisomorphic posets on n vertices.
#  NOTE: The posets in this list are represented by their COVERING relations;
#  i.e., a set of ordered pairs [i,j] with no instance of [i,k] and [k,j] for
#  any k. The posets are naturally labeled; i.e., every relation [i,j]
#  satisfies i<j. To obtain the actual transitive relation, use closure(.).
#
`posets/Posets`:=proc(n) local res,P,A,t; option remember;
  if not type(n,posint) then ERROR(`positive integer expected`,n) fi;
  if n = 1 then []
  elif n = 2 then [{},{[1,2]}]
  else
    res:=NULL;
    for P in posets[Posets](n-1) do;
      for A in posets[antichains](P,n-1) do;
        res:=res, P union {seq([t,n],t=A)}
      od;
    od;
    posets[rm_isom]([res]);
  fi
end:

#
#  W(P,X,z) returns the W polynomial of the poset P with vertex set X
#    in the variable z. If X=an integer n, then vertex set={1,...,n}.
#    The labeling of P is ignored. 
#    If P has no isolated points, W(P,z) will also work.
#  W(P,X,z,bad), where  bad  is a subset of cover relations of P,  will return
#    the W polynomial for a labeling in which descents occur at these places.
#  W(P,n,z,bad), W(P,z,bad) will work similarly.
#
`posets/W`:=proc()
  local P,X,anti,n,poly,old,new,i,j,k,l,d,bad,one,Zero,down,f;
  if type(args[nargs],'set') then
    bad:=args[nargs]; f:=nargs-1
  else
    bad:={}; f:=nargs
  fi;
  if f=2 then X:=map(op,args[1])
    elif type(args[2],'integer') then X:={$1..args[2]}
    else X:=args[2]
  fi;
  P:=posets['closure'](args[1]);
  anti:=table(posets['antichains'](P,X,0));
  n:=nops([indices(anti)]); down:=table([{}$n]);
  for i to n do;
    anti[i]:= anti[i] union
      map(proc(x,y) if member(x[2],y) then x[1] fi end,P,anti[i]);
    if "={} then Zero:=i elif nops(")=nops(X) then one:=i fi;
    for j to i-1 do;
      d:=nops(anti[i])-nops(anti[j]);
      if d>0 then
        if `posets/W/safe`(anti[i] minus anti[j],d,bad) then
          down[i]:={op(down[i]),j} fi;
      elif d<0 then
        if `posets/W/safe`(anti[j] minus anti[i],-d,bad) then
          down[j]:={op(down[j]),i} fi;
      fi;
    od;
  od;
  old:=subsop(Zero=1,[0$n]); poly:=0;
  for i to nops(X) do;
    new:=[seq(sum('old[down[l][k]]',k=1..nops(down[l])),l=1..n)];
    poly:=poly+new[one]*args[f]^i*(1-args[f])^(nops(X)-i); old:=new;
  od;
  expand(poly);
end:
#
`posets/W/safe`:=proc(DD,d,bad) local e;
  if nops(DD)<>d then RETURN(false) fi;
  if d=1 then RETURN(true) fi;
  for e in bad do;
    if member(e[1],DD) and member(e[2],DD) then RETURN(false) fi;
  od;
  true
end:

#
# antichains(P,X) returns a list of all antichains in the poset P with
#   vertex set X. (if X = an integer n, then vertex set={1...n}).
#   If the second argument is omitted, then no isolated points are assumed.
# antichains(P,X,b), where b is any third argument (e.g., 'true'), will do
#   the same, but will assume that P has already been transitively closed.
# antichains(P,n,b) works similarly.
#
`posets/antichains`:=proc() local P,X;
  if nargs=1 then
    X:=map(op,args[1])
  elif type(args[2],'integer') then
    X:={$1..args[2]} else X:=args[2]
  fi;
  if nargs<3 then P:=posets['closure'](args[1]) else P:=args[1] fi;
  `posets/antichains/sub`(P,X);
end:
#
`posets/antichains/sub`:=proc(P,X) local P1,P2,X1,e,x1;
  if nops(X)<2 then RETURN({{},X}) fi;
  X1:=X minus {X[1]}; P2:=P; P1:=NULL;
  for e in P do;
    if e[1]=X[1] then
      P2:=P2 minus {e}; X1:=X1 minus {e[2]};
    elif e[2]=X[1] then
      P2:=P2 minus {e}; X1:=X1 minus {e[1]};
    fi;
  od;
  for e in P2 do
    if member(e[1],X1) and member(e[2],X1) then P1:=P1,e fi;
  od;
  #map(<x union {y}|x,y>,`posets/antichains/sub`({P1},X1),X[1])
  x1 := {X[1]};
  {seq(e union x1, e=`posets/antichains/sub`({P1},X1))}
    union `posets/antichains/sub`(P2,X minus {X[1]});
end:

#
# chain(n) returns (the cover-relation for) an n-element chain.
#
`posets/chain`:=proc(n) local i; {seq([i,i+1],i=1..n-1)} end:

#
#  char_poly(P,z) returns the characteristic polynomial of the poset P.
#    P must have a unique minimal element. 
#    In particular, it cannot have any isolated points.
#  char_poly(P,X,z) and char_poly(P,n,z) can also be used for consistency.
#
`posets/char_poly`:=proc() local P,X,n,z,i,ord,filterP,ht,v,x0;
  if nargs<2 then ERROR(`Wrong number of arguments`) fi;
  if nargs=2 then X:=map(op,args[1])
    elif type(args[2],'integer') then X:={$1..args[2]}
    else X:=args[2]
  fi;
  n:=nops(X); ord:=table([seq(X[i]=i,i=1..n)]);
  filterP:=posets['filter'](args[1],X);
  if nops(filterP[1])>1 then
     ERROR(`Poset must have a unique minimal element`) fi;
  ht:=nops(filterP);
  P:=posets['closure'](args[1]) union map(x -> [x,x],X); 
  z:=array('sparse',1..n,1..n,map(proc(x,y) (y[x[1]],y[x[2]])=1 end,P,ord));
  v:=array(1..n);
  for i to ht do;
    for x0 in filterP[i] do;
      v[ord[x0]]:=args[nargs]^(ht-i)
    od;
  od;
  linalg['linsolve'](z,v)[ord[filterP[1][1]]];
end:
#

#
# closure(R) returns the transitive closure of any acyclic relation R.
# R need not be naturally labeled, just a set of ordered pairs [a,b].
#
`posets/closure`:=proc() local R,X,i,j,k,n;
  R:=args[1];
  X:=map(op,posets['filter'](R));
  n:=nops(X);
  for i to n-2 do;
    for j from i+1 to n-1 do;
      if not member([X[i],X[j]],R) then next fi;
      for k from j+1 to n do;
        if member([X[j],X[k]],R) then R:=R union {[X[i],X[k]]} fi;
      od;
    od;
  od;
  R;
end:

#
# covers(R) returns the list of cover relations for any acyclic relation R.
# R need not be naturally labeled, just a set of ordered pairs [a,b].
#
`posets/covers`:=proc() local R,n,i,j,k,X;
  R:=args[1];
  X:=map(op,posets['filter'](R));
  n:=nops(X);
  for i from n-1 by -1 to 2 do;
    for j to n-i do;
      if not member([X[j],X[j+i]],R) then next fi;
      for k from j+1 to j+i-1 do;
        if member([X[j],X[k]],R) and member([X[k],X[j+i]],R) then
          R:=R minus {[X[j],X[j+i]]} fi; 
      od;
    od;
  od;
  R;
end:

#
# dual(R) returns the dual of the relation R. That is, each ordered pair
# [a,b] in R (a list or set) is replaced by [b,a].
#
`posets/dual`:=proc(R) map(x -> [x[2],x[1]],R) end:

#
# extensions(P,X) returns a list of all linear extensions of the poset P
# with vertex set X. If the second argument is an integer n, then X={1..n}.
# The second argument may be omitted if there are no isolated points.
#
`posets/extensions`:=proc(P) local X,bot,Q,x,res,new;
  if nargs=1 then X:=map(op,P)
    elif type(args[2],'integer') then X:={$1..args[2]}
    else X:=args[2]
  fi;
  if nops(X)=1 then RETURN([op(X)]) fi;
  bot:=X minus map(e -> e[2],P);
  res:=[];
  for x in bot do;
    Q:=map(proc(e,y) if e[1]<>y then e fi end,P,x); 
    new:=`posets/extensions`(Q,X minus {x});
    res:=[op(res),op(map((z,y) -> [y,op(z)],new,x))];
  od;
  res;
end:

#
# filter(D) returns the filtration of acyclic graph D.
#   filtration = [F1,F2,...,F_r], where F1={minimal elements},
#   F2={minimal elements of D-F1}, etc...
# filter(D,X) = same as above but uses vertex set X (necessary if D has
#   isolated points). filter(D,n)= same but with vertex set = {1,...,n}.
# filter(D,X,'flag') or filter(D,n,'flag') does the same, but also assigns
#   flag:=true or false, according to whether D is ranked; i.e., all edges of
#   D are of the form [i,j], where i in F_k and j in F_{k+1} for some k.
#
`posets/filter`:=proc(P) local X,Q,Y,k,res,e;
  if nargs=1 then X:=map(op,P)
    elif type(args[2],'integer') then X:={$1..args[2]}
    else X:=args[2]
  fi;
  if nargs=3 then assign(args[3],true) fi;
  if P={} then RETURN([X]) fi;
  #Y:=map(<op(2,x)|x>,P);
  Y:={seq(e[2],e=P)};
  #Q:=map(proc(x,Z) if member(x[1],Z) and member(x[2],Z) then x fi end,P,Y);
  Q:={seq(`if`(member(e[1],Y) and member(e[2],Y),e,NULL),e=P)};
  res:=[X minus Y, op(`posets/filter`(Q,Y))];
  if nargs<3 then RETURN(res) fi;
  for e in P do;
    for k while not member(e[1],res[k]) do od;
    if k>=nops(res) or not member(e[2],res[k+1]) then
      assign(args[3],false); break
    fi;
  od;
  res;
end:

#
# height(D) = cardinality of longest path in the acyclic digraph D.
#
`posets/height`:=proc(P) local X,ht,i;
  if P={} then RETURN(1) fi;
  X:=map(op,P); ht:=`posets/height/ht`(P,X);
  max(seq(ht[X[i]],i=1..nops(X)));
end:
#
`posets/height/ht`:=proc(P,X) local ht,x,bot,top,T,Q,v,e;
  if P={} then
    for x in X do ht[x]:=1 od; 
  else
    bot:=map(y -> y[1],P); top:=X minus bot;
    T:={}; Q:={};    
    for e in P do;  
      if member(e[2],top) then T:={op(T),e} else Q:={op(Q),e} fi;  
    od;    
    ht:=`posets/height/ht`(Q,bot);
    for v in top do ht[v]:=1 od;  
    for e in T do ht[e[2]]:=max(ht[e[2]],1+ht[e[1]]) od;  
  fi;
  op(ht);
end:

#
# invariants(P) will partition the vertices of the poset P (or any acyclic
#   digraph) into types according to the number and level of vertices that
#   cover or are covered by each vertex. The number of vertices of each
#   type is invariant under isomorphism, and thus can be used in isomorphism
#   testing. The output will be a list of lists [L_1,...,L_m]. Assuming
#   [F_1,...,F_k] is the filtration of P, a typical L_j will be of the form
#   [V,i,n_1,...,n_k], where V is a set of vertices in F_i, and for every x
#   in V, n_j is the number of vertices of F_j that cover or are covered by x.
#   The L_j's are sorted lexicographically by the sublists [i,n_1,...,n_k].
# invariants(P,n) and invariants(P,X) can be used to specify the vertex set.
#   If the filtration F of P has already been computed, then this information
#   can be passed on by invoking invariants(P,F).
#
`posets/invariants`:=proc(P) local F,X,i,j,x,lev,res,grp,e,types,typ;
  if nargs=2 and type(args[2],'list') then F:=args[2]
    else F:=posets['filter'](args) fi;
  X:=map(op,F); 
  for j to nops(F) do;
    for x in F[j] do lev[x]:=j od;
  od;
  res:=table('sparse');
  for e in P do;
    res[e[1],lev[e[2]]]:=res[e[1],lev[e[2]]]+1;
    res[e[2],lev[e[1]]]:=res[e[2],lev[e[1]]]+1;
  od;
  types:=[]; grp:=table();
  for x in X do
    typ:=[lev[x],seq(res[x,i],i=1..nops(F))];
    if member(typ,types,'j') then
      grp[j]:=grp[j],x
    else
      types:=[op(types),typ];
      grp[nops(types)]:=x 
    fi;
  od;
  res:=[seq([{grp[i]},op(types[i])],i=1..nops(types))];
  sort(res,`posets/invariants/lex`);
end:
#
`posets/invariants/lex`:=proc(x,y) local i;
  for i from 2 to nops(x) do
    if x[i]<>y[i] then RETURN(evalb(x[i]<y[i])) fi;
  od;
  true
end:

#
# isom(P,Q) decides if the acyclic relations P and Q are isomorphic.
#   First the number of edges of P and Q are compared, then the number of
#   vertices in their filtrations, then the number of vertices of each type.
# isom(P) will return the number of potential relabelings of the vertices
#   that could be required to decide whether P is isomorphic to some Q.
#
`posets/isom`:=proc()
    local P,Q,filterP,filterQ,shapeP,shapeQ,typeP,typeQ,i,X,Y,w,perms;
  if nargs=1 then
    RETURN(convert(map(x -> nops(x[1])!,posets['invariants'](args[1])),`*`)) fi;
  P:=args[1]; Q:=args[2];
  if nops(P)<>nops(Q) then RETURN(false) fi;
  filterP:=posets['filter'](P);
  filterQ:=posets['filter'](Q);
  shapeP:=map(nops,filterP); shapeQ:=map(nops,filterQ);
  if shapeP<>shapeQ then RETURN(false) fi;
  typeP:=posets['invariants'](P,filterP);
  typeQ:=posets['invariants'](Q,filterQ);
  shapeP:=map(x -> subsop(1=nops(x[1]),x),typeP);
  shapeQ:=map(x -> subsop(1=nops(x[1]),x),typeQ);
  if shapeP<>shapeQ then RETURN(false) fi;
  X:=map(op,map(x -> x[1],typeP));
  Y:=map(op,map(x -> x[1],typeQ));
  Q:=subs({seq(Y[i]=X[i],i=1..nops(X))},Q);
  if P=Q then RETURN(true) fi;
  perms:=[seq(typeP[i][1]$nops(typeP[i][1]),i=1..nops(typeP))];
  for w in posets['permfit'](table(perms),nops(perms)) do;
    if Q=subs({seq(X[i]=w[i],i=1..nops(X))},P) then RETURN(true) fi; 
  od;
  false;
end:

#
#  mobius(P,X) returns a table of values for the mobius function of the
#    poset P with vertex set X.
#  mobius(P,n) does the same but assumes X={1,...,n}
#  mobius(P), does the same, but assumes there are no isolated points. 
#  mobius(P,[a,b]) returns the value of the mobius function at [a,b].
#
`posets/mobius`:=proc() local P,X,n,z,i,ord,v,a,b,single;
  single:=false; X:=map(op,args[1]);
  if nargs>1 then
    if type(args[2],'integer') then X:={$1..args[2]}
      elif type(args[2],'list') then single:=true;
        a:=args[2][1]; b:=args[2][2]; X:=X union {a,b}
      else X:=args[2]
    fi;
  fi;
  n:=nops(X); ord:=table([seq(X[i]=i,i=1..n)]);
  P:=posets['closure'](args[1]) union map(x -> [x,x],X); 
  z:=array('sparse',1..n,1..n,map(proc(x,y) (y[x[1]],y[x[2]])=1 end,P,ord));
  if single then
    v:=array('sparse',1..n,[(ord[b])=1]);
    linalg['linsolve'](z,v)[ord[a]];
  else
    table(map(proc(x,m,y) (op(x))=m[y[x[1]],y[x[2]]] end,P,
      linalg['inverse'](z),ord));
  fi;
end:

#
#  omega(P,X,z) returns the order polynomial of the poset P with vertex set X
#    in the variable z. If X=an integer n, then vertex set={1,..,n}. The 
#    labeling of P is ignored. If P has no isolated points, then omega(P,z)
#    will also work.
#  omega(P,X,z,bad), where  bad  is a subset of cover relations of P,  will
#    return the order poly of the labeled poset having descent set = bad.
#
`posets/omega`:=proc() local n,z,poly,i,j,new_args;
  if type(args[nargs],'set') then j:=nargs-1 else j:=nargs fi;
  if j=2 then
    n:=nops(map(op,args[1]))
  elif type(args[2],'integer') then
    n:=args[2]
  else
    n:=nops(args[2])
  fi;
  new_args:=op(subsop(j=z,[args]));
  poly:=(1+z)^n*subs(z=z/(1+z),posets['W'](new_args));
  poly:=expand(expand(poly,1+z));
  [seq(coeff(poly,z,i)*`posets/omega/bin`(args[j],i),i=1..n)];
  convert(",`+`);
end:
#
`posets/omega/bin`:=proc(x,j)
  if j=0 then 1 else x/j*`posets/omega/bin`(x-1,j-1) fi;
end:

#
# P &+ Q returns the ordinal sum of the posets P and Q. (Similarly, one can
# use P &+ Q &+ R, and so on). If one or more of the arguments has isolated
# points, say P, then one must use the syntax [P,X] &+ Q, where X= the vertex
# set of P. If X={1,2,...,n}, then [P,n] &+ Q can also be used. The output
# will always be a poset  on the vertex set {1,2,...,m}, where m=the number
# of vertices in P and Q. Note that P &+ Q will never have isolated vertices.
#
unprotect(`&+`);
`&+`:=proc() local P,X,i,j,x,n,top,bot;
  if nargs>2 then RETURN(args[1] &+ (&+(args[2..nargs]))) fi;
  for x to nargs do;
    if type(args[x],'list') then
      P[x]:=args[x][1]; X[x]:=args[x][2];
    else
      P[x]:=args[x]; X[x]:=map(op,P[x])
    fi;
    if type(X[x],'integer') then n[x]:=X[x]
      else n[x]:=nops(X[x]);
      P[x]:=subs({seq(X[x][i]=i,i=1..n[x])},P[x]);
    fi;
  od;
  top:={$1..n[1]} minus map(e -> e[1],P[1]);
  bot:={$1..n[2]} minus map(e -> e[2],P[2]);
  {op(P[1]),op(subs({seq(i=i+n[1],i=1..n[2])},P[2])),
  seq(seq([top[i],bot[j]+n[1]],j=1..nops(bot)),i=1..nops(top))};
end:

#
# Let B = a table with index set {1,...,n}, where B[i] = a subset of some
# n-element list X.  permfit(B) returns a list of permutations w of X, where
# w[i] belongs to B[i] for all i.
#
`posets/permfit`:=proc(B) local res,n,i,j,x,Bhat;
  if nargs=2 then n:=args[2] else n:=nops([indices(B)]) fi;
  if n=1 then
    if B[1]={} then RETURN([]) else RETURN([[op(B[1])]]) fi;
  fi;
  res:=NULL;
  for i in B[n] do;
    for j to n-1 do Bhat[j]:=B[j] minus {i} od;
    # res:=res,op(map(<[op(x),y]|x,y>,`posets/permfit`(Bhat),i));
    res:=res,seq([op(x),i],x=`posets/permfit`(Bhat,n-1))
  od;
  [res];
end:

#
# P &* Q returns the direct product of the posets P and Q. (Similarly, one can
# use P &* Q &* R, and so on). If one or more of the arguments has isolated
# points, say P, then one must use the syntax [P,X] &* Q, where X= the vertex
# set of P. If X={1,2,...,n}, then [P,n] &* Q can also be used. If all of
# the arguments are of the form [P,X] or [P,n], then the output will be of the
# form [PQ,m], where PQ= the (cover relation of the) product, and m=the number
# of vertices of PQ. The vertex set of the output will always be {1,2,...,m}.
#
unprotect(`&*`);
`&*`:=proc() local X,P,PQ,XY,flag,i,j,x,e;
  if nargs>2 then RETURN(args[1] &* (&*(args[2..nargs]))) fi;
  PQ:=NULL;
  for x to nargs do;
    if type(args[x],'list') then 
      P[x]:=args[x][1]; X[x]:=args[x][2]; flag[x]:=true;
    else
      P[x]:=args[x]; X[x]:=map(op,P[x]); flag[x]:=false;
    fi;
    if type(X[x],'integer') then X[x]:={$1..X[x]} fi;
  od;
  for e in P[1] do;
    for x in X[2] do PQ:=PQ,[[e[1],x],[e[2],x]] od;
  od;
  for e in P[2] do;
    for x in X[1] do PQ:=PQ,[[x,e[1]],[x,e[2]]] od;
  od;
  XY:=[seq(seq([X[1][i],X[2][j]],j=1..nops(X[2])),i=1..nops(X[1]))];
  PQ:=subs({seq(XY[i]=i,i=1..nops(XY))},{PQ});
  if flag[1] and flag[2] then [PQ,nops(XY)] else PQ fi;
end:

#
#  rm_isom(<list>) removes all isomorphic copies from <list>, a list or set
#    of acyclic digraphs. Each graph is assumed to have the same number of
#    vertices. The posets of the final result will be relabeled to use
#    vertices 1,2,3...., and will be naturally labeled.
#
`posets/rm_isom`:=proc()
  local Q,R,inven,subinv,shapes,i,j,k,l,F,X,grp,i0,w,blk,s,
    alive,inv,typ,types,perms;
  R:=table(args[1]); shapes:=[];
#
# This section partitions the posets into groups according to their filtration
#
  for i to nops([indices(R)]) do;
    F:=posets['filter'](R[i]);
    X:=map(op,F); s:=map(nops,F);
    R[i]:=subs({seq(X[k]=k,k=1..nops(X))},R[i]);
    if member(s,shapes,'j') then
      inven[j]:=inven[j],i;
    else
      shapes:=[op(shapes),s]; inven[nops(shapes)]:=i
    fi;
  od;
  alive:=[];
#
# This section refines the partition into groups according to the number of
# vertices of each type, as defined by invariants().
#
  for i to nops(shapes) do;
    if nops([inven[i]])=1 then alive:=[op(alive),inven[i]]; next fi;
    F:=`posets/rm_isom/part`(shapes[i]);
    types:=[]; subinv:=table();
    for j in [inven[i]] do;
      inv:=posets['invariants'](R[j],F);
      X:=map(op,map(x -> op(1,x),inv));
      typ:=map(x -> subsop(1=nops(x[1]),x),inv);
      R[j]:=subs({seq(X[l]=l,l=1..nops(X))},R[j]);
      if member(typ,types,'k') then
        if R[j]<>R[op(1,[subinv[k]])] then subinv[k]:=subinv[k],j fi;
      else
        types:=[op(types),typ]; subinv[nops(types)]:=j
      fi;
    od;
    for j to nops(types) do;
      grp:=[subinv[j]]; 
      if nops(grp)=1 then alive:=[op(alive),grp[1]]; next fi;
#
# Now, assuming that there are at least two distinct posets whose vertices
# all have the same type, generate all possible type-preserving re-labellings
# and remove the isomorphic copies.
# 
      blk:=`posets/rm_isom/part`(map(x -> op(1,x),types[j]));
      perms:=[seq(blk[l]$nops(blk[l]),l=1..nops(blk))]; 
      perms:=posets['permfit'](table(perms),nops(perms));
      while nops(grp)>0 do;
        i0:=grp[1]; alive:=[op(alive),i0];
        grp:=subsop(1=NULL,grp);
        for w in perms while nops(grp)>0 do;
          Q:=subs({seq(l=w[l],l=1..nops(X))},R[i0]);
          for k from nops(grp) by -1 to 1 do;
            if Q=R[grp[k]] then grp:=subsop(k=NULL,grp) fi;
          od;
        od;
      od;
    od;
  od;
  [seq(R[alive[l]],l=1..nops(alive))];
end:
#
`posets/rm_isom/part`:=proc(shape) local ps,i,j;
  ps:=[0];
  for i to nops(shape) do ps:=[op(ps),ps[i]+shape[i]] od;
  [seq({$ps[j]+1..ps[j+1]},j=1..nops(ps)-1)];
end:

#
#  zeta(P,X,z) returns the zeta polynomial of the poset P with vertex set X
#    in the variable z. If X=an integer n, then vertex set={1,...,n}.
#    If P has no isolated points, then zeta(P,z) also works.
#  Patched on 4/1/92 to remove bug related to copying tables.
#
`posets/zeta`:=proc() local num,poly,e,x,i,j,X,old,new,up;
  if nargs=2 then X:=map(op,args[1])
    elif type(args[2],'integer') then X:={$1..args[2]}
    else X:=args[2]
  fi;
  poly:=0; num:=nops(X); 
  for x in X do old[x]:=1; up[x]:={} od;
  for e in posets['closure'](args[1]) do up[e[1]]:={op(up[e[1]]),e[2]} od;
  for i while num>0 do;
    poly:=poly+num*`posets/zeta/bin`(args[nargs]-2,i-1);
    for x in X do new[x]:=convert([seq(old[up[x][j]],j=1..nops(up[x]))],`+`) od;
    num:=convert(convert(new,'list'),`+`); 
    old:=table([seq(X[j]=new[X[j]],j=1..nops(X))]);
  od;
  poly;
end:
#
`posets/zeta/bin`:=proc(x,j)
  if j=0 then 1 else x/j*`posets/zeta/bin`(x-1,j-1) fi;
end:

posets[antichains]:=eval(`posets/antichains`):
posets[chain]:=eval(`posets/chain`):
posets[char_poly]:=eval(`posets/char_poly`):
posets[closure]:=eval(`posets/closure`):
posets[covers]:=eval(`posets/covers`):
posets[dual]:=eval(`posets/dual`):
posets[extensions]:=eval(`posets/extensions`):
posets[filter]:=eval(`posets/filter`):
posets[height]:=eval(`posets/height`):
posets[invariants]:=eval(`posets/invariants`):
posets[isom]:=eval(`posets/isom`):
posets['J']:=eval(`posets/J`):
posets[Lattices]:=eval(`posets/Lattices`):
posets[mobius]:=eval(`posets/mobius`):
posets[omega]:=eval(`posets/omega`):
posets[permfit]:=eval(`posets/permfit`):
posets[Posets]:=eval(`posets/Posets`):
posets[rm_isom]:=eval(`posets/rm_isom`):
posets['W']:=eval(`posets/W`):
posets[zeta]:=eval(`posets/zeta`):
`combinat/posets` := eval(posets):

#save `posets.m`;
#quit
