#
## <SHAREFILE=system/algsubs/algsubs.mpl >
## <DESCRIBE>
## A generalized substitution facility to allow one to make
##                substitutions for which subs does not do the job, e.g.
##                1) algsubs( a+b=R, a+c+b+d ) ==> a+R+d
##                2) algsubs( a*b=R, a*b^2*c ) ==> R*b*c
##                3) algsubs( a^2=0, a^3*b+a^2+a/b+2 ) ==> a/b+2
##                4) algsubs( a/b=R, a^3/b+a*b ) ==> a^2*R+a*b
##                AUTHOR: Michael Monagan, monagan@inf.ethz.ch
## </DESCRIBE>
## <UPDATE=R4 >

#--> algsubs( pat=rep, expr );
#--> algsubs( pat=rep, expr, vars );
#--> algsubs( pat=rep, expr, exact );
#--> algsubs( pat=rep, expr, algebraic );
#
# This routine was written to do ``algebraic'' substitions.
# It is a generalization of the subs command to find occurences of
# the algebraic expression ``pat'' in an expression and replace them
# with the algebraic expression ``rep''.
#
# Three simple examples to show algsubs's main purpose, which
# distinguish it from subs
# 
#
# > algsubs( s^2=1-c^2, s^3 );
#
#                                        2
#                                  (1 - c ) s
#
# > algsubs( a+b=d, 1+a+b+c );
# 
#                                   1 + d + c
#
# > algsubs( a*b=c, 2*a*b^2-a*b*d );
#
#                                  2 b c - c d
# 
# Negative powers are handled
#
#
# > algsubs( a*b/c=d, a*b^2/c+a*b );
#
#                                   a b + d b
#
#
# Algsubs differs from subs in two other technical ways
#
# 1: It does not look inside procedures or subscripts
#    E.g. algsubs( P=x, P[c]*P ); ==> P[c]*x not x[c]*x
#
# 2: Function calls are evaluted after subsitution
#    Hence algsubs( x=0, sin(x) ); ==> 0 not sin(0)
#
# Substitution is ambiguous in cases where there is more than one
# variable and/or term in the pattern.
# Consider
#
# 	algsubs( x+y=z, x+y^2 );
#
# Should it return y^2-y+z or x^2+(-2*z+1)*x+z^2 ?
# Consider
#
#       algsubs( a+b=c, 3*a+2*b+d );
#
# Should the result be 3*c-b+d or 2*c+a+d ?
# Or perhaps left unsubstituted.
#
# The algsubs command works in two different modes, remainder and exact.
# In remainder mode we use a generalized polynomial division with remainder
# where the variable ordering decides the answer to the above questions.
# In exact mode, if pat is a sum of terms x1 + x2 + x3, then the
# pattern matches its input f1 + f2 + f3 only if the monomial x[i] divides
# f[i] and the coefficient is the same, i.e. sum( f[i] ) = c * sum( x[i] ).
#
# Hence the pattern `pat` must be a polynomial in the variables.
# Variables can be names for functions.  E.g. x,A[1],sin(omega)].
# If no variables are given, all variables in `pat` which are not constants
# e.g. Pi, sin(1), are not considered variables.
#
# 1) When there is only one variable, say x, then  algsubs(a(x)=y,f(x))
#	simply computes the remainder of f(x) divided by a(x)-y .
#	E.g. algsubs( s^2=1-c^2, s^3, s ) ==> (1-c^2)*s
#
# 2) When there is more than one variable, then a generalized remainder
#	is computed with respect to the variables.  This depends on
#	which variable is considered to be the ``main'' variable.
#	For example,  algsubs( s^2+c^2=1, s^2, [s,c] ) ==> 1-c^2
#	but  algsubs( s^2+c^2=1, s^2, [c,s] ) ==> s^2 with no effect.
#	The leading term in pat must divide exactly the leading term in f.
#	For example, algsubs( x^2*y=z, x^2+y^2, [x,y] ); has no effect
#	since x^2*y doesn't divide either x^2 or y^2.
#
# 3) Like subs, algsubs is applied recursively to the expression.
#	This includes to factors e.g. algsubs( x+y=z, (x^2+y)^3*a ) does
#	NOT expand the polynomial.  If expr is expanded in the variables,
#	and it is a polynomial, the remainder is computed.
#	The user can expand the polynomial first if desired.
#
# 4) If the replacement `rep` also contains the variables, e.g.
#	algsubs( x*y=x^2, x*y^2 ) ==> y*x^2, the replacement is NOT
#	considered part of the polynomial.  I.e. the computation is performed
#	as  algsubs( x*y=Z, x*y^2 ) ==> y*Z ==> y*x^2, where Z is a dummy.
#	This differs from simlify/siderels.
#
# By default, the variables are chosen to be indets( pat ) = {s,c}
# In the above examples, the result depends on the variable ordering
# The final argument can be used to specify the variable ordering.
#
# This routine differs from simplify/siderels which calls
# grobner[normalf] in the following ways
#
# 1: It maps onto factors rather than expanding
# 2: It handles negative powers in monomials.
# 3: It is simpler because with only one equation
# 4: The replacement is not considered part of the pattern
#    Hence algsubs( x*y=x^2 ) is possible
#
# Caveat1: It would be nice if it handled radicals -- 
#	algsubs(x^2=y,x^(5/2));  algsubs(x^(1/2)=y,x^(3/2));
#
# Author MBM: May/94, Aug/94, Oct/94
#

macro( VARIABLE={name,function} );
macro( reduce=`algsubs/reduce` );
macro( recurse=`algsubs/recurse` );
macro( expanded=`algsubs/expanded` );
macro( MAP = `algsubs/map_n` );
macro( TERMS = `algsubs/terms` );
macro( exact_reduce = `algsubs/exact` );
macro( monomial_match = `algsubs/match` );
macro( headtermcoeff = `algsubs/headtermcoeff` );
macro( genrem=`algsubs/genrem` );
macro( dogenrem=`algsubs/dogenrem` );

unprotect('algsubs'):
algsubs := proc(

	p::algebraic=algebraic,
	f,
	x::{name,function,list({name,function}),set({name,function})}
	)

    local g,pat,rep,vars,opt,v,bad;

    option `Copyright 1994 Wissenschaftliches Rechnen, ETH Zurich`;
    if type(f, {list,set,relation,range,series} ) then
        RETURN( MAP[2](algsubs,p,f,args[3..nargs]) );
    elif not type(f,name) and type(f,procedure) then
	RETURN(f);
    fi;

    pat := lhs(p); rep := rhs(p);

    opt := remainder;
    
    # Compute the variables if not given
    if nargs<3 or args[3]='exact' or args[3]='remainder' then
	vars := frontend( indets, [pat] );
	vars := select( type, vars, {function,name} );
	vars := [op(vars)];
    else
        if type(x,VARIABLE) then vars := [x] else vars := [op(x)] fi;
    fi;

    # Check that the substitution is valid
    for v in vars do
	if degree(pat,v)=FAIL or degree(rep,v)=FAIL then
	    ERROR(`cannot compute degree of pattern in`,v);
	fi;
    od;

    if nargs>2 and (args[3] = 'remainder' or args[3] = 'exact') then
	opt := args[3];
    fi;
    if nargs>3 then
	if args[4] = 'remainder' or args[4] = 'exact'
	then opt := args[4]
	else ERROR(`4th argument must be either 'remainder' or 'exact'`)
	fi
    fi;

    # simplify the input polynomial
    pat := collect(pat,vars,distributed,normal);
    rep := collect(rep,vars,distributed,normal);

    bad := {seq( degree(pat,v), v=vars )} union
           {seq( ldegree(pat,v), v=vars )};
    if bad minus {0} = {} then ERROR(`no variables in pattern`) fi;

    `algsubs/algsubs`(f,pat,rep,vars,opt)

end:
protect('algsubs'):


`algsubs/algsubs` := proc(f,pat,rep,vars,opt)
local g;

    option `Copyright 1994 Wissenschaftliches Rechnen, ETH Zurich`;
    g := recurse(f,pat,rep,vars,opt);
    frontend( reduce, [g,pat,rep,vars,opt], [{`+`,`*`,list},{}] );

end:

# Apply algsubs/algsubs recursively to nested algebraic expressions
recurse := proc(f,pat,rep,vars,opt)
local a;
    option `Copyright 1994 Wissenschaftliches Rechnen, ETH Zurich`;
    if type(f,{numeric,name}) then f
    elif type(f,'ratpoly'(anything,vars)) then f
    elif type(f,{`*`,`+`}) then map(recurse,f,pat,rep,vars,opt)
    elif type(f,function) then
	op(0,f)(seq( `algsubs/algsubs`(a,pat,rep,vars,opt), a=f ))
    elif type(f,{procedure,table}) then f
    else MAP[2](`algsubs/algsubs`,f,pat,rep,vars,opt);
    fi;
end:



# Do an exact substitution
exact_reduce := proc(f,pat,rep,vars)
local p,g,c1,i,j,k,P,G,succeed,p1;

	option `Copyright 1994 Wissenschaftliches Rechnen, ETH Zurich`;
	p := TERMS(pat,vars);
        g := collect(f,vars,distributed,normal);
        g := TERMS(g,vars);
	#if nops(p)>nops(g) then RETURN(g) fi;
        headtermcoeff(pat,vars,'headterm','headcoeff');
        for i to nops(p) do
            if normal(p[i]-headterm*headcoeff)=0 then break fi;
        od;
        if( i>nops(p) ) then ERROR(`could not compute the head term`) fi;
        p1 := p[i];
        P := subsop(i=NULL,p);

	for j to nops(g) do
	    if not monomial_match(g[j],p1,vars,'c1') then next fi;
	    G := subsop(j=NULL,g);
	    succeed := true;
	    for i to nops(P) do
		succeed := false;
	        for k to nops(G) while not succeed do
		    if normal(c1*P[i]-G[k])=0 and
			monomial_match(G[k],P[i],vars) then
			G := subsop(k=NULL,G);
			succeed := true;
		    fi;
		od;
		if not succeed then break fi;
	    od;
	    if succeed then
		# we have that c1*pat+G=f
		g := c1*rep+convert(G,`+`);
		RETURN( exact_reduce(g,pat,rep,vars) );
	    fi;
	od;
	f
end:

# Match the monomials f and p in vars.
monomial_match := proc(f,p,vars,c::name) local v,C,P,F,dp,df;

    option `Copyright 1994 Wissenschaftliches Rechnen, ETH Zurich`;
    C := 1;
    P := p;
    F := f;
    for v in vars do
        dp := degree(P,v);
	df := degree(f,v);
	if dp = 0 then
	    if nargs=4 then C := C*v^(df-dp) fi;
	elif dp > 0 then
	    if df<dp then RETURN(false) fi;
	    if nargs=4 then C := C*v^(df-dp) fi;
	elif dp < 0 then
	    if df>dp then RETURN(false) fi;
	    if nargs=4 then C := C*v^(df-dp) fi;
	fi;
	if nargs=4 then P := subs(v=1,P) fi;
	if nargs=4 then F := subs(v=1,F) fi;
    od;
    if nargs=4 then c := F*C/P fi;
    true

end:


# Apply algsubs to an algebraic expression
reduce := proc(f,pat,rep,vars,opt)
local c,p,r,n,X;
option system, `Copyright 1994 Wissenschaftliches Rechnen, ETH Zurich`;

    if type(f,numeric) then f
    elif not has(f,vars) then f

    elif f=pat then rep # Exact match ==> speed up this case

    elif expanded(f,vars) then
	if opt = exact then
	    if has(rep,vars) then
	        r := exact_reduce(f,pat,X,vars,opt);
	        r := collect(subs(X=rep,r),vars,distributed,normal)
	    else
	        r := exact_reduce(f,pat,rep,vars,opt)
	    fi;
	else
	    if has(rep,vars) then p := pat-X else p := pat-rep; fi;
	    r := genrem(f,p,vars,opt);
	    if has(r,X) then
		r := collect(subs(X=rep,r),vars,distributed,normal) fi;
	    r;
	fi;

    elif type(f,`+`) then
	#c := select( (f,v) -> not has(f,v), f, vars );
	c := remove( has, f, vars );
	p := select( has, [op(f)], vars );
	r := select( expanded, p, vars );
	r := convert(r,`+`);
	r := reduce(r,pat,rep,vars,opt);
        #p := select( (f,v) -> not expanded(f,v), p, vars );
        p := remove( expanded, p, vars );
	p := convert( map(reduce,p,pat,rep,vars,opt), `+` );
        p+r+c

    elif type(f,`*`) then
	#c := select( (f,v) -> not has(f,v), f, vars );
	c := remove( has, f, vars );
	p := select( has, [op(f)], vars );
	r := select( hastype, p, `+` );
	r := convert( map(reduce,r,pat,rep,vars,opt), `*` );
	#p := select( f -> not hastype(f,`+`), p );
	p := remove( hastype, p, `+` );
	p := convert(p,`*`);
	p := reduce(p,pat,rep,vars,opt);
	c*p*r;

    elif type(f,`^`) and op(2,f)<0 then 1/reduce(1/f,pat,rep,vars,opt)
    elif type(f,`^`) then map(reduce,f,pat,rep,vars,opt)
    else f
    fi

end:

# Computes the head term and head coefficient in vars
headtermcoeff := proc(pat,vars,headterm::name,headcoeff::name)
local ht,hc,v,d,l;
    option `Copyright 1994 Wissenschaftliches Rechnen, ETH Zurich`;
    ht := 1;
    hc := pat;
    for v in vars do
        d := degree(hc,v);
        l := ldegree(hc,v);
        if d=0 and l=0 then next fi; # ignore v
	if d<-l then d := l fi;
	hc := coeff(hc,v,d);
	ht := ht*v^d
    od;
    headterm := ht;
    headcoeff := hc;
end:

# Do a generalized division of f divided by p
genrem := proc(f,p,vars)
local ht,hc,v,d,l,F,G,V;
    option `Copyright 1994 Wissenschaftliches Rechnen, ETH Zurich`;
    headtermcoeff(p,vars,'ht','hc');
    V := select( proc(v,f) has(f,v) end, vars, ht );
    F := f;
    G := dogenrem(F,p,V,hc,ht);
    G := collect(G,vars,distributed,normal);
    while F <> G do
        F := G;
        G := dogenrem(F,p,V,hc,ht);
        G := collect(G,vars,distributed,normal);
    od;
    F;
end:

# Do one step of a generalized remainder of f divided by p
dogenrem := proc(f,p,vars,hc,ht)
local d1,d2,c,F,q,v;
    option `Copyright 1994 Wissenschaftliches Rechnen, ETH Zurich`;
    if type(f,`+`) then map(dogenrem,f,p,vars,hc,ht)
    elif not has(f,vars) then f
    elif type(f,name) then
	if f=ht then f-p/hc else f fi;
    elif type(f,`^`) then
	v := op(1,f);
	if not type(ht,{identical(v),identical(v)^integer}) then RETURN(f) fi;
	d1 := op(2,f);
	d2 := degree(ht,v);
	if d2>0 then
	    if d1<d2 then f; else f-v^(d1-d2)/hc*p; fi;
	elif d2<0 then
	    if d1>d2 then f; else f-v^(d1-d2)/hc*p; fi;
	fi;
    elif type(f,`*`) then
	F := select(has,f,vars);
	c := select(proc(f,v) not has(f,v) end,f,vars);
	q := 1;
	for v in vars do
	    d1 := degree(F,v);
	    if d1=0 then RETURN(f) fi;
	    d2 := degree(ht,v);
	    if d2>0 then
		if d1<d2 then RETURN(f) fi;
		q := q*v^(d1-d2);
	    elif d2<0 then
		if d1>d2 then RETURN(f) fi;
		q := q*v^(d1-d2);
	    else # d1=d2
	    fi;
	od;
        f-c*q*p/hc;
    fi;
end:


# Returns true iff f is expanded in vars
expanded := proc(f,vars) local t;
    option `Copyright 1994 Wissenschaftliches Rechnen, ETH Zurich`;
    if type(f,`*`) then
	for t in f do
	    if has(t,vars) and hastype(t,`+`) then RETURN(false) fi;
	od;
	true
    elif type(f,`+`) then
	for t in f do
	    if not expanded(t,vars) then RETURN(false) fi;
	od;
	true
    elif type(f,`^`) then
	evalb( not has(op(1,f),vars) or not type(op(1,f),`+`) )
    else true
    fi
end:


MAP := proc(f) local n,r;
    option `Copyright 1994 Wissenschaftliches Rechnen, ETH Zurich`;
    n := op(procname); # Utility routine to map over an object
    r := [args[2..nargs]];
    map( proc(x,f,n,r) f(op(subsop(n=x,r))) end, r[n],f,n,r );
end:


TERMS := proc(f,vars) local c,t,i;
    option `Copyright 1994 Wissenschaftliches Rechnen, ETH Zurich`;
    c := coeffs(f,vars,t); c := [c]; t := [t];
    [seq( c[i]*t[i], i=1..nops(c) )];
end:

#remove := proc(f)
#    option `Copyright 1994 Wissenschaftliches Rechnen, ETH Zurich`;
#    select( subs('F'=f, proc() not F(args) end), args[2..nargs] );
#end:

