#
## <SHAREFILE=plots/sglplot/sglplot.mpl >
## <DESCRIBE>
##                A routine for displaying a subgroup lattice in 3-dimensions
##                Examples in files plots/group.48.43 and plots/group.48.44
##                AUTHOR: Michael Monagan, monagan@inf.ethz.ch
## </DESCRIBE>

#
#--> sglplot(V,E): subgroup lattice plot in 3 dimensions
#
# This routine is intended to show how one could plot a subgroup lattice
# in Maple in 3-dimensions using the PLOT3D data structure.
#
# The lattice is a graph (V,E) input as follows:
#
# V:list(list(list(integer)))
# E:list(integer={integer,set(integer),list(integer)})
#
# The vertices V are input in a predetermined spacial arrangement as
# a list of layers or planes, each layer is a list of rows of vertices,
# given by a list of integers.  The output will place the vertices
# in the order described by V.
#
# The edges E are input as an adjacency list, namely a list
# of equations where the left hand side is the vertex, and teh
# right hand side is the adjacent vertices, specified by
# either a single integer, or a list or set of integers.
#
# The output is a PLOT3D structure corresponding to the lattice.
# Vertices are drawn as points edges as lines.
# The arrangement is in the order the vertices are given in V.
#
# This file contains Group.44.43 and Group.48.44 as examples below.
# Example:  sglplot( Group.44.43 );
#
# Author: MBM Sep/90
#


sglplot := proc( V::list(list(list(integer))),
		 E::list(integer={integer,list(integer),set(integer)})
	       )
	local i,j,k, x,y,z, l,m,n, C,P,xyz, e,s,t, X,Y,Z, l1,l2;

	l := nops(V);
	for i to l do
	    l1 := V[i];
	    m := nops(l1); 
	    for j to m do
	        l2 := l1[j];
	        n := nops(l2);
	        for k to n do
		    x := i + (k+1)/(n+3) - 1/2;
		    y := i/2 - j/(m+1) + 1/2;
		    z := i - 1/2;
		    xyz[l2[k]] := [z,x,y];
		od
	    od
	od;
	
	P := POINTS( map(op,[entries(xyz)]) );
	
	n := nops(E);
	for i to n do
	    s := lhs(E[i]);
	    t := NULL;
	    e := rhs(E[i]);
	    if type(e,numeric) then t := xyz[s], xyz[e]
	    else for e in rhs(E[i]) do t := t, xyz[s], xyz[e], xyz[s] od
	    fi;
	    C[i] := CURVELIST( [t] );
	od;

	C := seq( C[i], i=1..n );

	PLOT3D( P, C,
		STYLE(WIREFRAME), LABELS([X,Y,Z]), AXES(FRAME),
		ORIENTATION(5,80),
		VIEW(1/3..l-1/3,1/2..l+1/2,0..(l+1)/2 ) )
end:

macro( Vertextype =  Vertextype, Edgetype = Edgetype );

# Group 44.43
# The data here is intended to be plotted by "sglplot"
# Author: MBM Oct/89

Group.44.43 :=  

# The vertex planes

     [ [ [46,47,48],
	 [38,39,40,33,35,36,37],
	 [17,18,19,16,20,21,15,23,24,25,26,27,28],
	 [$1..13],
	 [0] ],

       [ [51],
	 [50,45,49],
	 [44,43,34,41,42],
	 [22,29,30,31,32],
	 [14] ]
     ],

# The edge adjacecies

     [
	51={50,45,49,46,47,48},

	44={17,18,19,50,43,22},
	43={16,20,21,50,22},
	34={15,50,45,49,22},
	41={23,24,25,29,42,49,30,22},
	42={26,27,28,49,32,31,22},

	46=47, 47=48,
	
	29=30, 30=31, 31=32,

	14={22,29,30,31,32,0},

	38={46,17,16,15,50},
	39={38,40,47,18,20,15,50},
	40={48,19,21,15,50},
	33={46,47,48,15,45},
	35={46,15,23,26,49},
	36={35,37,47,15,24,27,49},
	37={48,15,25,28,49},

	17=18, 18=19, 19=16, 16=20, 20=21,
	23=24, 24=25, 25=26, 26=27, 27=28,

	1={0,17,18,19,16,20,21,15,$23..28},
	2={0,23,29,3}, 3={0,24,29,4}, 4={0,25,29,5},
	5={0,23,30,6}, 6={0,24,30,7}, 7={0,25,30,8},
	8={0,26,31,9}, 9={0,27,31,10}, 10={0,28,31,11},
	11={0,26,32,12}, 12={0,27,32,13}, 13={0,28,32}

     ]:

# Group 48.44
# The data here is intended to be plotted by "sglplot"
# Author: MBM Oct/89

Group.48.44 := 

# The vertex planes

     [ [ [30,31,32],
	 [19,20,21,17,22,23,24],
	 [$4..9,3,10,14,15,11,12,13],
	 [1],
	 [0] ],

       [ [35],
	 [33,29,34],
	 [25,26,18,27,28],
	 [16],
	 [2] ]
     ],

# The edge adjacecies

     [
	35={30,31,32,33,29,34},
	25={4,5,6,33,26,16},
	26={7,8,9,33,16},
	18={3,33,29,34,16},
	27={10,14,15,34,16,28},
	28={11,12,13,34,16},

	30=31,31=32,

	2={16,0},
	19={20,30,33,4,7,3},
	20={31,33,5,8,3},
	21={20,32,33,6,9,3},
	17={30,31,32,29,3},
	22={30,23,34,3,10,11},
	23={31,34,3,14,12},
	24={32,23,34,3,15,13},

	1={16,0,$3..15},
	4=5,5=6,6=7,7=8,8=9,
	10=14,14=15,15=11,11=12,12=13

     ]:

#save `sglplot.m`;
#quit
