#
## <SHAREFILE=science/Wigner/Wigner.mpl >
## <DESCRIBE>
## Procedures to calculate the rotation matrix functions and
##                 Wigner_3j and Wigner_6j the 3-j and 6-j Wigner coefficients.
##                 SEE ALSO: Maple Technical Newsletter Vol. 1 No. 2, 1994.
##                 AUTHOR: Dennis Isbister, dji@adfa.edu.au
## </DESCRIBE>
## <UPDATE=R4 >

Wigner:=`Wigner `:
interface(prettyprint=false);
#  This Maple program is called dlmn.map.
#  It calculates the dj/nm(b) representation coefficients as defined 
#  in Edmond's book equations 4.1.15, p57
#
#
#  define n! = Gamma(n+1)
#
dlmn:=proc(j,n,m,t)
        local s,ss,f,d,prefactor;
        s:=0;                ss:=0:
        for s from 0 to min(j-m,j-n) do
            if j-m-s<0 or j-n-s<0 or m+n+s<0 then
                    f:=0
            else    f:=1/((j-m-s)!*(j-n-s)!*s!*(m+n+s)!);
            fi;
            ss:=ss + (cos(t/2))**(2*s+m+n)
                        *(sin(t/2))**(2*j-2*s-m-n)
                        *((-1)**s*f);
        od;
        prefactor:=(j-m)!*(j+m)!*(j-n)!*(j+n)!;
        d:=(-1)**(j-n)*ss*sqrt(prefactor);
        convert(d,trig);
        factor(");
end:
#
#  This Maple program is called Wigner_3j.map.
#  It calculates the 3j symbol as defined in Edmond's book equations 
#  3.6.10 for the Clebsh-Gordon coefficient and 3.7.3 for the 3j symbol 
#  defined in terms of the latter.
#
# Define the factorial ftn fac(n) to be GAMMA(n+1)
#
Wigner_3j:=proc(j1,j2,j3,m1,m2,m3)
        local s,ss,f,edmonds;
    if type([args],list(rational)) then
	# Correction
        if abs(m1)>j1 or abs(m2)>j2 or abs(m3)>j3 then RETURN (0) fi;
        if m1+m2+m3 <> 0 then RETURN (0) fi;
        if j3 < abs(j1-j2) or j3 > j1+j2 then RETURN (0) fi;
        ss:=0:
        for s from 0 to j3+m3 do
            if j1-m1-s < 0 or j3+m3-s < 0 or j2-j3+m1+s < 0 then 
            f:=0
            else        
            f:=(j1+m1+s)!*(j2+j3-m1-s)!/(s!*(j1-m1-s)!*(j2-j3+m1+s)!*(j3+m3-s)!):
            fi;
            ss:=ss+(-1)**s*f;
        od;
        f:=(j1+j2-j3)!*(j1-m1)!*(j2-m2)!*(j3-m3)!*(j3+m3)!
        /((j1+j2+j3+1)!*(j1-j2+j3)!*(-j1+j2+j3)!*(j1+m1)!*(j2+m2)!):
        edmonds:=(-1)**(-2*j1-m1-j2-m3)*ss*sqrt(f);
        simplify(edmonds)
    else
        'Wigner_3j'(args) # return unevaluated
    fi;
end:

#
#  This Maple program is called Wigner_6j.map.
#  It calculates the 6j symbol as defined in Edmond's book equations 
#  6.3.7 for the 6j symbol in terms of the w sum. 
#
# define the factorial function in terms of m! = Gamma(m+1)
#
# define the delta(a,b,c) = sqrt[(a+b-c)!(a-b+c)!(-a+b+c)!/(a+b+c+1)!]
# as  well test for the triangle inequality between j1,j2, and j3
#
#
delta:= proc(j1,j2,j3)
        local p;
        if abs(j1 - j2) <= j3 and j3 <= j1 + j2 then
        p:=(j1+j2-j3)!*(j1-j2+j3)!*(-j1+j2+j3)!/(j1+j2+j3+1)!;
        else       RETURN(0)
        fi;
        RETURN(sqrt(p))
end:

Wigner_6j:= proc(j1,j2,j3,l1,l2,l3)
        local s,smin,smax,ss,pref,f;
    if type([args],list(rational)) then
        pref:=delta(j1,j2,j3)*delta(j1,l2,l3)*delta(l1,j2,l3)*delta(l1,l2,j3):
        if (pref = 0) then RETURN(0) fi;
        smin:= max(j1+j2+j3,j1+l2+l3,l1+j2+l3,l1+l2+j3);
        smax:= min(j1+j2+l1+l2,j2+j3+l2+l3,j3+j1+l3+l1);
        ss:=0:
        for s from smin to smax do
            if s-j1-j2-j3<0 or s-j1-l2-l3<0 or s-l1-j2-l3<0 
                or s-l1-l2-j3<0 or j1+j2+l1+l2-s<0 
                or j2+j3+l2+l3-s<0 or j3+j1+l3+l1-s<0 then
                        
                    f:= infinity

            else    f:=(s-j1-j2-j3)!*(s-j1-l2-l3)!*(s-l1-j2-l3)!
                                    *(s-l1-l2-j3)!*(j1+j2+l1+l2-s)!
                                    *(j2+j3+l2+l3-s)!*(j3+j1+l3+l1-s)!:
            fi;

            ss:=ss+(-1)**s*(s+1)!/f:
        od;
        pref:=delta(j1,j2,j3)*delta(j1,l2,l3)*delta(l1,j2,l3)*delta(l1,l2,j3):
        simplify(pref*ss)
    else
       'Wigner_6j'(args) # return unevaluated
    fi;
end:
