#
## <SHAREFILE=calculus/PS/PS.mp  >
## <DESCRIBE>
##         The power series package PS is an improved version of the
##         library package powseries for manipulating formal power series.
##         Improvements are
##         1: a power series may be created from a given expression
##            (the coefficients don't have to be specified by equations)
##         2: it is no longer necessary to name each intermediate series
##         3: the trigonometric functions are implemented
##         4: series are automatically displayed to order Order
##            (using the print formatting facility in Maple V)
##         AUTHOR: Dominik Gruntz, gruntz@inf.ethz.ch
## </DESCRIBE>

# Package for formal power series.
#
# Calling sequence:   PS [ <funcname> ] ( <arguments> )
#
#   or else do:   with(PS);   or   with(PS,fcn1,fcn2,...);
#      and then use the simpler calling sequence
#         funcname ( <arguments> )
#
#            D. Gruntz, Apr 1992
#

macro(variable   = `PS/variable`):
macro(coeffmap   = PS[coeffmap] ):
macro(coeffzip   = PS[coeffzip] ):
macro(coeffshift = PS[coeffshift] ):
macro(powcreate1 = `PS/powcreate/algebraic`):
macro(powcreate2 = `PS/powcreate/equation`):
macro(powcoeff   = PS[powcoeff]):
macro(powcreate  = PS[powcreate]):
macro(fix        = `PS/fixedpoint`):
macro(odeprod    = `PS/odeprod` ):

# Creation of PS:
##################

powcreate := proc (p) local coeffproc, i; 
   if type(p, procedure) then
      if type(p, name) then coeffproc := p
      else coeffproc := subsop(3=remember,eval(p))
      fi
   elif type(p, algebraic) then RETURN(powcreate1(args))
   elif type(p, equation)  then RETURN(powcreate2(args))
   else ERROR(`wrong arguments`)
   fi;
   
   if nargs<2 then ERROR(`wrong arguments`) fi:

   if type(args[2],{list,set}) then
      for i to nops(args[2]) do
         coeffproc(op(1,args[2][i])) := op(2,args[2][i]):
      od
   fi:
   if not type(args[nargs], name) then ERROR(`last argument must be a name`) fi;

   coeffproc('VAR') := args[nargs];
   _powerseries(eval(coeffproc)): 
end: # powcreate

powcreate1 := proc(expr) local x, f, n, d, e, ind;
   if nargs=2 then x := args[2]
   else ind := indets(expr, name);
      if nops(ind)=1 then x := op(ind) else ERROR(`specify expansion variable`) fi;
   fi:

   if   type(expr, polynom(anything, x)) then PS[powpoly](expr, x)
   elif type(expr, series) and op(0,expr) = x then
      PS[add](
         PS[powpoly](convert(expr, polynom), x),
         powcreate(subs('m' = op(nops(expr), expr), 
            proc(k) if k < m then 0 else 'unknown' fi end), 
            x
         )
      )
   elif type(expr, `+`) then PS[add](op(map(procname, expr, x)))
   elif type(expr, `*`) then 
      if not has(op(1,expr), x) then # this is a constant
         PS[multconst](procname(expr/op(1,expr), x), op(1,expr))
      else n := numer(expr); d := denom(expr);
         if d <> 1 then 
            PS[quotient](procname(n,x), procname(d,x))
         else 
            PS[multiply](op(map(procname, expr, x)))
         fi
      fi
   elif type(expr, `^`) then e := op(2,expr);
      if not has(e, x) then
         PS[`^`](procname(op(1,expr), x), e)
      else
         PS[`^`](procname(op(1,expr), x), procname(e, x))
      fi;
   elif type(expr, function) then f := `pow`.(op(0,expr));
      if assigned(PS[f]) then
         PS[f]( procname(op(1,expr), x) )
      else
         ERROR(`don't know PS of `,op(0,expr))
      fi
   else
      ERROR(`don't know how to develop `, expr, `about `,x,`=0`);
   fi;
end: # powcreate1      

powcreate2 := proc(eqns)
# creates formal power series with initial conditions = init, 
#   and general term = recur( a(n-1)...a(n-m)i,n )
   local i, init, z, n, p, recur, temp, temp2, a, var;

   if type(args[nargs],name) then var := args[nargs] else var := 'x' fi;
   if nargs>1 then init:={args[2..nargs]} minus {var} else init:={} fi;
   a := op(0,op(1,eqns));
   map( 
      proc(x) if not (type(x,`=`) and type(op(1,x),function)) then x fi end, 
      init union {eqns}
   ) union 
   map(
      proc(x,a) if op(0,op(1,x))<>a or (not type(op(1,op(1,x)),integer)) then x fi end, 
      init, a
   );
   if " <> {} then ERROR(`invalid arguments`); fi;

   n := op(1,op(1,eqns)); recur := op(2,eqns);
   temp := map( 
      proc(x, a) if op(0,x)=a and nops(x)=1 then op(1,x) fi end,
      indets(recur,function), a 
   );
   if temp <> {} then
      if member(n, temp) then ERROR( a(n),`included in its own defn`) fi;
      temp2 := {};
      for z in temp do 
         if not has(z,n) then 
            temp2 := temp2 union { a(z) }
         elif type(-z+n,integer) then
            temp2 := temp2 union { seq(a(i), i=0..-z+n-1) }
         fi;
      od;
      map( type, subs(init, temp2), numeric);
      if has(", false) then ERROR(`insufficient initial conditions`) fi;
   fi;
   init := map( proc(x) [op(op(1,x)),op(2,x)] end, init);

   p := subs('_F'=recur, a=p, proc(n) option remember; _F end);
   powcreate(p, init, var);
end: # powcreate2

PS[powpoly] := proc (poly, var) local L, i, ci;
   if not ( nargs=2 and type(poly, polynom(anything,var)) ) then
      ERROR (`wrong type or number of arguments`)
   fi; L := NULL;
   for i from ldegree(poly, var) to degree(poly, var) do
      ci := coeff(poly, var, i); if ci <> 0 then L := L, [i, ci] fi
   od;
   powcreate(proc(k) 0 end, [L], var)
end: # powpoly

# Output of PS:
################

variable := proc(p) op(p)('VAR') end:

PS[tpsform] := proc(s) local var, deg, k, t;
   if nargs>=2 and type(args[2], integer) then deg := args[2] else deg := Order-1 fi;
   var := variable(s);

   t := 0:
   for k from 0 to deg do t := t + powcoeff(s,k) * var^k od: 
   series(t+O(var^k), var, deg+1)
end: # tpsform

`print/_powerseries` := proc(p):
   PS[tpsform](_powerseries(p))
end:

# Type testing: type(., powerseries);
######################################

`type/powerseries` := proc(arg) type(arg, _powerseries(procedure)) end:

# Operations on coefficients of a powerseries:
###############################################

powcoeff := proc(s, n);
   if type(n, integer) then
      if n < 0 then ERROR(`improper index`)
      else eval(op(1,s)(n))
      fi;
   else ERROR(`wrong arguments`) fi
end: # powcoeff

coeffmap := proc(f, u);
   powcreate(subs(['_f'=f, '_u'=u, 'arg'=args[3..nargs]], 
      proc(k) option remember; _f(powcoeff(_u,k), arg) end),
      variable(u)
   )
end: # coeffmap

coeffzip := proc(f, u, v) ;
   powcreate(subs(['_f'=f, '_u'=u, '_v'=v], 
      proc(k) option remember; 
         _f(powcoeff(_u,k),powcoeff(_v,k)) 
      end),
      variable(u)
   )
end: # coeffzip

coeffshift := proc(s, n);
   if n <> 0 then
      powcreate(subs(['_s'=s, '_n'=n],
         proc(k) if k < _n then 0 else powcoeff(_s,k-_n) fi end),
         variable(s)
      )
   else s fi;
end: # coeffshift

# fixedpoint computation:
##########################

PS[fixedpoint] := proc(F, var, C) local e, p, i;
   if not type(F,procedure) then
      ERROR(`first argument must be of type procedure`) fi:
   if nops([op(1,eval(F))]) > 1 then RETURN(fix(args)) fi:
   
   p('VAR') := var:
   if nargs>=3 and type(C,list) then for i to nops(C) do p(i-1) := C[i] od fi;
   e := F(_powerseries(p)):
   p := op(1,e):
   e:
end:

fix := proc(F, var) local e, p, n, unique, i;
   if not type(F,procedure) then
      ERROR(`first argument must be of type procedure`) fi:
      
   unique := proc() local p; p end;
   n := nops([op(1,eval(F))]); p := [seq(unique(), i=1..n)];
   for i to n do p[i]('VAR') := var od:
   e := F(op(map(_powerseries, [seq(p[i], i=1..n)]))):
   for i to n do assign(p[i], op(1,e[i])) od:
   e:
end:

# simple arithmetic on powerseries {+,-,*,/,^}:
################################################

PS[add] := proc (u, v) local w;
   if variable(u) <> variable(v) then ERROR(`different variables`) fi;
   w := coeffzip( (x,y)->(x+y), u, v);
   if nargs > 2 then PS[add](w, args[3..nargs]) else w fi
end: # add

PS[subtract] := proc (u,v);
   if variable(u) <> variable(v) then ERROR(`different variables`) fi;
   coeffzip((x,y)->(x-y), u, v)
end: # subtract

PS[negative] := proc (u);
   coeffmap(x->(-x), u)
end: # negative

PS[multconst] := proc(u, const);
   coeffmap(subs('_const'=const, x -> _const*x), u)
end: # multconst

PS[multiply] := proc (u, v) local w;
   if variable(u) <> variable(v) then ERROR(`different variables`) fi;
   w := powcreate(subs(['_u'=u, '_v'=v],
           proc(k) local i,t: t := 0:
              for i from 0 to k do
                 t := t + powcoeff(_u,i ) * powcoeff(_v,k-i)
              od:  t
           end),
           variable(u)
        );
   if nargs > 2 then PS[multiply](w, args[3..nargs]) else w fi
end: # multiply

PS[inverse] := proc(v) local w; # u0 = 1, uk = 0, k>0
   if powcoeff(v,0)=0 then
      ERROR(`inverse will have pole at zero`)
   fi;
   w := powcreate(subs(['_v'=v, '_w'=w],
          proc(k) local i, t: 
             if k=0 then t := 1 else t := 0 fi:
             for i from 0 to k-1 do
                t := t - powcoeff(_w,i) * powcoeff(_v,k-i)
             od: 
             t / powcoeff(_v,0)
          end),
          variable(v)
       );
end: # inverse

PS[quotient] := proc (U, V) local u, v, m, mu, mv, w;
   if variable(U) <> variable(V) then ERROR(`different variables`) fi;
   for mu from 0 while powcoeff(U, mu)=0 do od;
   for mv from 0 while powcoeff(V, mv)=0 do od;
   m := min(mu,mv);
   if m <> 0 then
      u := coeffshift(U,-m);
      v := coeffshift(V,-m);
   else u := U; v := V;
   fi;
   if powcoeff(v,0)=0 then
      ERROR(`inverse will have pole at zero`)
   fi;
   w := powcreate(subs(['_u'=u, '_v'=v, '_w'=w],
           proc(k) local i, t: t := 0:
              for i from 0 to k-1 do
                 t := t + powcoeff(_w,i) * powcoeff(_v,k-i)
              od: (powcoeff(_u, k)-t) / powcoeff(_v,0)
           end),
           variable(U)
        );
end: # quotient

PS[`^`] := proc(s, alpha) local m, u1, u2:   
   if type(alpha, powerseries) then
      PS[powexp](PS[multiply](alpha, PS[powlog](s)))
   else
      if   alpha=0 then RETURN(powcreate(proc(k) 0 end,[[0,1]],variable(s)))
      elif alpha=1 then RETURN(s)
      else 
         for m from 0 while powcoeff(s, m)=0 do od;

         if not type(alpha*m,integer) then 
            ERROR(`unable to power`) fi:
   
         u1 := coeffshift(s,-m):
         u2 := PS['fixedpoint'](subs(['_s'=u1, '_alpha'=alpha],
                  e -> PS[powint](
                          PS[multconst](
                             PS[multiply](
                                e,
                                PS[quotient](PS[powdiff](_s),_s)
                             ),
                             _alpha
                          ),
                          powcoeff(_s,0)^_alpha)
            ),
            variable(s)
         );
         coeffshift(u2,alpha*m)
      fi
   fi
end: # `^`

# integration and differentiation of powerseries:
##################################################

PS[powdiff] := proc (s) 
   powcreate(subs('_s'=s, k -> powcoeff(_s,k+1)*(k+1)), variable(s))
end: # powdiff

PS[powint] := proc (s) local c;
   if nargs=1 then c := 0 else c := args[2] fi;
   powcreate(subs('_s'=s, k -> powcoeff(_s,k-1)/(k)), [[0,c]], variable(s))
end: # powint

# powerseries reversion and composition:
#########################################

PS[reversion] :=  proc(u) local u1, v, powersOfU;
   if not ((powcoeff(u,0) = 0) and (powcoeff(u,1) <> 0)) then
      ERROR(`can not reverse`) fi;

   powersOfU := subs(['_u'=u], 
      proc(k) option remember; PS[`^`](_u,k)
   end); # powersOfG

   u1 := 1/powcoeff(u,1);
   v := powcreate(subs(['_u'=u, '_u1'=u1, '_v'=v, '_ui'=powersOfU],
           proc(k) local s, i:
              if   k=0 then 0
              elif k=1 then _u1
              else s := 0;
                 for i to k-1 do
                    s := s + powcoeff(_v,i) * powcoeff(_ui(i),k)
                 od;
                 -(_u1^k)*s
              fi
           end),
           variable(u)
        )
end: # reversion

PS[compose] := proc(F, G) local powersOfG; # F(G(x))
   if powcoeff(G,0) <> 0 then
      ERROR(`G`,`must have coefficient of x^0 = zero`) fi;

   powersOfG := subs(['_G'=G], 
      proc(k) option remember; PS[`^`](_G,k)
   end); # powersOfG

   powcreate(subs(['_G'=G, '_F'=F, '_Gi'=powersOfG],
      proc(k) local s, i;
         if k=0 then powcoeff(_F,0)
         else s := 0;
            for i to k do
               s := s + powcoeff(_F,i) * powcoeff(_Gi(i), k)
            od; s
         fi
      end),
      variable(G)
   )
end:

# elementary functions on powerseries {exp, log, ln, sin, cos, tan, sinh, cosh, tanh} :
########################################################################################

PS[powexp] := proc (s)
   PS['fixedpoint'](subs('_s'=s, 
      e -> PS[powint](PS[multiply](e, PS[powdiff](_s)),
                               exp(powcoeff(_s,0)))
      ), 
      variable(s)
   )
end: # powexp

PS[powlog] := proc (s)
   if powcoeff(s,0)=0 then 
      ERROR(`singularity encountered`)
   fi;
   PS[powint](PS[quotient](PS[powdiff](s), s),
                       log(powcoeff(s,0)))
end: # powlog 
PS[powln] := ":

PS[powtan] := proc(s) local t0:
   t0 := traperror(tan(powcoeff(s,0)));
   if t0=lasterror then ERROR(lasterror) fi;
   PS['fixedpoint'](subs(['_s'=s,'_t0'=t0],
        e -> PS[powint](
                PS[multiply](
                   PS[add](powcreate(proc(k) 0 end,[[0,1]],variable(_s)),
                                    PS[multiply](e,e)
                   ),
                   PS[powdiff](_s)
                ),
                _t0
             )
      ),
      variable(s)
   )
end: # powtan

PS[powsin] := proc(s);
   PS['fixedpoint'](subs([_s=s], 
      (s,c) -> ( 
         PS[powint](
            PS[multiply](c,PS[powdiff](_s)), 
            sin(powcoeff(_s,0))
         ),  
         PS[powint](
            PS[multiply](PS[negative](s),PS[powdiff](_s)),
            cos(powcoeff(_s,0))
         )
      )),
      variable(s)
   )[1]
end:
PS[powcos] := proc(s);
   PS['fixedpoint'](subs([_s=s], 
      (s,c) -> ( 
         PS[powint](
            PS[multiply](c,PS[powdiff](_s)), 
            sin(powcoeff(_s,0))
         ),  
         PS[powint](
            PS[multiply](PS[negative](s),PS[powdiff](_s)),
            cos(powcoeff(_s,0))
         )
      )),
      variable(s)
   )[2]
end:

PS[powsinh] := proc(s);
   PS['fixedpoint'](subs([_s=s], 
      (s,c) -> ( 
         PS[powint](
            PS[multiply](c,PS[powdiff](_s)), 
            sinh(powcoeff(_s,0))
         ),  
         PS[powint](
            PS[multiply](s,PS[powdiff](_s)),
            cosh(powcoeff(_s,0))
         )
      )),
      variable(s)
   )[1]
end:
PS[powcosh] := proc(s);
   PS['fixedpoint'](subs([_s=s], 
      (s,c) -> ( 
         PS[powint](
            PS[multiply](c,PS[powdiff](_s)), 
            sinh(powcoeff(_s,0))
         ),  
         PS[powint](
            PS[multiply](s,PS[powdiff](_s)),
            cosh(powcoeff(_s,0))
         )
      )),
      variable(s)
   )[2]
end:

PS[powtanh] := proc(s) local t1, t2;
   t1 := PS[powexp](s);
   t2 := PS[powexp](PS[negative](s));
   PS[quotient](PS[subtract](t1,t2), PS[add](t1,t2))
end:

# evaluation of expressions including powerserues:
###################################################

PS[evalpow] := proc (expr) local var, n, d, e, f, f1, f2;
   if nargs=2 then 
      var := args[2] 
   else
      var := map(variable, indets(expr, powerseries));
      if nops(var) > 1 then ERROR(`series in more than one variable`) else var := op(var) fi;
   fi;

   if not has(expr, _powerseries) then powcreate(expr, var)
   elif type(expr, powerseries) then expr
   elif type(expr, `+`) then
      PS[add](op(map(procname, [op(expr)], var)))
   elif type(expr, `*`) then
      if not (has(op(1,expr), _powerseries) or has(op(1,expr), var)) then # this is a constant
         PS[multconst](procname(expr/op(1,expr),var), op(1,expr))
      else
         n := numer(expr); d := denom(expr);
         if d <> 1 then 
            PS[quotient](procname(n,var), procname(d,var))
         else
            PS[multiply](op(map(procname, [op(expr)], var)))
         fi;
      fi
   elif type(expr, `^`) then e := op(2,expr);
      if has(e,var) then
         PS[`^`](procname(op(1,expr), var), procname(op(2,expr), var))
      else
         PS[`^`](procname(op(1,expr), var), e)
      fi;
   elif type(expr, function) then f := op(0,expr);
      if type(f, powerseries) then
         PS[compose](f, procname(op(1,expr),var))
      elif type(f, name) then 
         f1 := `pow`.(substring(f,2..length(f))):
         f2 := `pow`.f:
         if   assigned(PS[f]) then
            PS[f ](op(map(procname, [op(expr)], var)))
         elif assigned(PS[f1]) then
            PS[f1](op(map(procname, [op(expr)], var)))
         elif assigned(PS[f2]) then
            PS[f2](op(map(procname, [op(expr)], var)))
         elif f='prev' then PS[reversion](op(map(procname, [op(expr)], var)))
         elif f='pinv' then PS[inverse]  (op(map(procname, [op(expr)], var)))
         elif f='pquo' then PS[quotient] (op(map(procname, [op(expr)], var)))
         elif f='pmul' then PS[multiply] (op(map(procname, [op(expr)], var)))
         elif f='psub' then PS[subtract] (op(map(procname, [op(expr)], var)))
         elif f='pneg' then PS[negative] (op(map(procname, [op(expr)], var)))
         else
            ERROR(`unknown function `,f)
         fi
      else ERROR(`dont know`);
      fi
   else ERROR(`dont know`)
   fi
end:

# solving of ordinary differential equations:
##############################################

PS[ode] := proc(f, C) local var,i;
# y[n] = f(x,y,y', ..., y[n-1]), y(0)=C[1], y'(0)=C[2], y''(0)=C[3], y[n-1](0)=C[n]
# Examples:
# ode((x,y) -> y, [1], x);
# ode((x,y,yp,ypp,yppp) -> y, [3/2,-1/2,-3/2,1/2], x);
# ode( (x,y) -> y/(y*y+x), [2], x);
# ode( (x,y,yp) -> 1/(1+y^2), [0,1] );
   if nargs=3 then var := args[3] else var := 'x' fi:
   PS['fixedpoint'](
      subs(['_f'=f, '_C'=C, '_x'=var],
         y -> odeprod(_f, _C, _x, y, nops(_C))),
      var,
      [seq(C[i]/(i-1)!, i=1..nops(C))]
   )
end:
odeprod := proc(f, C, x, y, n) local argseq, yp;
   if C=[] then
      argseq := y; yp := y;
      to n-1 do
         yp := PS[powdiff](yp);
         argseq := argseq, yp;
      od;
      PS[evalpow](f(x, argseq), x);
   else
      PS[powint](
         odeprod(f, C[2..nops(C)], x, y, n),
         C[1]
      )
   fi;
end:

#save PS,
#     powcreate1, 
#     powcreate2,
#     variable,
#     odeprod,
#     fix,
#     `print/_powerseries`,
#     `type/powerseries`,
#
#     `PS.m`;
#
#quit
