## <SHAREFILE=numerics/trans/trans.mpl >
## <DESCRIBE>
##		 (update)
##               SEE ALSO: numerics/trans/trans.tex 
##               Package of routines for rational function approximations
##               to rational points, Taylor and asymptotic series, and functions
##               Includes special numerical versions of the
##               approximation algorithms to use the performance of the
##               hardware floating point arithmetic.
##               AUTHOR: Johannes Grotendorst, j.grotendorst@kfa-juelich.de
## </DESCRIBE>
## <UPDATE=R4update >

##########################
# standard input
###
# Convert from 5.2 to Release 3
#===============================================================================
#                          Program TRANS
#===============================================================================
#
#             A Maple package for transforming sequences, series
#                                and functions
#
#
#    Copyright (C) 1990 by J. Grotendorst, all rights reserved.
#
#    Error reports please to: Johannes Grotendorst
#                             Zentralinstitut fuer Angewandte Mathematik
#                             Forschungszentrum Juelich GmbH
#                             D-52425 Juelich
#                             Federal Republic of Germany
#                    E-mail : j.grotendorst@kfa-juelich.de
#
#
#        modifications: 24 July 90: - generation of FORTRAN functions
#                                     with parameters
#                        8 Aug  90: - numerical evaluation of Maple
#                                     constants (Pi, E, ...) when
#                                     generating a FORTRAN function
#                          Mar  91  - neccesary changes to use the
#                                     fortran function in MapleV
#                                   - rename and restructure of procedures
#                                     tay (now ratser) and asy (now ratgen)
#                                   - ratser applies to expansions with
#                                     data type series
#                                   - ratgen applies to generalized expansions
#                                     with data type `+`, e.g. asymptotic
#                                     expansions
#                          Oct 91   - changes due to the Maple package concept
#                          Sep 92   - special versions of the transformation
#                                     algorithms for use with evalhf
#                          Nov 93   - now ratser accepts expressions as input,
#                                     input checking is modified due to the 
#                                     automatic type checking facility in 
#                                     Rel. 2, driver programs for the numerical
#                                     transformation of series and sequences 
#                                     are included  
#===============================================================================
#  Maple V Release 2 is assumed
#===============================================================================
#
trans :=
table ([ratser    = proc () `trans/ratser` (args) end,
	ratgen    = proc () `trans/ratgen` (args) end,
	aitken    = proc () `trans/aitken_num` (args) end,
	epsilon   = proc () `trans/eps_num` (args) end,
	gb        = proc () `trans/gb_num` (args) end,
	lev       = proc () `trans/lev_num` (args) end,
	rho       = proc () `trans/rho_num` (args) end,
	rhoit     = proc () `trans/rhoit_num` (args) end,
	rich      = proc () `trans/rich_num` (args) end,
	sidi      = proc () `trans/sidi_num` (args) end,
	theta     = proc () `trans/theta_num` (args) end,
	thetait   = proc () `trans/thetait_num`  (args) end
	]):
`trans/ratser` := proc (f::algebraic, eqn::{`=`,name})
global  precision, `trans/ratser`;
#*******************************************************************************
#
#  procedure:   ratser
#
#  Description:
#
#  ratser transforms series expansions with data type "series" via
#  well-known linear as well as nonlinear sequence (series) transformations.
#  Generation of optimized FORTRAN code (functions) is supported.
#
#  Input parameters :
#
#  obligatory:
#
#  f           : an expression representing the function to be approximated
#  eqn         : equation x=a (if a is not specified then the expansion is
#                about the point x=0)
#
#  optional:
#
#  <norder>    : order of the series expansion used for the transformation
#                                         (Default: 6)
#  <typ>       : type of transformation: u, t, d, v, sh, gb, rh, r, at,
#                rhit, th, thit, su, st, sd, sv;
#                in case of Richardson extrapolation (r) or
#                rho transformation (rh) or
#                iterated rho_2 transformation (rhit)
#                a list of auxiliary parameters x_n can be specified.
#                (Default : x_n = 1/(n+1) for Richardson extrapolation
#                and x_n = n+1 for rho and rho_2 transformation).
#                Then the first element of the list characterizes
#                the type of transformation, i.e. it must be equal to
#                r, rh or rhit (Default: u)
#  <expts>     : list of real expansion points for the rational approximations
#                                         (Default: [ ] )
#  <sw>        : flag for the generation of a FORTRAN function: fort or nofort
#                                         (Default: nofort)
#  <precision> : type of the FORTRAN function: single or double
#                                         (Default : single)
#  <evalorder> : computational scheme: horner or confrac (Default : horner)
#
#  <fn>        : filename for the FORTRAN function (Default: `` =  terminal)
#
#*******************************************************************************
#
local norder ,typ, i, floats, x, s, s1, ndim, app, resnum, resden,
      x0, x1, x2, expts, resarb, nexp, a, na, fn, exp1, expn, expk1,
      expk2, j, k, head, head1, head2, ndig, exp1m, exp1e, expnm, expne,
      expk1m, expk1e, expk2m, expk2e, echar, sw, evalorder, boolist,
      nlist, laux, typ1, indarg, indarg1, l, lcoeden,
      degnum, degden, appnum, appden, y, tra, old_precision;
#
#  initializations
#
expts     := [ ];
laux      := [ ];
norder    := 7;
typ       := 'u';
fn        := ``;
sw        := 'nofort';
evalorder := ``;
ndig      := 10;
echar     := `.E`;
a         := array ('sparse', 1 .. 8, []);
if assigned (precision) then
  old_precision := precision
else
  old_precision := 'precision'
fi;
precision := 'single';
#
# check the formal parameter eqn 
#
if type (eqn, `=`) then
   if type (op(1, eqn), name) then
      x1 := op (1, eqn);
      x2 := op (2, eqn)
   else
      ERROR (`expecting a variable of data type name`)
   fi:
else
   x1 := eqn;
   x2 := 0
fi:
#
#  check the optional parameters
#
if nargs < 1 or 9 < nargs then
   ERROR (`wrong number of arguments`)
elif nargs > 2 then
#
# check the optional parameters
#
   for i from 3 to nargs do
      if (args [ i ] = 'u') or (args [ i ]  = 't')
	 or (args [ i ] = 'v') or (args [ i ] = 'd')
	 or (args [ i ] = 'su') or (args [ i ] = 'st')
	 or (args [ i ] = 'sd') or (args [ i ] = 'sv')
	 or (args [ i ] = 'sh')
	 or (args [ i ] = 'gb') or (args [ i ] = 'rh')
	 or (args [ i ] = 'r') or (args [ i ] = 'at')
	 or (args [ i ] = 'rhit') or (args [ i ] = 'th')
	 or (args [ i ] = 'thit')
	 and type (args [ i ], name) then
	 if a[3] = 0 then
	    a[1] := 1;
	 fi;
	 typ  := args [ i ];
      elif type (args [ i ], integer) and args [ i ] > 0 then
	 a[2] := 1;
	 norder := args [ i ] + 1;
      elif type (args [ i ], list) then
	 nlist := nops (args [ i ]);
	 if nlist > 0 then
	    typ1 := op (1, args [ i ]);
	    if typ1 = 'r' or typ1 = 'rh' or typ1 = 'rhit' then
	       typ  := op (1, args [ i ]);
	       if a[1] = 0 then
		  a[3] := 1;
	       fi;
	       laux := [seq(op (j, args [ i ]), j = 2 .. nlist)];
	    else
	       a[4]    := 1;
	       boolist := map (type, args [ i ], realcons);
	       if type (x2, realcons) then
		  ERROR (`expansion point of the series must be indeterminate`)
	       elif has (boolist, false) then
		  ERROR (`expansion points have to be of type real constant`)
	       fi;
	       expts := args [ i ];
	    fi;
	 fi;
      elif (args [ i ] = 'fort') or (args [ i ] = 'nofort')
	 and type (args [ i ], name) then
	 a[5]   := 1;
	 sw     := args [ i ];
      elif (args [ i ] = 'single') or (args [ i ] = 'double')
	 and type (args [ i ], name) then
	 a[6]      := 1;
	 precision := args [ i ];
	 if (precision = 'double') then
	    ndig      := 20;
	    echar     := `.D`;
	 fi;
      elif (args [ i ] = `horner`) or (args [ i ] = `confrac`)
	 and type (args [ i ], name) then
	 a[7]      := 1;
	 evalorder := args [ i ];
      elif type (args [ i ], name) then
	 a[8] := 1;
	 fn   := args [ i ];
      else
	 ERROR (`invalid argument found`)
      fi
   od
fi:
#
#  check the number of input arguments
#
na := sum (a ['i'], 'i' = 1 .. 8) + 2;
#
if na <> nargs then
   ERROR (`invalid argument found`)
fi:
#
# end of input check
# series expansion
#
s := series (f, eqn, norder);
#
# eliminate the order symbol
#
s1 := subs (O(1) = 0, s);
if s1 = 0 then
   ERROR (`series expansion is 0`)
fi:   
#
# check whether the series contains coefficients with data type "float"
#
x := op (0, s1);
if hastype (s1, float) or type (x, float) then
   x0 := x;
   x  := map (convert, x, rational);
   s1 := subs (x0 = x, s1);
   s1 := map (convert, s1, rational);
   floats := true;
else
   floats := false;
fi;
#
# determination of the number terms of the series
#
ndim := nops(s1)/2 - 1;
#
# check the parameters
#
if typ = 'rh' or typ = 'rhit' or typ = 'r' then
   if nops (laux) = 0 then
      if typ = 'rh' or typ = 'rhit' then
	 laux := [seq (k + 1, k = 0 .. ndim)]
      elif typ = 'r' then
	 laux := [seq (1 / (k + 1), k = 0 .. ndim)]
      fi
   elif nops (laux) > 0 and nops (laux) <= ndim then
      if typ = 'rh' then
	 ERROR (`too few parameters for the rho transformation`)
      elif typ = 'rhit' then
	 ERROR (`too few parameters for the iterated rho_2 transformation`)
      elif typ = 'r' then
	 ERROR (`too few parameters for the Richardson exptrapolation`)
      fi
   fi
fi;
#
# perform the specific series transformation
#
app := frontend (`trans/series_transform`, [s1, y, ndim, typ, laux],
		 [{`+`, `*`, list, series},{}]);
#
# normalization of the rational approximation
#
appnum  := expand (numer (app));
appden  := expand (denom (app));
degnum  := degree (appnum, y);
degden  := degree (appden, y);
lcoeden := tcoeff (appden, y);
#
if type (lcoeden, numeric) then
   resnum  := series (appnum / lcoeden, y, degnum + 1);
   resden  := series (appden / lcoeden, y, degden + 1)
else
   resnum  := series (appnum, y, degnum + 1);
   resden  := series (appden, y, degden + 1)
fi;
#
# indeterminants for the output expression
#
nexp := nops (expts);
if type (x2, name) and nexp > 0 then
   indarg  := indets (app, string) minus {constants, y, x2}
else
   indarg  := indets (app, string) minus {constants, y}
fi;
#
indarg1 := [x1, seq (l, l = indarg minus {x1})];
#
resnum := convert (subs (op (0, resnum) = x, resnum), polynom);
resden := convert (subs (op (0, resden) = x, resden), polynom);
#
if evalorder = `horner` then
   resnum  := convert (resnum, horner, indarg1);
   resden  := convert (resden, horner, indarg1);
   resarb  := resnum / resden
elif evalorder = `confrac` then
   resarb  := convert (resnum / resden, confrac, x1)
else
   resarb  := resnum / resden
fi;
#
# return the result
#
if sw = 'nofort' then
#
#  check whether the result contains a floating point number
#
   if floats then
      resarb := evalf (resarb)
   fi;
#  return a sequence of approximations
#
   if nexp > 0 then
      resarb := seq (subs (x2 = expts [ k ], resarb), k = 1 .. nexp)
   fi;
#
#  insert the actual arguments as table index and the result as
#  table value into the remember table
#
   `trans/ratser` (args) := resarb;
#
   precision := old_precision;
#
   RETURN (resarb)
else
#
#  determination of the fortran function header
#
   `trans/fortran_header` (typ, indarg1, precision, 'head', 'tra');
   head1 := head [1];
   head2 := head [2];
#
#  write the fortran function
#
   if nexp = 1 then
      if (fn <> ``) then writeto(fn) fi;
      lprint(`      `.head1);
      lprint(`      `.head2);
      fortran([tra = evalf (subs (x2 = expts [ 1 ], resarb), ndig)]);
      lprint(`      end`);
      if (fn <> ``) then writeto('terminal') fi;
   elif nexp > 1 then
      exp1  := evalf (expts [ 1 ] + (expts [ 2 ] -
	       expts [ 1 ]) / 2, 3);
      expn  := evalf (expts [ nexp - 1 ] + (expts [ nexp ] -\
	       expts [ nexp - 1 ]) / 2, 3);
      exp1m := op (1, exp1);
      expnm := op (1, expn);
#
      if exp1 = 0 then
	 exp1e := 0
      else
	 exp1e := op (2, exp1)
      fi;
#
      if expn = 0 then
	 expne := 0
      else
	 expne := op (2, expn)
      fi;
#
      if nexp = 2 then
	 if (fn <> ``) then writeto(fn) fi;
	 lprint(`      `.head1);
	 lprint(`      `.head2);
	 lprint(`      if (`.x1.` .le. `.exp1m.echar.exp1e.`) then `);
	 fortran([tra = evalf (subs (x2 = expts [ 1 ], resarb), ndig)]);
	 lprint(`      elseif (`.x1.` .gt. `.expnm.echar.expne.`) then `);
	 fortran([tra = evalf (subs (x2 = expts [ nexp ], resarb),
			  ndig)]);
	 lprint(`      end if`);
	 lprint(`      end`);
	 if (fn <> ``) then writeto('terminal') fi;
      elif nexp > 2 then
	 if (fn <> ``) then writeto(fn) fi;
	 lprint(`      `.head1);
	 lprint(`      `.head2);
	 lprint(`      if (`.x1.` .le. `.exp1m.echar.exp1e.`) then `);
	 fortran([tra = evalf (subs (x2 = expts [ 1 ], resarb), ndig)]);
	 for k from 2 to (nexp - 1) do
	    expk1  := evalf (expts [ k - 1 ] + (expts [ k ] -\
		      expts [ k - 1 ]) / 2, 3):
	    expk2  := evalf (expts [ k ] + (expts [ k + 1 ] -\
		      expts [ k ]) / 2, 3):
	    expk1m := op (1, expk1):
	    expk2m := op (1, expk2):
#
	    if expk1 = 0 then
	       expk1e := 0
	    else
	       expk1e := op (2, expk1)
	    fi:
#
	    if expk2 = 0 then
	       expk2e := 0
	    else
	       expk2e := op (2, expk2)
	    fi:
#
	    lprint(`      elseif (`.x1.` .gt. `.expk1m.echar.
	    expk1e.` .and. `.x1.` .le. `.expk2m.echar.expk2e.`) then`);
	    fortran([tra = evalf (subs (x2 = expts [ k ], resarb),
			   ndig)])
	 od;
	 lprint(`      elseif (`.x1.` .gt. `.expnm.echar. expne.`) then `);
	 fortran([tra = evalf (subs (x2 = expts [ nexp ], resarb),
		 ndig)]);
	 lprint(`      end if`);
	 lprint(`      end`);
	 if (fn <> ``) then writeto('terminal') fi;
      fi
   else
      if (fn <> ``) then writeto(fn) fi;
      lprint(`      `.head1);
      lprint(`      `.head2);
      fortran([tra = evalf (resarb, ndig)]);
      lprint(`      end`);
      if (fn <> ``) then writeto('terminal') fi
   fi;
#
   precision := old_precision;
#
   RETURN ()
fi
end:
`trans/fortran_header` := proc (typ, parlist, prec, head12, nam)
#******************************************************************************
#
#  description:
#
#  This procedure evaluates the FORTRAN function header
#
#  input parameters:
#
#  typ     : type of transformation
#  parlist : list with the formal parameters for the fortran function
#  prec    : precision type of the fortran function
#  head12  : line 1 and 2 of the fortran function
#  nam     : name of the fortran function
#
#******************************************************************************
#
local parg, j, nam1, namtr, b;
#
# function name
#
namtr :='tr';
nam1  := typ.namtr;
#
parg   := ``;
for j from 1 to nops (parlist)
do
   b[j] := op (j, parlist);
   if j = nops (parlist) then
      parg := cat (parg, b[j])
   else
      parg := cat (parg, b[j], `, `)
   fi;
od;
#
head12 [1] := `function `.nam1.` (`.parg.`)`;
#
# declaration for the parameters
#
if prec = 'single' then
   head12 [2] := `real `.nam1.`, `.parg
elif prec = 'double' then
   head12 [2] := `double precision `.nam1.`, `.parg
fi;
#
nam := nam1;
#
end:
`trans/series_transform` := proc (s, ep, nterm, typ, param)
#******************************************************************************
#
# description:
#
# This procedure performs the sequence transformation by calling
# the specific transformation procedure seccessively.
#
# input parameters:
#
# s    :  series expansion of type "series" or `+`
# ep   :  variable for "x - x0"
# nterm:  number of series terms
# typ  :  type of transformation
# param:  list with parameters for the rho, rhoit or r transformation
#
#******************************************************************************
#
local s_n, a_n, x_n, n, arr, xarr, ratapp;
#
# initialization of the first partial sum
#
s_n := 0;
#
#  loop over the sequence of partial sums
#
for n from 0 to nterm do
   if type (s, series) then
      a_n := op (2 * n + 1, s) * ep ** op (2 * n + 2, s)
   elif (type (s, `+`) or nops (s) = 1) then
      a_n := op (n + 1, s)
   else
      ERROR (`series type is wrong`)
   fi;
#
   s_n := s_n + a_n;
#
#  call the algorithm to perform the specified transformation
#
   if typ = 'rh' then
      x_n := op (n + 1, param);
      ratapp  := `trans/rho_sym` (s_n, x_n, n, arr, xarr)
   elif typ = 'sh' then
      ratapp  := `trans/eps_sym` (s_n, n, arr)
   elif typ = 'u' or typ = 't' or typ = 'd' or typ = 'v' then
      ratapp  := `trans/lev_sym` (s_n, a_n, n, arr, xarr, typ)
   elif typ = 'su' or typ = 'st' or typ = 'sd' or typ = 'sv' then
      ratapp  := `trans/sidi_sym` (s_n, a_n, n, arr, xarr, typ)
   elif typ = 'r' then
      x_n := op (n + 1, param);
      ratapp  := `trans/rich_sym` (s_n, x_n, n, arr, xarr)
   elif typ = 'gb' then
      ratapp := `trans/gb_sym` (s_n, a_n, n, arr, xarr)
   elif typ = 'at' then
      ratapp  := `trans/aitken_sym` (s_n, n, arr)
   elif typ = 'rhit' then
      x_n := op (n + 1, param);
      ratapp  := `trans/rhoit_sym` (s_n, x_n, n, arr, xarr)
   elif typ = 'th' then
      ratapp  := `trans/theta_sym` (s_n, n, arr, xarr)
   elif typ = 'thit' then
      ratapp  := `trans/thetait_sym` (s_n, n, arr)
   fi;
od;
#
RETURN (ratapp);
end:
`trans/ratgen` := proc (s::`+`)
global  precision, `trans/ratgen`;
#*******************************************************************************
#
#  procedure:   ratgen
#
#  Description:
#
#  ratgen transforms generalized series expansions, e.g. asymptotic
#  expansions, with data type `+` via well-known linear as well as
#  nonlinear sequence (series) transformations. Generation of
#  optimized FORTRAN code (functions) is supported.
#
#  Input parameters :
#
#  obligatory:
#
#  s           : generalized series of dataype `+`
#
#  optional:
#
#  <typ>       : type of transformation: u, t, d, v, sh, gb, rh, r, at,
#                rhit, th ,thit, su, st, sd, sv;
#                in case of Richardson extrapolation (r) or
#                rho transformation (rh) or
#                iterated rho_2 transformation (rhit)
#                a list of auxiliary parameters x_n can be specified.
#                (Default : x_n = 1/(n+1) for Richardson extrapolation
#                and x_n = n+1 for rho and rho_2 transformation).
#                Then the first element of the list characterizes
#                the type of transformation, i.e. it must be equal to
#                r, rh or rhit (Default: u)
#  <norder>    : order of transformation  (Default: number of terms of the
#                                         generalized series)
#  <sw>        : flag for the generation of a FORTRAN function: fort or nofort
#                                         (Default: nofort)
#  <precision> : type of the FORTRAN function: single or double
#                                         (Default: single)
#  <evalorder> : computational scheme: horner (Default: ``)
#
#  <fn>        : filename for FORTRAN function (Default: `` = terminal)
#
#*******************************************************************************
#
local s1, norder ,typ, typ1, i, j, k, floats, ndim1, ndim, app, resnum, resden,
      resarb, a, na, fn, ndig, sw, evalorder, lterm, nd1, nlist, laux,
      indarg, y, head, head1, head2, tra, old_precision;
#
# initializations
#
laux      := [ ];
norder    := 0;
typ       := 'u';
fn        := ``;
sw        := 'nofort';
evalorder := ``;
ndig      := 10;
a         := array ('sparse', 1 .. 7, []);
if assigned (precision) then
  old_precision := precision
else
  old_precision := 'precision'
fi;
precision := 'single';
#
#  check the input parameters
#
if nargs < 1 or 7 < nargs then
   ERROR (`wrong number of arguments`)
elif nargs > 1 then
   for i from 2 to nargs do
      if (args [ i ] = 'u') or (args [ i ]  = 't')
         or (args [ i ] = 'd') or (args [ i ] = 'v')
         or (args [ i ] = 'su') or (args [ i ] = 'st')
         or (args [ i ] = 'sd') or (args [ i ] = 'sv')
         or (args [ i ] = 'sh')
         or (args [ i ] = 'gb') or (args [ i ] = 'rh')
         or (args [ i ] = 'r') or (args [ i ] = 'at')
         or (args [ i ] = 'rhit') or (args [ i ] = 'th')
         or (args [ i ] = 'thit')
         and type (args [ i ], name) then
         if a[3] = 0 then
            a[1] := 1;
         fi;
         typ  := args [ i ];
      elif type (args [ i ], integer) and args [ i ] > 0 then
         a[2] := 1;
         norder := args [ i ];
      elif type (args [ i ], list) then
         nlist := nops (args [ i ]);
         if nlist > 0 then
            typ1 := op (1, args [ i ]);
            if typ1 = 'r' or typ1 = 'rh' or typ1 = 'rhit' then
               typ  := op (1, args [ i ]);
               if a[1] = 0 then
                  a[3] := 1;
               fi;
               laux := [seq (op (j, args [ i ]), j = 2 .. nlist)];
            fi;
         fi;
      elif (args [ i ] = 'fort') or (args [ i ] = 'nofort')
         and type (args [ i ], name) then
         a[4]   := 1;
         sw     := args [ i ];
      elif (args [ i ] = 'single') or (args [ i ] = 'double')
         and type (args [ i ], name) then
         a[5]   := 1;
         precision := args [ i ];
         if (precision = 'double') then
            ndig      := 20;
         fi;
      elif (args [ i ] = `horner`)
         and type (args [ i ], name) then
         a[6]        := 1;
         evalorder   := args [ i ];
      elif type (args [ i ], name) then
         a[7] := 1;
         fn   := args [ i ];
      else
         ERROR (`invalid argument found`)
      fi;
   od;
#
   na := sum (a ['i'] ,'i' = 1 .. 7) + 1;
#
#  check the number of input parameters
#
   if na <> nargs then
      ERROR (`invalid argument found`)
   fi
fi;
#
# eliminate the order symbol
#
lterm := op (nops(s),s);
if type(lterm, function) then
   if op(0, lterm) = O then
      s1 := subs (lterm = 0, s);
      if nops (s) = 2 then
         s1 := [s1]
      fi;
   fi;
else
   s1 := s
fi;
#
# check whether the series contains coefficients with data type float
#
if hastype (s1, float) then
   s1     := map (convert, s1, rational);
   floats := true;
else
   floats := false;
fi;
#
# determinate the number of terms and check the specified norder
#
ndim1 := nops (s1) - 1;
#
if norder = 0 then
   ndim := ndim1;
else
   if norder <= ndim1 then
      ndim := norder;
   else
      ERROR (`series order too small for specified transformation order`)
   fi;
fi;
#
# check the transformation parameters
#
if typ = 'gb' and ndim < ndim1 then
   nd1 := ndim + 1
else
   nd1 := ndim
fi;
#
if typ = 'rh' or typ = 'rhit' or typ = 'r' then
   if nops (laux) = 0 then
      if typ = 'rh' or typ = 'rhit' then
         laux := [seq (k + 1, k = 0 .. nd1)]
      elif typ = 'r' then
         laux := [seq (1 / (k + 1), k = 0 .. nd1)]
      fi
   elif nops (laux) > 0 and nops (laux) <= nd1 then
      if typ = 'rh' then
         ERROR (`too few parameters for the rho transformation`)
      elif typ = 'rhit' then
         ERROR (`too few parameters for the iterated rho_2 transformation`)
      elif typ = 'r' then
         ERROR (`too few parameters for the Richardson exptrapolation`)
      fi
   fi
fi;
#
#  perform the specific series transformation
#
app := frontend (`trans/series_transform`, [s1, y, ndim, typ, laux],
                 [{`+`, `*`, list, series},{}]);
#
#  determinate the output form
#
resnum := expand (numer (app));
resden := expand (denom (app));
#
#  determination of the indeterminats for the output expresssion
#
indarg  := indets (app, string) minus {constants};
#
#  convert into Horner scheme
#
if evalorder = `horner` then
   resnum  := convert (resnum, horner, indarg);
   resden  := convert (resden, horner, indarg);
   resarb  := resnum / resden
else
   resarb := resnum / resden
fi;
#
# return the result
#
if sw = 'nofort' then
#
#  check whether the result contains a floating-point number
#
   if floats then
      resarb := evalf (resarb)
   fi;
#
#  insert the actual arguments as table index and the result as
#  table value into the remember table
#
   `trans/ratgen` (args) := resarb;
#
   precision := old_precision;
#
   RETURN (resarb)
else
#
#  determination of the FORTRAN function header
#
   `trans/fortran_header` (typ, indarg, precision, 'head', 'tra');
   head1 := head [1];
   head2 := head [2];
#
#  write the fortran function
#
   if (fn <> ``) then writeto(fn) fi;
   lprint(`      `.head1);
   lprint(`      `.head2);
   fortran([tra = evalf (resarb, ndig)]);
   lprint(`      end`);
   if (fn <> ``) then writeto('terminal') fi;
#
   precision := old_precision;
#
   RETURN()
fi
end:
`trans/eps_num` := proc(f::algebraic, eqn::{`=`, name})
#*************************************************************************
#
# Description:
#
# This procedure is a driver program for transforming series
# or sequences numerically using the Shanks transformation.
#
# Input parameters:
#
# f     : sequence element s_n
# eqn   : equation n=min..max: print n, s_n and T(s_n)
#         from min up to max (Default: n=0..10, if min..max is
#         not specified)
#
# optional:
#
# arith : arithmetic system: h (hardeware floating point system)
#                            s (Maple infinite precision arithmetic)
#                            hs (h + s)
#                            (Default: out = h)
# tseq  : type of sequence: sump (summation) or seqp (sequence)
#         (Default: tseq = sump)
# ndig  : number of digits (Default: ndig = 16)
#
#*************************************************************************
#
local sum1, term1, arr1, arr2, i, j, m, n, eqn_lhs, eqn_rhs,
      nmin, nmax, arith, nout, app, tseq, t1, ndig;
#
# initialization
#
sum1  := 0:
nmin  := 0:
nmax  := 10:
ndig  := 16:
arith := h:
nout  := 3:
tseq  := sump:
#
# check input parameters
#
if type (eqn, `=`) then
   eqn_lhs := op (1, eqn);
   eqn_rhs := op (2, eqn);
   if type (eqn_lhs, name) then
      i := eqn_lhs
   else
       ERROR (`invalid argument found`)
   fi;
   if type (eqn_rhs, `..`) and
      type (op(1, eqn_rhs), integer)
      and op(1, eqn_rhs) >= 0 and
      type (op(2, eqn_rhs), integer) and
      op(2, eqn_rhs) >= 0 then
      nmin := op(1, eqn_rhs);
      nmax := op(2, eqn_rhs);
      if nmax < nmin then
         ERROR (`lower bound > upper bound`)
      fi;
   else
      ERROR (`invalid argument found`)
   fi;
elif type (eqn , name) then
   i := eqn
fi;
#
if nargs > 5 then
   ERROR (`wrong number of arguments`)
elif nargs > 2 then
   for n from 3 to nargs do
      if type (args [n], name) and
         (args[n] = h or args[n] = s or args[n] = hs) then
         arith := args [n]
      elif type (args [n], name) and
         (args[n] = sump or args[n] = seqp) then
         tseq := args [n]
      elif type (args[n], integer) and
         args [n] >= 0 then
         ndig := args [n]
      else
         ERROR (`invalid argument found`)
      fi
   od
fi;
#
# end of input checking
#
arr1 := array (0 .. nmax):
arr2 := array (0 .. nmax):
Digits := ndig:
#
for j from 0 to nmax do
   if type (f, procedure) then
      term1 := f (j)
   elif type (f, array) then
      term1 := f [j]
   else
      term1 := subs(i = j, f)
   fi:
   if not type (term1, complex) then
      ERROR(`invalid sequence element found`)
   fi;
   if tseq = sump then
      sum1  := sum1 + term1:
      t1 := sum1
   else
      t1 := term1
   fi;
   app[1] := j:
   app[2] := evalhf (t1):
   if arith = h then	
      app[3] := evalhf (`trans/eps` (t1, j, var(arr1)))
   elif arith = s then
      app[3] := evalf (`trans/eps` (t1, j, arr2))
   elif arith = hs then
      nout   := 4:
      app[3] := evalhf (`trans/eps` (t1, j, var(arr1))):
      app[4] := evalf (`trans/eps` (t1, j, arr2))
   fi;
   if j >= nmin then
      print(seq (app[m], m = 1 .. nout))
   fi;
od
end:
`trans/eps_sym` := proc (s_k, k, arr)
#***********************************************************************
#
#  Procedure  : eps
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 18 june 1990
#  Update     : 1 july 1990
#               normalize intermediate expressions
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  The epsilon algorithm due to Wynn is used to perform Shanks'
#  series transformation.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  k    : number of the last element
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         the epsilon table
#
#***********************************************************************
#
local j, aux1, aux2, diff1;
#
# initialization
#
arr [ k ]    := s_k;
#
# recursive computation of the epsilon table
#
if k = 0 then
   RETURN (s_k)
else
   aux2 := 0;
   for j from k by -1 to 1
   do
      aux1          := aux2;
      aux2          := arr [ j - 1 ];
      diff1         := normal (arr [ j ] - aux2);
      arr [ j - 1 ] := normal (aux1 + 1 / diff1)
   od
fi;
#
if modp (k, 2) = 0 then
   RETURN (arr [ 0 ])
else
   RETURN (arr [ 1 ])
fi
end:
`trans/eps` := proc (s_k, k, arr)
#***********************************************************************
#
#  Procedure  : eps
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 24 september 1992
#               version of eps for numerical transformation
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  The epsilon algorithm due to Wynn is used to perform Shanks'
#  series transformation.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  k    : number of the last element
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         the epsilon table
#
#***********************************************************************
#
local j, aux1, aux2, diff1;
#
# initialization
#
arr [ k ]    := s_k;
#
# recursive computation of the epsilon table
#
if k = 0 then
   RETURN (s_k)
else
   aux2 := 0;
   for j from k by -1 to 1
   do
      aux1          := aux2;
      aux2          := arr [ j - 1 ];
      diff1         := arr [ j ] - aux2;
      if evalf (abs (diff1)) < evalhf (DBL_MIN) then
         arr [j - 1] := evalhf (DBL_MAX)
      else
         arr [j - 1] := aux1 + 1 / diff1
      fi
   od
fi;
#
if modp (k, 2) = 0 then
   RETURN (arr [0])
else
   RETURN (arr [1])
fi
end:
`trans/rho_num` := proc(f::algebraic, x::algebraic, eqn::{`=`, name})
#*************************************************************************
#
# Description:
#
# This procedure is a driver program for transforming series
# or sequences numerically using the rho transformation.
#
# Input parameters:
#
# f     : sequence element s_n
# x     : element x_n of the auxiliary sequence
# eqn   : equation n=min..max: print n, s_n and T(s_n)
#         from min up to max (Default: n=0..10, if min..max is
#         not specified)
#
# optional:
#
# arith : arithmetic system: h (hardeware floating point system)
#                            s (Maple infinite precision arithmetic)
#                            hs (h + s)
#                            (Default: out = h)
# tseq  : type of sequence: sump (summation) or seqp (sequence)
#         (Default: tseq = sump)
# ndig  : number of digits (Default: ndig = 16)
#
#*************************************************************************
#
local sum1, term1, arr1, arr2, arr3, arr4, i, j, m, n, eqn_lhs, eqn_rhs,
      nmin, nmax, arith, nout, app, tseq, t1, xterm1, ndig;
#
# initialization
#
sum1  := 0:
nmin  := 0:
nmax  := 10:
ndig  := 16:
arith := h:
nout  := 3:
tseq  := sump:
#
# check input parameters
#
if type (eqn, `=`) then
   eqn_lhs := op (1, eqn);
   eqn_rhs := op (2, eqn);
   if type (eqn_lhs, name) then
      i := eqn_lhs
   else
       ERROR (`invalid argument found`)
   fi;
   if type (eqn_rhs, `..`) and
      type (op(1, eqn_rhs), integer)
      and op(1, eqn_rhs) >= 0 and
      type (op(2, eqn_rhs), integer) and
      op(2, eqn_rhs) >= 0 then
      nmin := op(1, eqn_rhs);
      nmax := op(2, eqn_rhs);
      if nmax < nmin then
         ERROR (`lower bound > upper bound`)
      fi;
   else
      ERROR (`invalid argument found`)
   fi;
elif type (eqn , name) then
   i := eqn
fi;
#
if nargs > 6 then
   ERROR (`wrong number of arguments`)
elif nargs > 3 then
   for n from 4 to nargs do
      if type (args [n], name) and
         (args[n] = h or args[n] = s or args[n] = hs) then
         arith := args [n]
      elif type (args [n], name) and
         (args[n] = sump or args[n] = seqp) then
         tseq := args [n]
      elif type (args[n], integer) and
         args [n] >= 0 then
         ndig := args [n]
      else
         ERROR (`invalid argument found`)
      fi
   od
fi;
#
# end of input checking
#
arr1   := array (0 .. nmax):
arr2   := array (0 .. nmax):
arr3   := array (0 .. nmax):
arr4   := array (0 .. nmax):
Digits := ndig:
#
for j from 0 to nmax do
   if type (f, procedure) then
      term1 := f (j)
   elif type (f, array) then
      term1 := f [j]
   else
      term1 := subs(i = j, f)
   fi:
   if type (x, procedure) then
      xterm1 := x (j)
   elif type (x, array) then
      xterm1 := x [j]
   else
      xterm1 := subs(i = j, x)
   fi:
   if not type (term1, complex) or 
      not type (xterm1, complex) then
      ERROR (`invalid sequence element found`)
   fi:
   if tseq = sump then
      sum1  := sum1 + term1:
      t1 := sum1
   else
      t1 := term1
   fi;
   app[1] := j:
   app[2] := evalhf (t1):
   if arith = h then	
      app[3] := evalhf (`trans/rho` (t1, xterm1, j, var(arr1), var(arr2)))
   elif arith = s then
      app[3] := evalf (`trans/rho` (t1, xterm1, j, arr1, arr2))
   elif arith = hs then
      nout   := 4:
      app[3] := evalhf (`trans/rho` (t1, xterm1, j, var(arr1), var(arr2))):
      app[4] := evalf (`trans/rho` (t1, xterm1, j, arr3, arr4))
   fi;
   if j >= nmin then
      print(seq (app[m], m = 1 .. nout))
   fi;
od:
#
end:
`trans/rho_sym` := proc (s_k, x_k, k, arr, xarr)
#***********************************************************************
#
#  Procedure  : rho
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 18 june 1990
#  Update     : 1 july 1990
#               normalize intermediate expressions
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  The rho algorithm due to Wynn is used to perform the rho
#  transformation.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  x_k  : element x_k of the auxiliary sequence
#  k    : number of the last element
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         the rho table
#  xarr : 1-dimensional array containing the auxiliary sequence x_k
#         required in the generalized version of the rho algorithm
#
#***********************************************************************
#
local j, aux1, aux2, diff1;
#
# initializations
#
arr  [ k ] := s_k;
xarr [ k ] := x_k;
#
# recursive computation of the rho table.
#
if k = 0 then
   RETURN (s_k)
else
   aux2 := 0;
   for j from k by -1 to 1
   do
      aux1          := aux2;
      aux2          := arr [ j - 1 ];
      diff1         := normal (arr [ j ] - aux2);
      arr [ j - 1 ] := normal (aux1 + (xarr [ k ] - xarr [ j - 1 ]) / diff1)
   od;
   if modp (k, 2) = 0 then
      RETURN (arr [ 0 ])
   else
      RETURN (arr [ 1 ])
   fi
fi
end:
`trans/rho` := proc (s_k, x_k, k, arr, xarr)
#***********************************************************************
#
#  Procedure  : rho
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 24 september 1992
#               version of rho for numerical transformation
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  The rho algorithm due to Wynn is used to perform the rho
#  transformation.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  x_k  : element x_k of the auxiliary sequence
#  k    : number of the last element
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         the rho table
#  xarr : 1-dimensional array containing the auxiliary sequence x_k
#         required in the generalized version of the rho algorithm
#
#***********************************************************************
#
local j, aux1, aux2, diff1;
#
# initializations
#
arr  [ k ] := s_k;
xarr [ k ] := x_k;
#
# recursive computation of the rho table.
#
if k = 0 then
   RETURN (s_k)
else
   aux2 := 0;
   for j from k by -1 to 1
   do
      aux1          := aux2;
      aux2          := arr [ j - 1 ];
      diff1         := arr [ j ] - aux2;
      if evalf (abs (diff1)) < evalhf (DBL_MIN) then
         arr [j - 1] := evalhf (DBL_MAX)
      else
         arr [j - 1] := aux1 + (xarr [ k ] - xarr [ j - 1 ]) / diff1
      fi
   od;
   if modp (k, 2) = 0 then
      RETURN (arr [0])
   else
      RETURN (arr [1])
   fi
fi
end:
`trans/rhoit_num` := proc(f::algebraic, x::algebraic, eqn::{`=`, name})
#*************************************************************************
#
# Description:
#
# This procedure is a driver program for transforming series
# or sequences numerically using the iterated rho_2 transformation.
#
# Input parameters:
#
# f     : sequence element s_n
# x     : element x_n of the auxiliary sequence
# eqn   : equation n=min..max: print n, s_n and T(s_n)
#         from min up to max (Default: n=0..10, if min..max is
#         not specified)
#
# optional:
#
# arith : arithmetic system: h (hardeware floating point system)
#                            s (Maple infinite precision arithmetic)
#                            hs (h + s)
#                            (Default: out = h)
# tseq  : type of sequence: sump (summation) or seqp (sequence)
#         (Default: tseq = sump)
# ndig  : number of digits (Default: ndig = 16)
#
#*************************************************************************
#
local sum1, term1, arr1, arr2, arr3, arr4, i, j, m, n, eqn_lhs, eqn_rhs,
      nmin, nmax, arith, nout, app, tseq, t1, xterm1, ndig;
#
# initialization
#
sum1  := 0:
nmin  := 0:
nmax  := 10:
ndig  := 16:
arith := h:
nout  := 3:
tseq  := sump:
#
# check input parameters
#
if type (eqn, `=`) then
   eqn_lhs := op (1, eqn);
   eqn_rhs := op (2, eqn);
   if type (eqn_lhs, name) then
      i := eqn_lhs
   else
       ERROR (`invalid argument found`)
   fi;
   if type (eqn_rhs, `..`) and
      type (op(1, eqn_rhs), integer)
      and op(1, eqn_rhs) >= 0 and
      type (op(2, eqn_rhs), integer) and
      op(2, eqn_rhs) >= 0 then
      nmin := op(1, eqn_rhs);
      nmax := op(2, eqn_rhs);
      if nmax < nmin then
         ERROR (`lower bound > upper bound`)
      fi;
   else
      ERROR (`invalid argument found`)
   fi;
elif type (eqn , name) then
   i := eqn
fi;
#
if nargs > 6 then
   ERROR (`wrong number of arguments`)
elif nargs > 3 then
   for n from 4 to nargs do
      if type (args [n], name) and
         (args[n] = h or args[n] = s or args[n] = hs) then
         arith := args [n]
      elif type (args [n], name) and
         (args[n] = sump or args[n] = seqp) then
         tseq := args [n]
      elif type (args[n], integer) and
         args [n] >= 0 then
         ndig := args [n]
      else
         ERROR (`invalid argument found`)
      fi
   od
fi;
#
# end of input checking
#
arr1   := array (0 .. nmax):
arr2   := array (0 .. nmax):
arr3   := array (0 .. nmax):
arr4   := array (0 .. nmax):
Digits := ndig:
#
for j from 0 to nmax do
   if type (f, procedure) then
      term1 := f (j)
   elif type (f, array) then
      term1 := f [j]
   else
      term1 := subs(i = j, f)
   fi:
   if type (x, procedure) then
      xterm1 := x (j)
   elif type (x, array) then
      xterm1 := x [j]
   else
      xterm1 := subs(i = j, x)
   fi:
   if not type (term1, complex) or 
      not type (xterm1, complex) then
      ERROR (`invalid sequence element found`)
   fi:
   if tseq = sump then
      sum1  := sum1 + term1:
      t1 := sum1
   else
      t1 := term1
   fi;
   app[1] := j:
   app[2] := evalhf (t1):
   if arith = h then	
      app[3] := evalhf (`trans/rhoit` (t1, xterm1, j, var(arr1), var(arr2)))
   elif arith = s then
      app[3] := evalf (`trans/rhoit` (t1, xterm1, j, arr1, arr2))
   elif arith = hs then
      nout   := 4:
      app[3] := evalhf (`trans/rhoit` (t1, xterm1, j, var(arr1), var(arr2))):
      app[4] := evalf (`trans/rhoit` (t1, xterm1, j, arr3, arr4))
   fi;
   if j >= nmin then
      print(seq (app[m], m = 1 .. nout))
   fi;
od:
#
end:
`trans/rhoit_sym` := proc (s_k, x_k, k, arr, xarr)
#***********************************************************************
#
#  Procedure  : rhoit
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 18 june 1990
#  Update     : 1 july 1990
#               normalize intermediate expressions
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  The iterated rho_2 algorithm is used to perform the series
#  transformation.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  x_k  : element x_k of the auxiliary sequence
#  k    : number of the last element
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         the table for the iterated rho_2 transformation
#  xar  : 1-dimensional array containing the auxiliary sequence x_k
#
#***********************************************************************
#
local j , diff1, diff2, m, dnom;
#
# initializations
#
arr  [ k ] := s_k;
xarr [ k ] := x_k;
#
# perform the iterated rho_2 algorithm
#
if k < 2 then
   RETURN (s_k)
else
   m := k;
   for j from 1 to iquo (k, 2)
   do
      m         := m - 2;
      diff1     := normal (arr [ m + 1 ] - arr [ m ]);
      diff2     := normal (arr [ m + 2 ] - arr [ m + 1 ]);
      dnom      := normal ((xarr [ k ] - xarr [ m + 1 ]) * diff1 -
                   (xarr [ k - 1 ] - xarr [ m ]) * diff2);
      arr [ m ] := normal (arr [ m + 1 ] + (xarr [ k ] - xarr [ m ]) *
                   diff1 * diff2 / dnom)
   od;
   if modp (k, 2) = 0 then
      RETURN (arr [ 0 ])
   else
      RETURN (arr [ 1 ])
   fi
fi
end:
`trans/rhoit` := proc (s_k, x_k, k, arr, xarr)
#***********************************************************************
#
#  Procedure  : rhoit
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 24 september 1992
#               version of rhoit for numerical transformation
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  The iterated rho_2 algorithm is used to perform the series
#  transformation.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  x_k  : element x_k of the auxiliary sequence
#  k    : number of the last element
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         the table for the iterated rho_2 transformation
#  xar  : 1-dimensional array containing the auxiliary sequence x_k
#
#***********************************************************************
#
local j , diff1, diff2, m, dnom;
#
# initializations
#
arr  [ k ] := s_k;
xarr [ k ] := x_k;
#
# perform the iterated rho_2 algorithm
#
if k < 2 then
   RETURN (s_k)
else
   m := k;
   for j from 1 to iquo (k, 2)
   do
      m         := m - 2;
      diff1     := arr [ m + 1 ] - arr [ m ];
      diff2     := arr [ m + 2 ] - arr [ m + 1 ];
      dnom      := (xarr [ k ] - xarr [ m + 1 ]) * diff1 -
                   (xarr [ k - 1 ] - xarr [ m ]) * diff2;
      if evalf (abs (dnom)) < evalhf (DBL_MIN) then
         arr [m] := evalhf (DBL_MAX)
      else
         arr [m] := arr [ m + 1 ] + (xarr [ k ] - xarr [ m ]) *
                    diff1 * diff2 / dnom
      fi
   od;
   if modp (k, 2) = 0 then
      RETURN (arr [0])
   else
      RETURN (arr [1])
   fi
fi
end:
`trans/theta_num` := proc(f::algebraic, eqn::{`=`, name})
#*************************************************************************
#
# Description:
#
# This procedure is a driver program for transforming series
# or sequences numerically using the theta transformation.
#
# Input parameters:
#
# f     : sequence element s_n
# eqn   : equation n=min..max: print n, s_n and T(s_n)
#         from min up to max (Default: n=0..10, if min..max is
#         not specified)
#
# optional:
#
# arith : arithmetic system: h (hardeware floating point system)
#                            s (Maple infinite precision arithmetic)
#                            hs (h + s)
#                            (Default: out = h)
# tseq  : type of sequence: sump (summation) or seqp (sequence)
#         (Default: tseq = sump)
# ndig  : number of digits (Default: ndig = 16)
#
#*************************************************************************
#
local sum1, term1, arr1, arr2, arr3, arr4, i, j, m, n, eqn_lhs, eqn_rhs,
      nmin, nmax, arith, nout, app, tseq, t1, ndig;
#
# initialization
#
sum1  := 0:
nmin  := 0:
nmax  := 10:
ndig  := 16:
arith := h:
nout  := 3:
tseq  := sump:
#
# check input parameters
#
if type (eqn, `=`) then
   eqn_lhs := op (1, eqn);
   eqn_rhs := op (2, eqn);
   if type (eqn_lhs, name) then
      i := eqn_lhs
   else
       ERROR (`invalid argument found`)
   fi;
   if type (eqn_rhs, `..`) and
      type (op(1, eqn_rhs), integer)
      and op(1, eqn_rhs) >= 0 and
      type (op(2, eqn_rhs), integer) and
      op(2, eqn_rhs) >= 0 then
      nmin := op(1, eqn_rhs);
      nmax := op(2, eqn_rhs);
      if nmax < nmin then
         ERROR (`lower bound > upper bound`)
      fi;
   else
      ERROR (`invalid argument found`)
   fi;
elif type (eqn , name) then
   i := eqn
fi;
#
if nargs > 5 then
   ERROR (`wrong number of arguments`)
elif nargs > 2 then
   for n from 3 to nargs do
      if type (args [n], name) and
         (args[n] = h or args[n] = s or args[n] = hs) then
         arith := args [n]
      elif type (args [n], name) and
         (args[n] = sump or args[n] = seqp) then
         tseq := args [n]
      elif type (args[n], integer) and
         args [n] >= 0 then
         ndig := args [n]
      else
         ERROR (`invalid argument found`)
      fi
   od
fi;
#
# end of input checking
#
arr1 := array (-1 .. nmax):
arr2 := array (-1 .. nmax):
arr3 := array (-1 .. nmax):
arr4 := array (-1 .. nmax):
Digits := ndig:
#
for j from 0 to nmax do
   if type (f, procedure) then
      term1 := f (j)
   elif type (f, array) then
      term1 := f [j]
   else
      term1 := subs(i = j, f)
   fi:
   if not type (term1, complex) then
      ERROR (`invalid sequence element found`)
   fi:
   if tseq = sump then
      sum1  := sum1 + term1:
      t1 := sum1
   else
      t1 := term1
   fi;
   app[1] := j:
   app[2] := evalhf (t1):
   if arith = h then	
      app[3] := evalhf (`trans/theta` (t1, j, var(arr1), var(arr2)))
   elif arith = s then
      app[3] := evalf (`trans/theta` (t1, j, arr1, arr2))
   elif arith = hs then
      nout   := 4:
      app[3] := evalhf (`trans/theta` (t1, j, var(arr1), var(arr2))):
      app[4] := evalf (`trans/theta` (t1, j, arr3, arr4))
   fi;
   if j >= nmin then
      print(seq (app[m], m = 1 .. nout))
   fi;
od
end:
`trans/theta_sym` := proc (s_k, k, arr, xarr)
#***********************************************************************
#
#  Procedure  : theta
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 18 june 1990
#  Update     : 1 july 1990
#               normalize intermediate expressions
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  The theta algorithm due to Brezinski is used to perform the theta
#  transformation.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  k    : number of the last element
#  arr  : 1-dimensional array
#  xarr : 1-dimensional array
#
#***********************************************************************
#
local j, aux1, aux2, aux3, dnom, diff1, jmax;
#
# recursive computation of the theta table
#
if k = 0 then
   arr [ k ] := s_k;
   RETURN (s_k)
elif k > 0 then
   jmax := iquo (2 * k + 1, 3);
   if (modp (k, 2) = 0) then
      aux2      := 0;
      aux1      := arr [ 0 ];
      arr [ 0 ] := s_k;
      for j from 1 to jmax
      do
         aux3 := aux2;
         aux2 := aux1;
         if j < jmax then
            aux1 := arr [ j ]
         fi;
         if (modp (j, 2) = 0) then
            dnom        := normal (arr [ j - 1 ] - 2 *
                           xarr [ j - 1 ] + aux2);
            arr [ j ] := normal (aux3 + (xarr [ j - 2 ] - aux3) *
                         (arr [ j - 1 ] - xarr [ j - 1 ]) / dnom)
         else
            diff1     := normal (arr [ j - 1 ] - xarr [ j - 1 ]);
            arr [ j ] := normal (aux3 + 1 / diff1)
         fi
      od;
#
      if (modp (jmax, 2) = 0) then
         RETURN (arr [ jmax ])
      else
         RETURN (arr [ jmax - 1 ])
      fi
   else
      aux2       := 0;
      aux1       := xarr [ 0 ];
      xarr [ 0 ] := s_k;
      for j from 1 to jmax
      do
         aux3 := aux2;
         aux2 := aux1;
         if j < jmax then
            aux1 := xarr [ j ]
         fi;
         if (modp (j, 2) = 0) then
            dnom         := normal (xarr [ j - 1 ] - 2 *
                            arr [ j - 1 ] + aux2);
            xarr [ j ] := normal (aux3 + (arr [ j - 2 ] - aux3) *
                          (xarr [ j - 1 ] - arr [ j - 1 ]) / dnom)
         else
            diff1      := normal (xarr [ j - 1 ] - arr [ j - 1 ]);
            xarr [ j ] := normal (aux3 + 1 / diff1)
         fi
      od;
#
      if (modp (jmax, 2) = 0) then
         RETURN (xarr [ jmax ])
      else
         RETURN (xarr [ jmax - 1 ])
      fi
   fi
fi
end:
`trans/theta` := proc (s_k, k, arr, xarr)
#***********************************************************************
#
#  Procedure  : theta
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 21 september 1992
#               version of theta for numerical transformation
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  The theta algorithm due to Brezinski is used to perform the theta
#  transformation.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  k    : number of the last element
#  arr  : 1-dimensional array
#  xarr : 1-dimensional array
#
#***********************************************************************
#
local j, aux1, aux2, aux3, dnom, diff1, jmax;
#
# recursive computation of the theta table
#
if k = 0 then
   arr [ k ] := s_k;
   RETURN (s_k)
elif k > 0 then
   jmax := iquo (2 * k + 1, 3);
   if (modp (k, 2) = 0) then
      aux2      := 0;
      aux1      := arr [ 0 ];
      arr [ 0 ] := s_k;
      for j from 1 to jmax
      do
         aux3 := aux2;
         aux2 := aux1;
         if j < jmax then
            aux1 := arr [ j ]
         fi;
         if (modp (j, 2) = 0) then
            dnom      := arr [ j - 1 ] - 2 * xarr [ j - 1 ] + aux2;
            if evalf (abs (dnom)) < evalhf (DBL_MIN) then
               arr [j] := evalhf (DBL_MAX)
            else
               arr [j] := aux3 + (xarr [ j - 2 ] - aux3) *
                          (arr [ j - 1 ] - xarr [ j - 1 ]) / dnom
            fi
         else
            diff1     := arr [ j - 1 ] - xarr [ j - 1 ];
            if evalf (abs (diff1)) < evalhf (DBL_MIN) then
               arr [j] := evalhf (DBL_MAX)
            else
               arr [j] := aux3 + 1 / diff1
            fi
         fi
      od;
#
      if (modp (jmax, 2) = 0) then
         RETURN (arr [ jmax ])
      else
         RETURN (arr [ jmax - 1 ])
      fi
   else
      aux2       := 0;
      aux1       := xarr [ 0 ];
      xarr [ 0 ] := s_k;
      for j from 1 to jmax
      do
         aux3 := aux2;
         aux2 := aux1;
         if j < jmax then
            aux1 := xarr [ j ]
         fi;
         if (modp (j, 2) = 0) then
            dnom       := xarr [ j - 1 ] - 2 * arr [ j - 1 ] + aux2;
            if evalf (abs (dnom)) < evalhf (DBL_MIN) then
               xarr [j] := evalhf (DBL_MAX)
            else
               xarr [j] := aux3 + (arr [ j - 2 ] - aux3) *
                           (xarr [ j - 1 ] - arr [ j - 1 ]) / dnom
            fi
         else
            diff1      := xarr [ j - 1 ] - arr [ j - 1 ];
            if evalf (abs (diff1)) < evalhf (DBL_MIN) then
               xarr [j] := evalhf (DBL_MAX)
            else
               xarr [ j ] := aux3 + 1 / diff1
            fi
         fi
      od;
#
      if (modp (jmax, 2) = 0) then
         RETURN (xarr [ jmax ])
      else
         RETURN (xarr [ jmax - 1 ])
      fi
   fi
fi
end:
`trans/thetait_num` := proc(f::algebraic, eqn::{`=`, name})
#*************************************************************************
#
# Description:
#
# This procedure is a driver program for transforming series
# or sequences numerically using the iterated theta_2 transformation.
#
# Input parameters:
#
# f     : sequence element s_n
# eqn   : equation n=min..max: print n, s_n and T(s_n)
#         from min up to max (Default: n=0..10, if min..max is
#         not specified)
#
# optional:
#
# arith : arithmetic system: h (hardeware floating point system)
#                            s (Maple infinite precision arithmetic)
#                            hs (h + s)
#                            (Default: out = h)
# tseq  : type of sequence: sump (summation) or seqp (sequence)
#         (Default: tseq = sump)
# ndig  : number of digits (Default: ndig = 16)
#
#*************************************************************************
#
local sum1, term1, arr1, arr2, arr3, arr4, i, j, m, n, eqn_lhs, eqn_rhs,
      nmin, nmax, arith, nout, app, tseq, t1, ndig;
#
# initialization
#
sum1  := 0:
nmin  := 0:
nmax  := 10:
ndig  := 16:
arith := h:
nout  := 3:
tseq  := sump:
#
# check input parameters
#
if type (eqn, `=`) then
   eqn_lhs := op (1, eqn);
   eqn_rhs := op (2, eqn);
   if type (eqn_lhs, name) then
      i := eqn_lhs
   else
       ERROR (`invalid argument found`)
   fi;
   if type (eqn_rhs, `..`) and
      type (op(1, eqn_rhs), integer)
      and op(1, eqn_rhs) >= 0 and
      type (op(2, eqn_rhs), integer) and
      op(2, eqn_rhs) >= 0 then
      nmin := op(1, eqn_rhs);
      nmax := op(2, eqn_rhs);
      if nmax < nmin then
         ERROR (`lower bound > upper bound`)
      fi;
   else
      ERROR (`invalid argument found`)
   fi;
elif type (eqn , name) then
   i := eqn
fi;
#
if nargs > 5 then
   ERROR (`wrong number of arguments`)
elif nargs > 2 then
   for n from 3 to nargs do
      if type (args [n], name) and
         (args[n] = h or args[n] = s or args[n] = hs) then
         arith := args [n]
      elif type (args [n], name) and
         (args[n] = sump or args[n] = seqp) then
         tseq := args [n]
      elif type (args[n], integer) and
         args [n] >= 0 then
         ndig := args [n]
      else
         ERROR (`invalid argument found`)
      fi
   od
fi;
#
# end of input checking
#
arr1 := array (-1 .. nmax):
arr2 := array (-1 .. nmax):
arr3 := array (-1 .. nmax):
arr4 := array (-1 .. nmax):
Digits := ndig:
#
for j from 0 to nmax do
   if type (f, procedure) then
      term1 := f (j)
   elif type (f, array) then
      term1 := f [j]
   else
      term1 := subs(i = j, f)
   fi:
   if not type (term1, complex) then
      ERROR (`invalid sequence element found`)
   fi:
   if tseq = sump then
      sum1  := sum1 + term1:
      t1 := sum1
   else
      t1 := term1
   fi;
   app[1] := j:
   app[2] := evalhf (t1):
   if arith = h then	
      app[3] := evalhf (`trans/thetait` (t1, j, var(arr1), var(arr2)))
   elif arith = s then
      app[3] := evalf (`trans/thetait` (t1, j, arr1, arr2))
   elif arith = hs then
      nout   := 4:
      app[3] := evalhf (`trans/thetait` (t1, j, var(arr1), var(arr2))):
      app[4] := evalf (`trans/thetait` (t1, j, arr3, arr4))
   fi;
   if j >= nmin then
      print(seq (app[m], m = 1 .. nout))
   fi;
od
end:
`trans/thetait_sym` := proc (s_k, k, arr)
#***********************************************************************
#
#  Procedure  : thetait
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 18 june 1990
#  Update     : 1 july 1990
#               normalize intermediate expressions
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  The iterated theta_2 algorithm is used to perform the sequence
#  (series) transformation.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  k    : number of the last element
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         theta_2 table
#
#***********************************************************************
#
local j, m, dnom, diff0, diff1, diff2;
#
# initialization
#
arr [ k ] := s_k;
#
# recursive computation of the theta_2 table
#
if k < 3 then
   RETURN (s_k);
else
   m := k;
   for j from 1 to iquo (k, 3)
   do
      m         := m - 3;
      diff0     := normal (arr [ m + 1 ] - arr [ m ]);
      diff1     := normal (arr [ m + 2 ] - arr [ m + 1 ]);
      diff2     := normal (arr [ m + 3 ] - arr [ m + 2 ]);
      dnom      := normal (diff2 * (diff1 - diff0) -
                   diff0 * (diff2 - diff1));
      arr [ m ] := normal (arr [ m + 1 ] - diff0 * diff1 *
                   (diff2 - diff1) / dnom)
   od;
#
   RETURN (arr [modp (k, 3) ])
fi
end:
`trans/thetait` := proc (s_k, k, arr)
#***********************************************************************
#
#  Procedure  : thetait
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 24 september 1992
#               version of thetait for numerical transformation
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  The iterated theta_2 algorithm is used to perform the sequence
#  (series) transformation.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  k    : number of the last element
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         theta_2 table
#
#***********************************************************************
#
local j, m, dnom, diff0, diff1, diff2;
#
# initialization
#
arr [ k ] := s_k;
#
# recursive computation of the theta_2 table
#
if k < 3 then
   RETURN (s_k);
else
   m := k;
   for j from 1 to iquo (k, 3)
   do
      m         := m - 3;
      diff0     := arr [ m + 1 ] - arr [ m ];
      diff1     := arr [ m + 2 ] - arr [ m + 1 ];
      diff2     := arr [ m + 3 ] - arr [ m + 2 ];
      dnom      := diff2 * (diff1 - diff0) -
                   diff0 * (diff2 - diff1);
      if evalf (abs (dnom)) < evalhf (DBL_MIN) then
         arr [m] := evalhf (DBL_MAX)
      else
         arr [m] := arr [ m + 1 ] - diff0 * diff1 *
                   (diff2 - diff1) / dnom
      fi
   od;
#
   RETURN (arr [modp (k, 3) ])
fi
end:
`trans/aitken_num` := proc(f::algebraic, eqn::{`=`, name})
#*************************************************************************
#
# Description:
#
# This procedure is a driver program for transforming series 
# or sequences numerically using the iterated Aitken transformation
#
# Input parameters:
#
# f     : sequence element s_n
# eqn   : equation n=min..max: print n, s_n and T(s_n)
#         from min up to max (Default: n=0..10, if min..max is
#         not specified)
#
# optional:
#
# arith : arithmetic system: h (hardeware floating point system)
#                            s (Maple infinite precision arithmetic)
#                            hs (h + s)
#                            (Default: out = h)
# tseq  : type of sequence: sump (summation) or seqp (sequence)
#         (Default: tseq = sump)
# ndig  : number of digits (Default: ndig = 16)
#
#*************************************************************************
#
local sum1, term1, arr1, arr2, i, j, m, n, eqn_lhs, eqn_rhs, 
      nmin, nmax, arith, nout, app, tseq, t1, ndig;
#
# initialization
#
sum1  := 0:
nmin  := 0:
nmax  := 10:
ndig  := 16:
arith := h:
nout  := 3:
tseq  := sump:
#
# check input parameters
#
if type (eqn, `=`) then
   eqn_lhs := op (1, eqn);
   eqn_rhs := op (2, eqn);
   if type (eqn_lhs, name) then
      i := eqn_lhs
   else
       ERROR (`invalid argument found`)
   fi;
   if type (eqn_rhs, `..`) and
      type (op(1, eqn_rhs), integer)
      and op(1, eqn_rhs) >= 0 and
      type (op(2, eqn_rhs), integer) and
      op(2, eqn_rhs) >= 0 then
      nmin := op(1, eqn_rhs);
      nmax := op(2, eqn_rhs);
      if nmax < nmin then
         ERROR (`lower bound > upper bound`)
      fi;
   else
      ERROR (`invalid argument found`)
   fi;
elif type (eqn , name) then
   i := eqn
fi;
#
if nargs > 5 then
   ERROR (`wrong number of arguments`)
elif nargs > 2 then
   for n from 3 to nargs do
      if type (args [n], name) and
         (args[n] = h or args[n] = s or args[n] = hs) then
         arith := args [n]
      elif type (args [n], name) and
         (args[n] = sump or args[n] = seqp) then
         tseq := args [n]
      elif type (args[n], integer) and
         args [n] >= 0 then
         ndig := args [n]
      else
         ERROR (`invalid argument found`)
      fi
   od
fi;
#
# end of input checking
#
arr1 := array(0 .. nmax):
arr2 := array(0 .. nmax):
Digits := ndig:
#
for j from 0 to nmax do
   if type(f, procedure) then
      term1 := f(j)
   elif type (f, array) then
      term1 := f [j]
   else
      term1 := subs(i = j, f)
   fi:
   if not type (term1, complex) then
      ERROR (`invalid sequence element found`)
   fi:
   if tseq = sump then
      sum1  := sum1 + term1:
      t1 := sum1
   else
      t1 := term1
   fi;
   app[1] := j:
   app[2] := evalhf (t1):
   if arith = h then	
      app[3] := evalhf (`trans/aitken` (t1, j, var(arr1)))
   elif arith = s then
      app[3] := evalf (`trans/aitken` (t1, j, arr2))
   elif arith = hs then
      nout   := 4:
      app[3] := evalhf (`trans/aitken` (t1, j, var(arr1))):
      app[4] := evalf (`trans/aitken` (t1, j, arr2))
   fi;
   if j >= nmin then
      print(seq (app[m], m = 1 .. nout))
   fi;
od
end:
`trans/aitken_sym` := proc (s_k, k, arr)
#***********************************************************************
#
#  Procedure  : aitken
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 18 june 1990
#  Update     : 1 july 1990
#               normalize intermediate expressions
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  A recursive algorithm is used to perform the iterated Aitken
#  transformation.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  k    : number of the last element
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         the Aitken table
#
#***********************************************************************
#
local j, dnom, m;
#
# initialization
#
arr [ k ] := s_k;
#
# perform the iterated Aitken transformation
#
if k < 2 then
   RETURN (s_k)
else
   m := k;
   for j from 1 to iquo (k, 2)
   do
      m         := m - 2;
      dnom      := normal (arr [ m + 2 ] - 2 * arr [ m + 1 ] + arr [ m ]);
      arr [ m ] := normal (arr [ m ] - (arr [ m ] - arr [ m + 1 ]) ** 2 / dnom)
   od;
#
   if modp (k, 2) = 0 then
      RETURN (arr [ 0 ])
   else
      RETURN (arr [ 1 ])
   fi
fi
end:
`trans/aitken` := proc (s_k, k, arr)
#***********************************************************************
#
#  Procedure  : aitken
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 24 september 1992
#               version of aitken for numerical transformaion
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  A recursive algorithm is used to perform the iterated Aitken
#  transformation.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  k    : number of the last element
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         the Aitken table
#
#***********************************************************************
#
local j, dnom, m;
#
# initialization
#
arr [ k ] := s_k;
#
# perform the iterated Aitken transformation
#
if k < 2 then
   RETURN (s_k)
else
   m := k;
   for j from 1 to iquo (k, 2)
   do
      m         := m - 2;
      dnom      := arr [ m + 2 ] - 2 * arr [ m + 1 ] + arr [ m ];
      if evalf (abs (dnom)) < evalhf (DBL_MIN) then
         arr [m] := evalhf (DBL_MAX)
      else
         arr [m] := arr [ m ] - (arr [ m ] - arr [ m + 1 ]) ** 2 / dnom
      fi
   od;
#
   if modp (k, 2) = 0 then
      RETURN (arr [0])
   else
      RETURN (arr [1])
   fi
fi
end:
`trans/rich_num` := proc(f::algebraic, x::algebraic, eqn::{`=`, name})
#*************************************************************************
#
# Description:
#
# This procedure is a driver program for transforming series
# or sequences numerically using the Richardson extrapolation process.
#
# Input parameters:
#
# f     : sequence element s_n
# x     : element x_n of the auxiliary sequence
# eqn   : equation n=min..max: print n, s_n and T(s_n)
#         from min up to max (Default: n=0..10, if min..max is
#         not specified)
#
# optional:
#
# arith : arithmetic system: h (hardeware floating point system)
#                            s (Maple infinite precision arithmetic)
#                            hs (h + s)
#                            (Default: out = h)
# tseq  : type of sequence: sump (summation) or seqp (sequence)
#         (Default: tseq = sump)
# ndig  : number of digits (Default: ndig = 16)
#
#*************************************************************************
#
local sum1, term1, arr1, arr2, arr3, arr4, i, j, m, n, eqn_lhs, eqn_rhs, 
      nmin, nmax, arith, nout, app, tseq, t1, xterm1, ndig;
#
# initialization
#
sum1  := 0:
nmin  := 0:
nmax  := 10:
ndig  := 16:
arith := h:
nout  := 3:
tseq  := sump:
#
# check input parameters
#
if type (eqn, `=`) then
   eqn_lhs := op (1, eqn);
   eqn_rhs := op (2, eqn);
   if type (eqn_lhs, name) then
      i := eqn_lhs
   else
       ERROR (`invalid argument found`)
   fi;
   if type (eqn_rhs, `..`) and
      type (op(1, eqn_rhs), integer)
      and op(1, eqn_rhs) >= 0 and
      type (op(2, eqn_rhs), integer) and
      op(2, eqn_rhs) >= 0 then
      nmin := op(1, eqn_rhs);
      nmax := op(2, eqn_rhs);
      if nmax < nmin then
         ERROR (`lower bound > upper bound`)
      fi;
   else
      ERROR (`invalid argument found`)
   fi;
elif type (eqn , name) then
   i := eqn
fi;
#
if nargs > 6 then
   ERROR (`wrong number of arguments`)
elif nargs > 3 then
   for n from 4 to nargs do
      if type (args [n], name) and
         (args[n] = h or args[n] = s or args[n] = hs) then
         arith := args [n]
      elif type (args [n], name) and
         (args[n] = sump or args[n] = seqp) then
         tseq := args [n]
      elif type (args[n], integer) and
         args [n] >= 0 then
         ndig := args [n]
      else
         ERROR (`invalid argument found`)
      fi
   od
fi;
#
# end of input checking
#
arr1   := array (0 .. nmax):
arr2   := array (0 .. nmax):
arr3   := array (0 .. nmax):
arr4   := array (0 .. nmax):
Digits := ndig:
#
for j from 0 to nmax do
   if type (f, procedure) then
      term1 := f (j)
   elif type (f, array) then
      term1 := f [j]
   else
      term1 := subs(i = j, f)
   fi:
   if type (x, procedure) then
      xterm1 := x (j)
   elif type (x, array) then
      xterm1 := x [j]
   else
      xterm1 := subs(i = j, x)
   fi:
   if not type (term1, complex) or
      not type (xterm1, complex) then
      ERROR (`invalid sequence element found`)
   fi:
   if tseq = sump then
      sum1  := sum1 + term1:
      t1 := sum1
   else
      t1 := term1
   fi;
   app[1] := j:
   app[2] := evalhf (t1):
   if arith = h then	
      app[3] := evalhf (`trans/rich` (t1, xterm1, j, var(arr1), var(arr2)))
   elif arith = s then
      app[3] := evalf (`trans/rich` (t1, xterm1, j, arr1, arr2))
   elif arith = hs then
      nout   := 4:
      app[3] := evalhf (`trans/rich` (t1, xterm1, j, var(arr1), var(arr2))):
      app[4] := evalf (`trans/rich` (t1, xterm1, j, arr3, arr4))
   fi;
   if j >= nmin then
      print(seq (app[m], m = 1 .. nout))
   fi;
od:
#
end:
`trans/rich_sym` := proc (s_k, x_k, k, arr, xarr)
#***********************************************************************
#
#  Procedure  : rich
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 18 june 1990
#  Update     : 1 july 1990
#               normalize intermediate expressions
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  Neville's recursive scheme is used to perform the Richardson
#  extrapolation process.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  x_k  : element x_k of the auxiliary sequence
#  k    : number of the last element
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         Neville's table
#  xarr : 1-dimensional array to store the auxiliary sequence
#
#***********************************************************************
#
local j, diff1;
#
# initializations
#
arr  [ k ] := s_k;
xarr [ k ] := x_k;
#
# recursive computation of Neville's table
#
for j from 1 to k
do
   diff1         := normal (xarr [ k - j ] - xarr [ k ]);
   arr [ k - j ] := normal ((xarr [ k - j ] * arr [ k - j + 1 ] -
                    xarr [ k ] * arr [ k - j ]) / diff1)
od;
#
RETURN (arr [ 0 ]);
#
end:
`trans/rich` := proc (s_k, x_k, k, arr, xarr)
#***********************************************************************
#
#  Procedure  : rich
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 24 september 1992
#               version of rich for numerical transformation
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  Neville's recursive scheme is used to perform the Richardson
#  extrapolation process.
#
#  Input parameters :
#
#  s_k  : sequence element s_k of the sequence to be accelerated
#  x_k  : element x_k of the auxiliary sequence
#  k    : number of the last element
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         Neville's table
#  xarr : 1-dimensional array to store the auxiliary sequence
#
#***********************************************************************
#
local j, diff1;
#
# initializations
#
arr  [ k ] := s_k;
xarr [ k ] := x_k;
#
# recursive computation of Neville's table
#
for j from 1 to k
do
   diff1         := xarr [ k - j ] - xarr [ k ];
   if evalf (abs (diff1)) < evalhf (DBL_MIN) then
      arr [j - 1] := evalhf (DBL_MAX)
   else
      arr [k - j] := (xarr [ k - j ] * arr [ k - j + 1 ] -
                     xarr [ k ] * arr [ k - j ]) / diff1
   fi
od;
#
RETURN (arr [0]);
#
end:
`trans/gb_num` := proc(f::algebraic, eqn::{`=`, name})
#*************************************************************************
#
# Description:
#
# This procedure is a driver program for transforming series
# or sequences numerically using the transformation of Germain-Bonne.
#
# Input parameters:
#
# f     : sequence element s_n
# eqn   : equation n=min..max: print n, s_n and T(s_n)
#         from min up to max (Default: n=0..10, if min..max is
#         not specified)
#
# optional:
#
# arith : arithmetic system: h (hardeware floating point system)
#                            s (Maple infinite precision arithmetic)
#                            hs (h + s)
#                            (Default: out = h)
# tseq  : type of sequence: sump (summation) or seqp (sequence)
#         (Default: tseq = sump)
# ndig  : number of digits (Default: ndig = 16)
#
#*************************************************************************
#
local sum1, term1, term0, arr1, arr2, arr3, arr4, i, j, m, n, eqn_lhs,
      eqn_rhs, nmin, nmax, arith, nout, app, tseq, t1, ndig;
#
# initialization
term0 := 0:
sum1  := 0:
nmin  := 0:
nmax  := 10:
ndig  := 16:
arith := h:
nout  := 3:
tseq  := sump:
#
# check input parameters
#
if type (eqn, `=`) then
   eqn_lhs := op (1, eqn);
   eqn_rhs := op (2, eqn);
   if type (eqn_lhs, name) then
      i := eqn_lhs
   else
       ERROR (`invalid argument found`)
   fi;
   if type (eqn_rhs, `..`) and
      type (op(1, eqn_rhs), integer)
      and op(1, eqn_rhs) >= 0 and
      type (op(2, eqn_rhs), integer) and
      op(2, eqn_rhs) >= 0 then
      nmin := op(1, eqn_rhs);
      nmax := op(2, eqn_rhs);
      if nmax < nmin then
         ERROR (`lower bound > upper bound`)
      fi;
   else
      ERROR (`invalid argument found`)
   fi;
elif type (eqn , name) then
   i := eqn
fi;
#
if nargs > 5 then
   ERROR (`wrong number of arguments`)
elif nargs > 2 then
   for n from 3 to nargs do
      if type (args [n], name) and
         (args[n] = h or args[n] = s or args[n] = hs) then
         arith := args [n]
      elif type (args [n], name) and
         (args[n] = sump or args[n] = seqp) then
         tseq := args [n]
      elif type (args[n], integer) and
         args [n] >= 0 then
         ndig := args [n]
      else
         ERROR (`invalid argument found`)
      fi
   od
fi;
#
# end of input checking
#
arr1 := array (0 .. nmax):
arr2 := array (0 .. nmax):
arr3 := array (0 .. nmax):
arr4 := array (0 .. nmax):
Digits := ndig:
#
for j from 0 to nmax do
   if type (f, procedure) then
      term1 := f (j)
   elif type (f, array) then
      term1 := f [j]
   else
      term1 := subs(i = j, f)
   fi:
   if not type (term1, complex) then
      ERROR (`invalid sequence element found`)
   fi:
   if tseq = sump then
      sum1  := sum1 + term1:
      t1    := term1
   else
      sum1  := term1:
      t1    := term1 - term0:
      term0 := term1:
   fi:
   app[1] := j:
   app[2] := evalhf (sum1):
   if arith = h then	
      app[3] := evalhf (`trans/gb` (sum1, t1, j, var(arr1), var(arr2)))
   elif arith = s then
      app[3] := evalf (`trans/gb` (sum1, t1, j, arr1, arr2))
   elif arith = hs then
      nout   := 4:
      app[3] := evalhf (`trans/gb` (sum1, t1, j, var(arr1), var(arr2))):
      app[4] := evalf (`trans/gb` (sum1, t1, j, arr3, arr4))
   fi;
   if j >= nmin then
      print(seq (app[m], m = 1 .. nout))
   fi;
od
end:
`trans/gb_sym` := proc (s_k, x_k, k, arr, xarr)
#***********************************************************************
#
#  Procedure  : gb
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 18 june 1990
#  Update     : 1 july 1990
#               normalize intermediate expressions
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  Neville's recursive scheme is used to perform the transformation
#  of Germain-Bonne.
#
#  Input parameters :
#
#  s_k  : partial sum s_k of the series to be accelerated
#  x_k  : term x_k of the series
#  k    : number of the last partial sum
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         Neville's table
#  xarr : 1-dimensional array to store the terms of the series
#
#***********************************************************************
#
local j, diff1;
#
# initializations
#
arr  [ k ] := s_k;
xarr [ k ] := x_k;
#
# recursive computation of Neville's table
#
for j from 2 to k
do
   diff1         := normal (xarr [ k - j + 1] - xarr [ k ]);
   arr [ k - j ] := normal( (xarr [ k - j + 1 ] *
                    arr [ k - j + 1 ] - xarr [ k ] *
                    arr [ k - j ]) / diff1)
od;
RETURN (arr [ 0 ]);
end:
`trans/gb` := proc (s_k, x_k, k, arr, xarr)
#***********************************************************************
#
#  Procedure  : gb
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 24 september 1992
#               version of gb for numerical transformation
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  Neville's recursive scheme is used to perform the transformation
#  of Germain-Bonne.
#
#  Input parameters :
#
#  s_k  : partial sum s_k of the series to be accelerated
#  x_k  : term x_k of the series
#  k    : number of the last partial sum
#  arr  : 1-dimensional array to store the actual counterdiagonal of
#         Neville's table
#  xarr : 1-dimensional array to store the terms of the series
#
#***********************************************************************
#
local j, diff1;
#
# initializations
#
arr  [ k ] := s_k;
xarr [ k ] := x_k;
#
# recursive computation of Neville's table
#
for j from 2 to k
do
   diff1         := xarr [ k - j + 1] - xarr [ k ];
   if evalf (abs (diff1)) < evalhf (DBL_MIN) then
      arr [k - j] := evalhf (DBL_MAX)
   else
      arr [k - j] := (xarr [ k - j + 1 ] *
                     arr [ k - j + 1 ] - xarr [ k ] *
                     arr [ k - j ]) / diff1
   fi
od;
RETURN (arr [0]);
end:
`trans/lev_num` := proc(f::algebraic, eqn::{`=`, name}, typ::name)
#*************************************************************************
#
# Description:
#
# This procedure is a driver program for transforming series
# or sequences numerically using the transformations of Levin.
#
# Input parameters:
#
# f     : sequence element s_n
# eqn   : equation n=min..max: print n, s_n and T(s_n)
#         from min up to max (Default: n=0..10, if min..max is
#         not specified)
# typ   : type of transformation :u, t, d, v
#
# optional:
#
# arith : arithmetic system: h (hardeware floating point system)
#                            s (Maple infinite precision arithmetic)
#                            hs (h + s)
#                            (Default: out = h)
# tseq  : type of sequence: sump (summation) or seqp (sequence)
#         (Default: tseq = sump)
# ndig  : number of digits (Default: ndig = 16)
#
#*************************************************************************
#
local sum1, term1, term0, arr1, arr2, arr3, arr4, i, j, m, n, eqn_lhs,
      eqn_rhs, nmin, nmax, arith, nout, app, tseq, t1, typ1, ndig;
#
# initialization
term0 := 0:
sum1  := 0:
nmin  := 0:
nmax  := 10:
ndig  := 16:
arith := h:
nout  := 3:
tseq  := sump:
#
# check input parameters
#
if type (eqn, `=`) then
   eqn_lhs := op (1, eqn);
   eqn_rhs := op (2, eqn);
   if type (eqn_lhs, name) then
      i := eqn_lhs
   else
       ERROR (`invalid argument found`)
   fi;
   if type (eqn_rhs, `..`) and
      type (op(1, eqn_rhs), integer)
      and op(1, eqn_rhs) >= 0 and
      type (op(2, eqn_rhs), integer) and
      op(2, eqn_rhs) >= 0 then
      nmin := op(1, eqn_rhs);
      nmax := op(2, eqn_rhs);
      if nmax < nmin then
         ERROR (`lower bound > upper bound`)
      fi;
   else
      ERROR (`invalid argument found`)
   fi;
elif type (eqn , name) then
   i := eqn
fi:
#
if typ = 'u' then
   typ1 := 1
elif typ = 't' then
   typ1 := 2
elif typ = 'd' then
   typ1 := 3
elif typ = 'v' then
   typ1 := 4
fi:
#
if nargs > 6 then
   ERROR (`wrong number of arguments`)
elif nargs > 3 then
   for n from 4 to nargs do
      if type (args [n], name) and
         (args[n] = h or args[n] = s or args[n] = hs) then
         arith := args [n]
      elif type (args [n], name) and
         (args[n] = sump or args[n] = seqp) then
         tseq := args [n]
      elif type (args[n], integer) and
         args [n] >= 0 then
         ndig := args [n]
      else
         ERROR (`invalid argument found`)
      fi
   od
fi:
#
# end of input checking
#
arr1 := array (-1 .. nmax):
arr2 := array (-1 .. nmax):
arr3 := array (-1 .. nmax):
arr4 := array (-1 .. nmax):
Digits := ndig:
#
for j from 0 to nmax do
   if type (f, procedure) then
      term1 := f (j)
   elif type (f, array) then
      term1 := f [j]
   else
      term1 := subs(i = j, f)
   fi:
   if not type (term1, complex) then
      ERROR (`invalid sequence element found`)
   fi:
   if tseq = sump then
      sum1  := sum1 + term1:
      t1    := term1
   else
      sum1  := term1:
      t1    := term1 - term0:
      term0 := term1:
   fi:
   app[1] := j:
   app[2] := evalhf (sum1):
   if arith = h then	
      app[3] := evalhf (`trans/lev` (sum1, t1, j, var(arr1), var(arr2), typ1))
   elif arith = s then
      app[3] := evalf (`trans/lev` (sum1, t1, j, arr1, arr2, typ1))
   elif arith = hs then
      nout   := 4:
      app[3] := evalhf (`trans/lev` (sum1, t1, j, var(arr1), var(arr2), typ1)):
      app[4] := evalf (`trans/lev` (sum1, t1, j, arr3, arr4, typ1))
   fi;
   if j >= nmin then
      print(seq (app[m], m = 1 .. nout))
   fi;
od
end:
`trans/lev_sym` := proc (s_k, a_k, k, arlo, arup, typ)
#***********************************************************************
#
#  Procedure  : lev
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 18 june 1990
#  Update     : 1 july 1990
#               normalize intermediate expressions
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  lev converts a series expansion to a rational function
#  using a series transformation of D. Levin (J. Comput. Math.,
#  Vol.B3, 1973, pp.371-388 ).
#
#  In Levin's transformations it is assumed that the partial sums s_n
#  of a series can be written as s_n = x + r_n (x limit or antilimit)
#  and that the first term of the Poincare-type asymptotic expansion
#  of the remainder r_n is proportional to:
#
#  (k+1) * a_k                            (u transformation)
#  a_k                                    (t transformation)
#  a_{k+1}                                (d transformation)
#  a_{k+1} * a_k / (a_k - a_{k+1})        (v transformation)
#
#  (a_k, k >= 0,  are the terms of the series)
#
#  Input parameters :
#
#  s_k   : partial sum s_k of the series to be accelerated
#  a_k   : term a_k of the series
#  k     : number of the last partial sum
#  arlo  : 1-dim. array(-1 .. n) to store the counterdiagonal of the
#          numerator table arlo[1] .. arlo[n] and intermediate values of a_k
#          in arlo [-1]
#  arup  : 1-dim. array(-1 .. n) to store the counterdiagonal of the
#          denominator table in arup[1] .. arup[n] and intermediate
#          values of s_k
#          in arup [-1]
#  typ   : type of transformation: u, t, d or v
#
#***********************************************************************
#
local j, fact, k1;
#
#  determine the special part of the specified transformation
#
k1 := k;
#
if typ = 'u' then
   arlo [ k1 ] := normal (1 / ((k1 + 1) ** 2 * a_k))
elif typ = 't' then
   arlo [ k1 ] := normal (1 / ((k1 + 1) * a_k))
elif typ = 'd' then
   if k = 0 then
      arlo [ k1 ] := 1
   else
      k1          := k - 1;
      arlo [ k1 ] := normal (1 / ((k1 + 1) * a_k))
   fi
elif typ = 'v' then
   if k = 0 then
      arlo [-1]   := a_k;
      arlo [ k1 ] := 1
   else
      k1          := k - 1;
      arlo [ k1 ] := normal ((arlo [-1] - a_k) / ((k1 + 1) * arlo [-1] * a_k));
      arlo [-1]   := a_k
   fi
fi;
#
if typ = 'd'  or typ = 'v' then
   if k = 0 then
      arup [-1]   := s_k;
      arup [ k1 ] := s_k
   else
      arup [ k1 ] := arup [-1] * arlo [ k1 ];
      arup [-1]   := s_k
   fi;
else
   arup [ k1 ] := s_k * arlo [ k1 ]
fi;
#
for j from 1 to k1
do
   fact            := (k1 - j + 1) * k1 ** (j - 1) / (k1 + 1) ** j;
   arup [ k1 - j ] := normal (arup [ k1 - j + 1 ] - fact * arup [ k1 - j ]);
   arlo [ k1 - j ] := normal (arlo [ k1 - j + 1 ] - fact * arlo [ k1 - j ])
od;
RETURN (normal (arup [ 0 ] / arlo [ 0 ]));
end:
`trans/lev` := proc (s_k, a_k, k, arlo, arup, typ)
#***********************************************************************
#
#  Procedure  : lev
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 21 september 1992
#               version of lev for numerical transformation 
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  lev converts a series expansion to a rational function
#  using a series transformation of D. Levin (J. Comput. Math.,
#  Vol.B3, 1973, pp.371-388 ).
#
#  In Levin's transformations it is assumed that the partial sums s_n
#  of a series can be written as s_n = x + r_n (x limit or antilimit)
#  and that the first term of the Poincare-type asymptotic expansion
#  of the remainder r_n is proportional to:
#
#  (k+1) * a_k                            (u transformation)
#  a_k                                    (t transformation)
#  a_{k+1}                                (d transformation)
#  a_{k+1} * a_k / (a_k - a_{k+1})        (v transformation)
#
#  (a_k, k >= 0,  are the terms of the series)
#
#  Input parameters :
#
#  s_k   : partial sum s_k of the series to be accelerated
#  a_k   : term a_k of the series
#  k     : number of the last partial sum
#  arlo  : 1-dim. array(-1 .. n) to store the counterdiagonal of the
#          numerator table arlo[1] .. arlo[n] and intermediate values
#          of a_k in arlo [-1]
#  arup  : 1-dim. array(-1 .. n) to store the counterdiagonal of the
#          denominator table in arup[1] .. arup[n] and intermediate
#          values of s_k in arup [-1]
#  typ   : type of transformation: 1, 2, 3, 4 (corresponds to u, t, d, v)
#
#***********************************************************************
#
local j, fact, k1;
#
#  determine the special part of the specified transformation
#
k1 := k;
#
if typ = 1 then
   arlo [ k1 ] := 1 / ((k1 + 1) ** 2 * a_k)
elif typ = 2 then
   arlo [ k1 ] := 1 / ((k1 + 1) * a_k)
elif typ = 3 then
   if k = 0 then
      arlo [ k1 ] := 1
   else
      k1          := k - 1;
      arlo [ k1 ] := 1 / ((k1 + 1) * a_k)
   fi
elif typ = 4 then
   if k = 0 then
      arlo [-1]   := a_k;
      arlo [ k1 ] := 1
   else
      k1          := k - 1;
      arlo [ k1 ] := (arlo [-1] - a_k) / ((k1 + 1) * arlo [-1] * a_k);
      arlo [-1]   := a_k
   fi
fi;
#
if (typ = 3  or typ = 4) then
   if k = 0 then
      arup [-1]   := s_k;
      arup [ k1 ] := s_k
   else
      arup [ k1 ] := arup [-1] * arlo [ k1 ];
      arup [-1]   := s_k
   fi;
else
   arup [ k1 ] := s_k * arlo [ k1 ]
fi;
#
for j from 1 to k1
do
   fact            := (k1 - j + 1) * k1 ** (j - 1) / (k1 + 1) ** j;
   arup [ k1 - j ] := arup [ k1 - j + 1 ] - fact * arup [ k1 - j ];
   arlo [ k1 - j ] := arlo [ k1 - j + 1 ] - fact * arlo [ k1 - j ]
od;
#
if evalf (abs(arlo [0])) < evalhf (DBL_MIN) then
   RETURN (evalhf (DBL_MAX))
else
   RETURN (arup [0] / arlo [0])
fi:
end:
`trans/sidi_num` := proc(f::algebraic, eqn::{`=`, name}, typ::name)
#*************************************************************************
#
# Description:
#
# This procedure is a driver program for transforming series
# or sequences numerically using the transformations of Sidi.
#
# Input parameters:
#
# f     : sequence element s_n
# eqn   : equation n=min..max: print n, s_n and T(s_n)
#         from min up to max (Default: n=0..10, if min..max is
#         not specified)
# typ   : type of transformation :u, t, d, v
#
# optional:
#
# arith : arithmetic system: h (hardeware floating point system)
#                            s (Maple infinite precision arithmetic)
#                            hs (h + s)
#                            (Default: out = h)
# tseq  : type of sequence: sump (summation) or seqp (sequence)
#         (Default: tseq = sump)
# ndig  : number of digits (Default: ndig = 16)
#
#*************************************************************************
#
local sum1, term1, term0, arr1, arr2, arr3, arr4, i, j, m, n, eqn_lhs,
      eqn_rhs, nmin, nmax, arith, nout, app, tseq, t1, typ1, ndig;
#
# initialization
term0 := 0:
sum1  := 0:
nmin  := 0:
nmax  := 10:
ndig  := 16:
arith := h:
nout  := 3:
tseq  := sump:
#
# check input parameters
#
if type (eqn, `=`) then
   eqn_lhs := op (1, eqn);
   eqn_rhs := op (2, eqn);
   if type (eqn_lhs, name) then
      i := eqn_lhs
   else
       ERROR (`invalid argument found`)
   fi;
   if type (eqn_rhs, `..`) and
      type (op(1, eqn_rhs), integer)
      and op(1, eqn_rhs) >= 0 and
      type (op(2, eqn_rhs), integer) and
      op(2, eqn_rhs) >= 0 then
      nmin := op(1, eqn_rhs);
      nmax := op(2, eqn_rhs);
      if nmax < nmin then
         ERROR (`lower bound > upper bound`)
      fi;
   else
      ERROR (`invalid argument found`)
   fi;
elif type (eqn , name) then
   i := eqn
fi:
#
if typ = 'u' then
   typ1 := 1
elif typ = 't' then
   typ1 := 2
elif typ = 'd' then
   typ1 := 3
elif typ = 'v' then
   typ1 := 4
fi:
#
if nargs > 6 then
   ERROR (`wrong number of arguments`)
elif nargs > 3 then
   for n from 4 to nargs do
      if type (args [n], name) and
         (args[n] = h or args[n] = s or args[n] = hs) then
         arith := args [n]
      elif type (args [n], name) and
         (args[n] = sump or args[n] = seqp) then
         tseq := args [n]
      elif type (args[n], integer) and
         args [n] >= 0 then
         ndig := args [n]
      else
         ERROR (`invalid argument found`)
      fi
   od
fi:
#
# end of input checking
#
arr1 := array (-1 .. nmax):
arr2 := array (-1 .. nmax):
arr3 := array (-1 .. nmax):
arr4 := array (-1 .. nmax):
Digits := ndig:
#
for j from 0 to nmax do
   if type (f, procedure) then
      term1 := f (j)
   elif type (f, array) then
      term1 := f [j]
   else
      term1 := subs(i = j, f)
   fi:
   if not type (term1, complex) then
      ERROR (`invalid sequence element found`)
   fi:
   if tseq = sump then
      sum1  := sum1 + term1:
      t1    := term1
   else
      sum1  := term1:
      t1    := term1 - term0:
      term0 := term1:
   fi:
   app[1] := j:
   app[2] := evalhf (sum1):
   if arith = h then	
      app[3] := evalhf (`trans/sidi` (sum1, t1, j, var(arr1), var(arr2), typ1))
   elif arith = s then
      app[3] := evalf (`trans/sidi` (sum1, t1, j, arr1, arr2, typ1))
   elif arith = hs then
      nout   := 4:
      app[3] := evalhf (`trans/sidi` (sum1, t1, j, var(arr1), var(arr2), typ1)):
      app[4] := evalf (`trans/sidi` (sum1, t1, j, arr3, arr4, typ1))
   fi;
   if j >= nmin then
      print(seq (app[m], m = 1 .. nout))
   fi;
od
end:
`trans/sidi_sym` := proc (s_k, a_k, k, arlo, arup, typ)
#***********************************************************************
#
#  Procedure  : sidi
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 31 july 1990
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  sidi converts a series expansion to a rational function
#  using a transformation due to Sidi
#  (A. Sidi J. Comput.Appl.Math. Vol. 7 (1981), 37-40)
#
#  As in Levin's transformations it is assumed that the partial sums s_n
#  of a series can be written as s_n = x + r_n (x limit or antilimit)
#  and that the first term of the asymptotic expansion
#  of the remainder r_n is proportional to:
#
#  (k+1) * a_k                            (su transformation)
#  a_k                                    (st transformation)
#  a_{k+1}                                (sd transformation)
#  a_{k+1} * a_k / (a_k - a_{k+1})        (sv transformation)
#
#  (a_k, k >= 0,  are the terms of the series)
#
#  Input parameters :
#
#  s_k   : partial sum s_k of the series to be accelerated
#  a_k   : term a_k of the series
#  k     : number of the last partial sum
#  arlo  : 1-dim. array(-1 .. n) to store the counterdiagonal of the
#          numerator table arlo[1] .. arlo[n] and intermediate values of
#          a_k in arlo [-1]
#  arup  : 1-dim. array(-1 .. n) to store the counterdiagonal of the
#          denominator table in arup[1] .. arup[n] and intermediate
#          values of s_k in arup [-1]
#  typ   : type of transformation: su, st, sd or sv
#
#***********************************************************************
#
local j, fact, k1;
#
#  determine the special part of the specified transformation
#
k1 := k;
#
if typ = 'su' then
   arlo [ k1 ] := normal (1 / ((k1 + 1) * a_k))
elif typ = 'st' then
   arlo [ k1 ] := normal (1 / a_k)
elif typ = 'sd' then
   if k = 0 then
      arlo [ k1 ] := 1
   else
      k1          := k - 1;
      arlo [ k1 ] := normal (1 / a_k)
   fi
elif typ = 'sv' then
   if k = 0 then
      arlo [ k1 ] := 1;
      arlo [-1]   := a_k
   else
      arlo [ k1 ] := normal ((arlo [-1] - a_k) / (arlo [-1] * a_k));
      arlo [-1]   := a_k
   fi
fi;
#
if typ = 'sd'  or typ = 'sv' then
   if k = 0 then
      arup [ k ] := s_k;
      arup [-1]  := s_k
   else
      arup [ k1 ] := arup [-1] * arlo [ k1 ];
      arup [-1]   := s_k
   fi;
else
   arup [ k1 ] := s_k * arlo [ k1 ]
fi;
#
if k1 > 0 then
   arup [ k1 - 1 ] := arup [ k1 ] - arup [ k1 - 1 ];
   arlo [ k1 - 1 ] := arlo [ k1 ] - arlo [ k1 - 1 ];
   for j from 2 to k1
   do
      fact            := (k1 - 1) * k1 / ((k1 + j - 2) * (k1 + j - 1));
      arup [ k1 - j ] := normal (arup [ k1 - j + 1 ] - fact * arup [ k1 - j ]);
      arlo [ k1 - j ] := normal (arlo [ k1 - j + 1 ] - fact * arlo [ k1 - j ])
   od;
fi;
RETURN (normal (arup [ 0 ] / arlo [ 0 ]));
end:
`trans/sidi` := proc (s_k, a_k, k, arlo, arup, typ)
#***********************************************************************
#
#  Procedure  : sidi
#
#  Programmer : Johannes Grotendorst
#
#  Date       : Juelich, 24 september 1992
#               version of sidi for numerical transformation
#
#-----------------------------------------------------------------------
#
#  Description:
#
#  sidi converts a series expansion to a rational function
#  using a transformation due to Sidi
#  (A. Sidi J. Comput.Appl.Math. Vol. 7 (1981), 37-40)
#
#  As in Levin's transformations it is assumed that the partial sums s_n
#  of a series can be written as s_n = x + r_n (x limit or antilimit)
#  and that the first term of the asymptotic expansion
#  of the remainder r_n is proportional to:
#
#  (k+1) * a_k                            (su transformation)
#  a_k                                    (st transformation)
#  a_{k+1}                                (sd transformation)
#  a_{k+1} * a_k / (a_k - a_{k+1})        (sv transformation)
#
#  (a_k, k >= 0,  are the terms of the series)
#
#  Input parameters :
#
#  s_k   : partial sum s_k of the series to be accelerated
#  a_k   : term a_k of the series
#  k     : number of the last partial sum
#  arlo  : 1-dim. array(-1 .. n) to store the counterdiagonal of the
#          numerator table arlo[1] .. arlo[n] and intermediate values
#          of a_k in arlo [-1]
#  arup  : 1-dim. array(-1 .. n) to store the counterdiagonal of the
#          denominator table in arup[1] .. arup[n] and intermediate
#          values of s_k in arup [-1]
#  typ   : type of transformation: 1, 2, 3, 4 (su, st, sd or sv)
#
#***********************************************************************
#
local j, fact, k1;
#
#  determine the special part of the specified transformation
#
k1 := k;
#
if typ = 1 then
   arlo [ k1 ] := 1 / ((k1 + 1) * a_k)
elif typ = 2 then
   arlo [ k1 ] := 1 / a_k
elif typ = 3 then
   if k = 0 then
      arlo [ k1 ] := 1
   else
      k1          := k - 1;
      arlo [ k1 ] := 1 / a_k
   fi
elif typ = 4 then
   if k = 0 then
      arlo [ k1 ] := 1;
      arlo [-1]   := a_k
   else
      arlo [ k1 ] := (arlo [-1] - a_k) / (arlo [-1] * a_k);
      arlo [-1]   := a_k
   fi
fi;
#
if (typ = 3  or typ = 4) then
   if k = 0 then
      arup [ k ] := s_k;
      arup [-1]  := s_k
   else
      arup [ k1 ] := arup [-1] * arlo [ k1 ];
      arup [-1]   := s_k
   fi;
else
   arup [ k1 ] := s_k * arlo [ k1 ]
fi;
#
if k1 > 0 then
   arup [ k1 - 1 ] := arup [ k1 ] - arup [ k1 - 1 ];
   arlo [ k1 - 1 ] := arlo [ k1 ] - arlo [ k1 - 1 ];
   for j from 2 to k1
   do
      fact            := (k1 - 1) * k1 / ((k1 + j - 2) * (k1 + j - 1));
      arup [ k1 - j ] := arup [ k1 - j + 1 ] - fact * arup [ k1 - j ];
      arlo [ k1 - j ] := arlo [ k1 - j + 1 ] - fact * arlo [ k1 - j ]
   od;
fi;
#
if evalf (abs (arlo [0])) < evalhf (DBL_MIN) then
   RETURN (evalhf (DBL_MAX))
else
   RETURN (arup [0] / arlo [0])
fi
end:
#
#save `trans`,
#`trans/ratser`, `trans/ratgen`, `trans/eps`, `trans/aitken`, `trans/rho`,
#`trans/rhoit`, `trans/theta`, `trans/thetait`, `trans/rich`,`trans/gb`, 
#`trans/lev`, `trans/sidi`, `trans/series_transform`, `trans/fortran_header`, 
#`trans/rich_num`, `trans/aitken_num`,`trans/rho_num`, `trans/sidi_num`,
#`trans/rhoit_num`, `trans/lev_num`, `trans/thetait_num`, `trans/gb_num`,
#`trans/eps_num`, `trans/theta_num`, 
#`trans/rich_sym`, `trans/aitken_sym`,`trans/rho_sym`, `trans/sidi_sym`,
#`trans/rhoit_sym`, `trans/lev_sym`, `trans/thetait_sym`, `trans/gb_sym`,
#`trans/eps_sym`, `trans/theta_sym`, 
#`trans.m`;
#quit
